#include "BluetoothServerPlugin.h"
#include "BluetoothServerConfig.h"
#include "BluetoothOptionsDialog.h"

#include "BluetoothServiceProvider.h"

#include "BluetoothAcceptor.h"
#include "BluetoothConnector.h"

#include <QDebug>
#include <QMaemo5ValueButton>

class BluetoothServerPluginPrivate
{
public:
    PluginManager *manager;

    BluetoothServerConfig *config;
    BluetoothServerConfig *dialogConfig;
    BluetoothServiceProvider *provider;
    ServicePerformerFactory *factory;

    QMaemo5ValueButton *control;
    BluetoothOptionsDialog *dialog;
};

BluetoothServerPlugin::BluetoothServerPlugin(QObject *parent)
    : ColumbusPlugin(parent)
{
    qDebug() << "BluetoothServerPlugin: ctor()";

    this->d = new BluetoothServerPluginPrivate;

    d->config = new BluetoothServerConfig(this);
    d->dialogConfig = new BluetoothServerConfig(this);
    d->manager = NULL;
    d->control = NULL;
    d->dialog = NULL;
    d->provider = NULL;
}

BluetoothServerPlugin::~BluetoothServerPlugin()
{
    qDebug() << "BluetoothServerPlugin: dtor()";
    this->shutdown();
    delete this->d;
}

QString BluetoothServerPlugin::id() const
{
    static const QString id = "bluetooth-server";
    return id;
}

QString BluetoothServerPlugin::version() const
{
    static const QString version = "0.1.0";
    return version;
}

QString BluetoothServerPlugin::friendlyName() const
{
    static const QString friendlyName = tr("Bluetooth");
    return friendlyName;
}

bool BluetoothServerPlugin::initialize(PluginManager *manager)
{
    qDebug() << "BluetoothServerPlugin: Initializing plugin.";

    d->config->load();
    d->dialogConfig->load();

    d->manager = manager;

    d->control = new QMaemo5ValueButton(this->friendlyName());
    //d->control->setValueLayout(QMaemo5ValueButton::ValueUnderTextCentered);
    d->control->setValueText(tr("Disabled"));

    d->dialog = new BluetoothOptionsDialog(d->dialogConfig, d->manager->controller()->logger(), d->control);

    QObject::connect(d->control, SIGNAL(clicked()), d->dialog, SLOT(show()));
    QObject::connect(d->dialog, SIGNAL(finished(int)), this, SLOT(onDialogFinished()));

    if(d->config->isEnabled() && d->config->isRunOnStartup())
    {
        qDebug() << "BluetoothServerPlugin: Automatically starting up service.";
        this->startup();
    }

    return true;
}

QWidget* BluetoothServerPlugin::control()
{
    return d->control;
}

bool BluetoothServerPlugin::startup()
{
    qDebug() << "BluetoothServerPlugin: Starting services.";
    AbstractDataFormatter *formatter = d->manager->controller()->logger()->formatters().value(d->config->formatId());

    if(formatter == NULL)
    {
        qWarning() << "BluetoothServerPlugin: Unknown logging format:" << d->config->formatId();
        return false;
    }

    qDebug() << "BluetoothServerPlugin: Constructing server factory for" << formatter->id() << "data streaming.";
    d->factory = new ServicePerformerFactory(formatter, d->manager->controller()->logger(), this);

    if(d->config->mode() == BluetoothServerConfig::MasterMode)
    {
        qDebug() << "BluetoothServerPlugin: Creating master mode provider.";
        d->provider = new BluetoothAcceptor(d->config->channel(), d->factory, this);
    }
    else if(d->config->mode() == BluetoothServerConfig::SlaveMode)
    {
        qDebug() << "BluetoothServerPlugin: Creating slave mode provider.";
        d->provider = new BluetoothConnector(d->config->peerAddress(), d->config->channel(), d->factory, this);
    }

    QObject::connect(d->provider, SIGNAL(statusChanged(QString)), d->control, SLOT(setValueText(QString)));
    return d->provider->initialize();
}

void BluetoothServerPlugin::shutdown()
{
    qDebug() << "BluetoothServerPlugin: Shutting down services.";

    if(d->provider != NULL)
    {
        qDebug() << "BluetoothServerPlugin: Destroying redundant service performer.";
        delete d->provider;
        d->provider = NULL;
    }

    if(d->factory != NULL)
    {
        qDebug() << "BluetoothServerPlugin: Destroying redundant performer factory.";
        delete d->factory;
        d->factory = NULL;
    }
}

void BluetoothServerPlugin::onDialogFinished()
{
    qDebug() << "BluetoothServerPlugin: Processing configuration.";

    if(*d->config == *d->dialogConfig)
    {
        qDebug() << "BluetoothServerPlugin: Configuration not changed.";
        d->dialogConfig->save(); //  We need to do this just incase the
                                 // special isRunOnStartup value has changed.
        return;
    }

    d->dialogConfig->save();
    d->config->load();

    this->shutdown();

    if(!d->config->isEnabled())
    {
        qDebug() << "BluetoothServerPlugin: Doing nothing, service is not enabled.";
        d->control->setValueText(tr("Disabled"));
        return;
    }

    this->startup();
}
