/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 3 of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * Author: Damian Waradzyn
 */
SDL_GLES_Context *context;

void setQuadSize(GLfloat w, GLfloat h, GLfloat quad[12]) {
    quad[3] = w;
    quad[7] = h;
    quad[9] = w;
    quad[10] = h;
}

void setBoxSize(GLfloat w, GLfloat h, GLfloat box[12]) {
    box[4] = h;
    box[6] = w;
    box[7] = h;
    box[9] = w;
}

Texture fontMediumMask, fontMedium, barbg, barbgmask;

const int MEDIUM_FONT_METRICS[256] = { 8, 8, 8, 8, 8, 8, 8, 8, 8, 56, 0, 8, 8, 0, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 4, 5, 5, 11,
        8, 12, 11, 3, 5, 5, 7, 11, 4, 5, 4, 4, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 4, 4, 11, 11, 11, 7, 13, 9, 9, 9, 10, 8, 7, 10, 10, 3, 3, 8, 7, 11, 10,
        10, 8, 10, 8, 9, 7, 10, 9, 11, 10, 7, 10, 5, 4, 5, 11, 7, 7, 8, 8, 7, 8, 8, 4, 8, 8, 3, 3, 7, 3, 13, 8, 8, 8, 8, 5, 7, 5, 8, 7, 9, 7, 7, 7,
        8, 4, 8, 11, 8, 8, 8, 4, 8, 7, 13, 7, 7, 7, 17, 9, 5, 14, 8, 10, 8, 8, 4, 4, 7, 7, 8, 7, 13, 7, 13, 7, 5, 13, 8, 7, 7, 8, 5, 8, 8, 8, 8, 4,
        7, 7, 13, 6, 8, 11, 5, 13, 7, 7, 11, 5, 5, 7, 8, 8, 4, 7, 5, 6, 8, 13, 13, 13, 7, 9, 9, 9, 9, 9, 9, 13, 9, 8, 8, 8, 8, 3, 3, 3, 3, 10, 10,
        10, 10, 10, 10, 10, 11, 10, 10, 10, 10, 10, 7, 8, 8, 8, 8, 8, 8, 8, 8, 13, 7, 8, 8, 8, 8, 3, 3, 3, 3, 8, 8, 8, 8, 8, 8, 8, 11, 8, 8, 8, 8, 8,
        7, 8, 7 };

GLfloat boxVertices[12] = { 0.0f, 0.0f, 0.0f, 0.0f, 256.0f, 0.0f, 256.0f, 256.0f, 0.0f, 256.0f, 0.0f, 0.0f, };

GLfloat fakeFogColorsLandscape[16] = { 0.0f, 0.0f, 0.0f, 1.0f, 0.0f, 0.0f, 0.0f, 1.0f, 0.0f, 0.0f, 0.0f, 0.01f, 0.0f, 0.0f, 0.0f, 0.01f };
GLfloat fakeFogColorsPortrait[16] = { 0.0f, 0.0f, 0.0f, 1.0f, 0.0f, 0.0f, 0.0f, 0.01f, 0.0f, 0.0f, 0.0f, 1.0f, 0.0f, 0.0f, 0.0f, 0.01f };

int debugTilesGridInitialized = FALSE;
GLfloat debugTilesGrid[(TILES_X + TILES_Y + 2 + MAX_ZOOM_HELPER_TILES * 2) * 3 * 2];

GLfloat zoomBarLines[40 * 3 * 2];

GLfloat quadVertices[12] = { 0.0f, 0.0f, 0.0f, /* Bottom Left Of The Quad */
1.0f, 0.0f, 0.0f, /* Bottom Right Of The Quad */
0.0f, 1.0f, 0.0f, /* Top Left Of The Quad */
1.0f, 1.0f, 0.0f, /* Top Right Of The Quad */
};

GLfloat arrowVertices[12] = { -8.0f, 24.0f, 0.0f, 0.0f, 8.0f, 0.0f, 0.0f, -24.0f, 0.0f, 8.0f, 24.0f, 0.0f, };
GLfloat directionVertices[9] = { 0.0f, 18.0f, 0.0f, -6.0f, 6.0f, 0.0f, 6.0f, 6.0f, 0.0f };

GLfloat texCoordsLandscape[][8] = { { 0.0f, 0.0f, /* Bottom Left Of The Texture */
1.0f, 0.0f, /* Bottom Right Of The Texture */
0.0f, 1.0f, /* Top Left Of The Texture */
1.0f, 1.0f, /* Top Right Of The Texture */
} };

GLfloat texCoordsPortrait[][8] = { { 1.0f, 0.0f, /* Top Right Of The Texture */
1.0f, 1.0f, /* Bottom Right Of The Texture */
0.0f, 0.0f, /* Bottom Left Of The Texture */
0.0f, 1.0f, /* Top Left Of The Texture */

} };

GLfloat texCoordsPartTile[8] = { 0.0f, 0.0f, 1.0f, 0.0f, 0.0f, 1.0f, 1.0f, 1.0f, };

SDL_Surface* loadTextureFromFile(char *filename, t_tile* tile);

GLuint createTexture(GLushort * pixels4444, int w, int h, int isTile);

char buf[500];

int stringWidth(char * str) {
    int result = 0, i = 0;
    while (str[i] != '\0') {
        result += MEDIUM_FONT_METRICS[(int) str[i]];
        i++;
    }
    return result;
}

void drawStringAlpha(char* str, GLfloat r, GLfloat g, GLfloat b, GLfloat a, int useMask, GLenum sfactor, GLenum dfactor) {
    int i, len, charTileSize, xpos = 0;
    //    int j;
    len = strlen(str);
    GLfloat *vertices, *fontTextureCoords;

    int tx, ty;

    if (len == 0) {
        return;
    }

    charTileSize = fontMediumMask.size / 16;
    setQuadSize(charTileSize, charTileSize, quadVertices);

    vertices = malloc(sizeof(GLfloat) * (len) * 18);
    fontTextureCoords = malloc(sizeof(GLfloat) * len * 12);

    for (i = 0; i < len; i++) {
        //        fprintf(stderr, "------------------------------\ni = %d\n", i);
        vertices[i * 18] = xpos;
        vertices[i * 18 + 1] = 0;
        vertices[i * 18 + 2] = 0;

        vertices[i * 18 + 3] = xpos;
        vertices[i * 18 + 4] = charTileSize;
        vertices[i * 18 + 5] = 0;

        vertices[i * 18 + 6] = xpos + charTileSize;
        vertices[i * 18 + 7] = charTileSize;
        vertices[i * 18 + 8] = 0;

        vertices[i * 18 + 9] = xpos;
        vertices[i * 18 + 10] = 0;
        vertices[i * 18 + 11] = 0;

        vertices[i * 18 + 12] = xpos + charTileSize;
        vertices[i * 18 + 13] = charTileSize;
        vertices[i * 18 + 14] = 0;

        vertices[i * 18 + 15] = xpos + charTileSize;
        vertices[i * 18 + 16] = 0;
        vertices[i * 18 + 17] = 0;

        xpos += MEDIUM_FONT_METRICS[(int) str[i]];
        //        fprintf(stderr, "Width of '%c' is %d. xpos = %d\n", str[i], MEDIUM_FONT_METRICS[(int)str[i]], xpos);

        //        for (j = 0; j < 18; j++) {
        //            fprintf(stderr, "vertices[%d] = %g\n", i * 18 + j, vertices[i * 18 + j]);
        //        }

        tx = (str[i] % 16) * charTileSize;
        ty = str[i] / 16 * charTileSize;

        fontTextureCoords[i * 12 + 0] = tx / (GLfloat) fontMediumMask.size;
        fontTextureCoords[i * 12 + 1] = ty / (GLfloat) fontMediumMask.size;

        fontTextureCoords[i * 12 + 2] = tx / (GLfloat) fontMediumMask.size;
        fontTextureCoords[i * 12 + 3] = (ty + charTileSize) / (GLfloat) fontMediumMask.size;

        fontTextureCoords[i * 12 + 4] = (tx + charTileSize) / (GLfloat) fontMediumMask.size;
        fontTextureCoords[i * 12 + 5] = (ty + charTileSize) / (GLfloat) fontMediumMask.size;

        fontTextureCoords[i * 12 + 6] = tx / (GLfloat) fontMediumMask.size;
        fontTextureCoords[i * 12 + 7] = ty / (GLfloat) fontMediumMask.size;

        fontTextureCoords[i * 12 + 8] = (tx + charTileSize) / (GLfloat) fontMediumMask.size;
        fontTextureCoords[i * 12 + 9] = (ty + charTileSize) / (GLfloat) fontMediumMask.size;

        fontTextureCoords[i * 12 + 10] = (tx + charTileSize) / (GLfloat) fontMediumMask.size;
        fontTextureCoords[i * 12 + 11] = ty / (GLfloat) fontMediumMask.size;

        //        for (j = 0; j < 12; j++) {
        //            fprintf(stderr, "textureCoords[%d] = %g\n", i * 12 + j, fontTextureCoords[i * 12 + j]);
        //        }
    }
    if (useMask) {
        glEnable(GL_BLEND);
        glBlendFunc(GL_DST_COLOR, GL_ZERO);
        glColor4f(1.0, 1.0, 1.0, a);
        glBindTexture(GL_TEXTURE_2D, fontMediumMask.name);
        glVertexPointer(3, GL_FLOAT, 0, vertices);
        glTexCoordPointer(2, GL_FLOAT, 0, fontTextureCoords);
        glDrawArrays(GL_TRIANGLES, 0, len * 6);
    }

    glColor4f(r, g, b, a);
    glBlendFunc(sfactor, dfactor);
    glBindTexture(GL_TEXTURE_2D, fontMedium.name);
    glVertexPointer(3, GL_FLOAT, 0, vertices);
    glTexCoordPointer(2, GL_FLOAT, 0, fontTextureCoords);
    glDrawArrays(GL_TRIANGLES, 0, len * 6);

    free(vertices);
    free(fontTextureCoords);
}

void inline drawString(char* str, GLfloat r, GLfloat g, GLfloat b, int useMask) {
    drawStringAlpha(str, r, g, b, 1.0, useMask, GL_ONE, GL_ONE_MINUS_SRC_ALPHA);
}

#include "uielement.c"

float color = 1.0;

void drawBoundingBox() {
    glDisable(GL_BLEND);
    glBindTexture(GL_TEXTURE_2D, 0);
    glColor4f(0, 0, 1, 1.0f);

    GLfloat boundingTrapezoidVertices[12] = {
    // first point:
            (canvas.boundingTrapezoid[0].tilex - canvas.tilex) * TILE_SIZE + canvas.boundingTrapezoid[0].x, (canvas.boundingTrapezoid[0].tiley
                    - canvas.tiley) * TILE_SIZE + canvas.boundingTrapezoid[0].y, 0,
            // second point:
            (canvas.boundingTrapezoid[1].tilex - canvas.tilex) * TILE_SIZE + canvas.boundingTrapezoid[1].x, (canvas.boundingTrapezoid[1].tiley
                    - canvas.tiley) * TILE_SIZE + canvas.boundingTrapezoid[1].y, 0,
            // third point:
            (canvas.boundingTrapezoid[2].tilex - canvas.tilex) * TILE_SIZE + canvas.boundingTrapezoid[2].x, (canvas.boundingTrapezoid[2].tiley
                    - canvas.tiley) * TILE_SIZE + canvas.boundingTrapezoid[2].y, 0,
            // fourth point:
            (canvas.boundingTrapezoid[3].tilex - canvas.tilex) * TILE_SIZE + canvas.boundingTrapezoid[3].x, (canvas.boundingTrapezoid[3].tiley
                    - canvas.tiley) * TILE_SIZE + canvas.boundingTrapezoid[3].y, 0, };

    glVertexPointer(3, GL_FLOAT, 0, boundingTrapezoidVertices);
    glDrawArrays(GL_LINE_LOOP, 0, 4);

}

#define MAX_BUILDINGS_VERTICES (2000*18)
GLfloat buildingsVertices[MAX_BUILDINGS_VERTICES];

int wallCount;

void drawBuildings() {
    if (canvas.buildings == NULL) {
        return;
    }

    canvas.buildingsHeight = 20;
    int i, j, k, verticeCount;
    GLfloat translationX = (canvas.buildingsTileX - tiles[0][0][currentTilesIdx] -> tilex - getCanvasShiftX()) * TILE_SIZE;
    GLfloat translationY = (canvas.buildingsTileY - tiles[0][0][currentTilesIdx] -> tiley - getCanvasShiftY()) * TILE_SIZE;
    glTranslatef(translationX, translationY, 0.0);
    glDisable(GL_TEXTURE_2D);
    glEnable(GL_BLEND);
    glColor4f(.4, .4, .4, .6);
    if (canvas.buildingsVerticesPrecomputed == 0) {
        wallCount = 0;
        for (i = 0; i < canvas.buildings -> len; i++) {
            GArray* polygon = g_array_index(canvas.buildings, GArray*, i);

            for (j = 0; j < polygon -> len - 1; j += 2) {

                //    1|\--------|3
                //     | \       |
                //     |  \      |  1st triangle = vertices 0, 1, 2
                //     |   \     |
                //     |    \    |  2nd triangle = vertices 3, 2, 1
                //     |     \   |
                //     |      \  |  vertices 0 and 1: x == polygon[j * 2], y == polygon[j * 2 + 1]
                //     |       \ |  vertices 2 and 3: x == polygon[k], y == polygin[k + 1], where k == j or k == 0 (joining last vertice with first one)
                //    0|--------\|2

                if (j + 2 < polygon -> len) {
                    k = j + 2;
                } else {
                    k = 0;
                }
                verticeCount = wallCount * 18;
                // 1st triangle, vertice 0
                buildingsVertices[verticeCount++] = g_array_index(polygon, double, j );
                buildingsVertices[verticeCount++] = g_array_index(polygon, double, j + 1);
                buildingsVertices[verticeCount++] = 0;

                // 1st triangle, vertice 1
                buildingsVertices[verticeCount++] = g_array_index(polygon, double, j );
                buildingsVertices[verticeCount++] = g_array_index(polygon, double, j + 1);
                buildingsVertices[verticeCount++] = canvas.buildingsHeight;

                // 1st triangle, vertice 2
                buildingsVertices[verticeCount++] = g_array_index(polygon, double, k);
                buildingsVertices[verticeCount++] = g_array_index(polygon, double, k + 1);
                buildingsVertices[verticeCount++] = 0;

                // 2nd triangle, vertice 2
                buildingsVertices[verticeCount++] = g_array_index(polygon, double, k);
                buildingsVertices[verticeCount++] = g_array_index(polygon, double, k + 1);
                buildingsVertices[verticeCount++] = 0;

                // 2nd triangle, vertice 1
                buildingsVertices[verticeCount++] = g_array_index(polygon, double, j );
                buildingsVertices[verticeCount++] = g_array_index(polygon, double, j + 1);
                buildingsVertices[verticeCount++] = canvas.buildingsHeight;

                // 2nd triangle, vertice 3
                buildingsVertices[verticeCount++] = g_array_index(polygon, double, k);
                buildingsVertices[verticeCount++] = g_array_index(polygon, double, k + 1);
                buildingsVertices[verticeCount++] = canvas.buildingsHeight;

                //            fprintf(stderr, "building triangle %d (i=%d/%d, j=%d/%d, k=%d)\n", wallCount, i, canvas.buildings -> len, j, polygon -> len/2, k);
                //            for (k=0; k<18; k++) {
                //                fprintf(stderr, "%f, ", buildingsVertices[triangleCount * 18 + k]);
                //            }
                //
                //            fprintf(stderr, "\n");
                wallCount++;
                if (wallCount * 18 >= MAX_BUILDINGS_VERTICES) {
                    break;
                }
            }
            if (wallCount * 18 >= MAX_BUILDINGS_VERTICES) {
                break;
            }
        }
        canvas.buildingsVerticesPrecomputed = 1;
    }
    glVertexPointer(3, GL_FLOAT, 0, buildingsVertices);
    glDrawArrays(GL_TRIANGLES, 0, wallCount * 6);
    glEnable(GL_TEXTURE_2D);
    glTranslatef(-translationX, -translationY, 0.0);
}

void drawTile_debug(t_tile *tile) {
    if (tile == NULL) {
        return;
    }
    switch (tile -> state) {
        case STATE_GARBAGE:
            glColor4f(0, 0, 0, 1.0);
            break;
        case STATE_EMPTY:
            glColor4f(0, 0, 0, 0.5);
            break;
        case STATE_QUEUE_DELAYED:
            if (tile -> oldTiles[0] != NULL) {
                glColor4f(0.5, 0, 0.5, 0.5);
            } else {
                glColor4f(.2, .0, .0, 0.5);
            }
            break;
        case STATE_QUEUED:
            glColor4f(.2, .2, .2, 0.5);
            break;
        case STATE_ERROR:
            glColor4f(1, 0, 0, 0.5);
            break;
        case STATE_LOADING:
            glColor4f(0, 0, 1, 0.5);
            break;
        case STATE_LOADED:
            glColor4f(0, 1, 0, 0.5);
            break;
        case STATE_GL_TEXTURE_NOT_CREATED:
            glColor4f(0, 0.2, 0, 0.5);
            break;
        case STATE_SCALED:
            glColor4f(0.5, 0.5, 0.5, 0.5);
            break;
        case STATE_LOADED_SYNC_WAIT:
            glColor4f(1, 0, 1, 0.5);
            break;

    }
    glVertexPointer(3, GL_FLOAT, 0, quadVertices);
    glDrawArrays(GL_TRIANGLE_STRIP, 0, 4);

    //
    //        sprintf(buf, "%d", count);
    //        glScalef(20.0, 20.0, 20.0);
    //        drawString(buf, 0.1f, .0, 1.0, TRUE);
    //        glScalef(.05, .05, .05);
}

void drawTile(t_tile* tile) {
    int i;
    if (tile == NULL || tile -> visible == FALSE) {
        return;
    }

    if (tile->state == STATE_ERROR) {
        glColor4f(1.0f, 0.0f, 0.0f, 1.0f);
        setBoxSize(TILE_SIZE, TILE_SIZE, boxVertices);
        glVertexPointer(3, GL_FLOAT, 0, boxVertices);
        glDrawArrays(GL_LINE_LOOP, 0, 4);
        //        if (tile -> errorMessage) {
        //            sprintf(buf, "Error: %s", tile->errorMessage);
        //            drawString(buf, 1.0f, .2, .2, TRUE);
        //        }
    }

    if (FALSE && (tile -> state == STATE_LOADED) && nowMillis - tile->stateChangeTime < tile->transitionTime) {
        color = (nowMillis - tile->stateChangeTime) / (float) tile->transitionTime;
        //color = 1.0 - sqrt(sqrt(1.0 - color));
    } else {
        color = 1.0;
    }

    // If there is old tile to display and loaded tile to blend with it and they are the same size use multitexturing for smoother transition.
    if (FALSE && tile -> state == STATE_LOADED && color < 1.0 && tile -> oldTiles[0] != NULL && ((t_tile *) tile -> oldTiles[0]) -> zoom
            <= canvas.zoom) {
        t_tile * oldTile = tile -> oldTiles[0];
        glActiveTexture(GL_TEXTURE0);
        glEnable(GL_TEXTURE_2D);
        GLfloat *textcoords;
        if (oldTile -> zoom < canvas.zoom) {
            setQuadSize(TILE_SIZE, TILE_SIZE, quadVertices);
            // TODO should be the same as for blending
            //textcoords = texCoordsHalfTile[(tile -> tiley % 2) * 2 + tile -> tilex % 2];
            //            glTexCoordPointer(2, GL_FLOAT, 0, textcoords);
        } else {
            setQuadSize(TILE_SIZE, TILE_SIZE, quadVertices);
            textcoords = texCoordsLandscape[0];
            //            glTexCoordPointer(2, GL_FLOAT, 0, textcoords);
        }
        glVertexPointer(3, GL_FLOAT, 0, quadVertices);

        //        glBindTexture(GL_TEXTURE_2D, oldTile -> texture);
        //        glTexEnvi(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_DECAL);
        //        //------------------------
        //        glActiveTexture(GL_TEXTURE1);
        //        glEnable(GL_TEXTURE_2D);
        //        glTexCoordPointer(2, GL_FLOAT, 0, texCoordsLandscape[0]);
        //        glBindTexture(GL_TEXTURE_2D, tile -> texture);
        //        glVertexPointer(3, GL_FLOAT, 0, quadVertices);
        glTexEnvi(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_COMBINE);
        glTexEnvi(GL_TEXTURE_ENV, GL_COMBINE_RGB, GL_INTERPOLATE); //Interpolate RGB with RGB
        glTexEnvi(GL_TEXTURE_ENV, GL_SRC0_RGB, GL_PREVIOUS);
        glTexEnvi(GL_TEXTURE_ENV, GL_SRC1_RGB, GL_TEXTURE);
        //GL_CONSTANT refers to the call we make with glTexEnvfv(GL_TEXTURE_ENV, GL_TEXTURE_ENV_COLOR, mycolor)
        glTexEnvi(GL_TEXTURE_ENV, GL_SRC2_RGB, GL_CONSTANT);
        glTexEnvi(GL_TEXTURE_ENV, GL_OPERAND0_RGB, GL_SRC_COLOR);
        glTexEnvi(GL_TEXTURE_ENV, GL_OPERAND1_RGB, GL_SRC_COLOR);
        glTexEnvi(GL_TEXTURE_ENV, GL_OPERAND2_RGB, GL_SRC_COLOR);
        //------------------------
        glTexEnvi(GL_TEXTURE_ENV, GL_COMBINE_ALPHA, GL_INTERPOLATE); //Interpolate ALPHA with ALPHA
        glTexEnvi(GL_TEXTURE_ENV, GL_SRC0_ALPHA, GL_PREVIOUS);
        glTexEnvi(GL_TEXTURE_ENV, GL_SRC1_ALPHA, GL_TEXTURE);
        //GL_CONSTANT refers to the call we make with glTexEnvfv(GL_TEXTURE_ENV, GL_TEXTURE_ENV_COLOR, mycolor)
        glTexEnvi(GL_TEXTURE_ENV, GL_SRC2_ALPHA, GL_CONSTANT);
        glTexEnvi(GL_TEXTURE_ENV, GL_OPERAND0_ALPHA, GL_SRC_ALPHA);
        glTexEnvi(GL_TEXTURE_ENV, GL_OPERAND1_ALPHA, GL_SRC_ALPHA);
        glTexEnvi(GL_TEXTURE_ENV, GL_OPERAND2_ALPHA, GL_SRC_ALPHA);
        //        //------------------------
        //        GLfloat mycolor[4];
        //        mycolor[0] = mycolor[1] = mycolor[2] = 0.0; //RGB doesn't matter since we are not using it
        //        mycolor[3] = color; //Set the blend factor with this
        //        glTexEnvfv(GL_TEXTURE_ENV, GL_TEXTURE_ENV_COLOR, mycolor);
        //        glDrawArrays(GL_TRIANGLE_STRIP, 0, 4);
        //        glActiveTexture(GL_TEXTURE0);


        glClientActiveTexture(GL_TEXTURE0);
        glActiveTexture(GL_TEXTURE0);
        glColor4f(1.0, 1.0, 1.0, 1.0);
        glBindTexture(GL_TEXTURE_2D, oldTile -> texture);
        glVertexPointer(2, GL_FLOAT, 0, quadVertices);
        glTexCoordPointer(2, GL_FLOAT, 0, textcoords);
        //        glTexEnvi(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_REPLACE);

        /* Setup Mask (RGBA Texture, White to Transparent Gradient) */
        glClientActiveTexture(GL_TEXTURE1);
        glActiveTexture(GL_TEXTURE1);
        glEnable(GL_TEXTURE_2D);
        glColor4f(1.0, 1.0, 1.0, color);
        glBindTexture(GL_TEXTURE_2D, tile -> texture);
        glEnableClientState(GL_VERTEX_ARRAY);
        glEnableClientState(GL_TEXTURE_COORD_ARRAY);
        glVertexPointer(2, GL_FLOAT, 0, quadVertices);
        glTexCoordPointer(2, GL_FLOAT, 0, texCoordsLandscape[0]);

        /* Draw Both */
        glDrawArrays(GL_TRIANGLE_STRIP, 0, 4);

    } else {
        // Use blending if multitexturing is not applicable.
        glEnable(GL_BLEND);
        glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
        if ((tile -> state != STATE_LOADED) || color < 1.0) {
            for (i = 0; i < 4; i++) {
                t_tile * oldTile = tile -> oldTiles[i];
                if (oldTile != NULL) {
                    oldTile -> textureUsedTime = nowMillis;
                    if (oldTile -> zoom > canvas.zoom) {
                        setQuadSize(TILE_SIZE / 2, TILE_SIZE / 2, quadVertices);
                        glTexCoordPointer(2, GL_FLOAT, 0, texCoordsLandscape[0]);
                        glTranslatef(TILE_SIZE / 2 * (i / 2), TILE_SIZE / 2 * (i % 2), 0);
                    } else if (oldTile -> zoom < canvas.zoom) {
                        setQuadSize(TILE_SIZE, TILE_SIZE, quadVertices);
                        int zoomDiff = canvas.zoom - oldTile -> zoom;
                        GLfloat zoomDiffPow = pow(2.0, zoomDiff);
                        GLfloat size = 1.0 / zoomDiffPow;
                        GLfloat left = (tile -> tilex - (oldTile -> tilex * zoomDiffPow)) * size;
                        GLfloat right = left + size;
                        GLfloat up = (tile -> tiley - (oldTile -> tiley * zoomDiffPow)) * size;
                        GLfloat down = up + size;
                        //                        fprintf(
                        //                                stderr,
                        //                                "oldtilex=%d, oldtiley=%d, tilex=%d, tiley=%d, oldzoom=%d, zoom=%d, zoomDiff=%d, zommDiffPow=%f, size=%f, left=%f, right=%f, up=%f, down=%f\n",
                        //                                oldTile -> tilex, oldTile -> tiley, tile -> tilex, tile-> tiley, oldTile -> zoom, canvas.zoom, zoomDiff, zoomDiffPow,
                        //                                size, left, right, up, down);
                        texCoordsPartTile[0] = left;
                        texCoordsPartTile[1] = up;
                        texCoordsPartTile[2] = right;
                        texCoordsPartTile[3] = up;
                        texCoordsPartTile[4] = left;
                        texCoordsPartTile[5] = down;
                        texCoordsPartTile[6] = right;
                        texCoordsPartTile[7] = down;
                        glTexCoordPointer(2, GL_FLOAT, 0, texCoordsPartTile);
                    } else {
                        setQuadSize(TILE_SIZE, TILE_SIZE, quadVertices);
                        glTexCoordPointer(2, GL_FLOAT, 0, texCoordsLandscape[0]);
                    }

                    if (tile -> state != STATE_LOADED) {
                        glColor4f(1.0, 1.0, 1.0, 1.0);
                    } else {
                        glColor4f(1.0, 1.0, 1.0, 1.0 - color);
                    }

                    glBindTexture(GL_TEXTURE_2D, oldTile->texture);
                    glVertexPointer(3, GL_FLOAT, 0, quadVertices);
                    glDrawArrays(GL_TRIANGLE_STRIP, 0, 4);

                    if (oldTile -> zoom > canvas.zoom) {
                        glTranslatef(-TILE_SIZE / 2 * (i / 2), -TILE_SIZE / 2 * (i % 2), 0);
                    }
                }
            }
            glColor4f(1.0, 1.0, 1.0, color);
        } else {
            glDisable(GL_BLEND);
            glColor4f(1.0, 1.0, 1.0, 1.0);
        }

        if (tile -> state == STATE_LOADED) {
            glBindTexture(GL_TEXTURE_2D, tile->texture);
            setQuadSize(TILE_SIZE, TILE_SIZE, quadVertices);
            glVertexPointer(3, GL_FLOAT, 0, quadVertices);
            glTexCoordPointer(2, GL_FLOAT, 0, texCoordsLandscape[0]);
            glDrawArrays(GL_TRIANGLE_STRIP, 0, 4);
            tile -> textureUsedTime = nowMillis;
        }
    }
    if (options.showGrid) {
        glDisable(GL_BLEND);
        setBoxSize(TILE_SIZE, TILE_SIZE, boxVertices);
        glBindTexture(GL_TEXTURE_2D, 0);
        glColor4f(0.7f, .7f, 0.7f, 1.0f);
        glVertexPointer(3, GL_FLOAT, 0, boxVertices);
        glDrawArrays(GL_LINE_LOOP, 0, 4);
        int count = 0;
        for (i = 0; i < 4; i++) {
            if (tile -> oldTiles[i] != NULL) {
                if (((t_tile*) tile -> oldTiles[i]) -> texture != 0) {
                    count++;
                }
            }
        }
        sprintf(buf, "x=%d, y=%d, state=%d, tc=%d", tile->tilex, tile -> tiley, tile -> state, count);
        drawString(buf, 0.1f, .0, 1.0, TRUE);
        glTranslatef(0, 16, 0);
        sprintf(buf, "trans=%ld", tile->stateChangeTime);
        drawString(buf, 0.1f, .0, 1.0, TRUE);
        glTranslatef(0, -16, 0);
    }
}

/* general OpenGL initialization function */
int initGL(GLvoid) {
    SDL_GLES_Init(SDL_GLES_VERSION_1_1);

    context = SDL_GLES_CreateContext();
    SDL_GLES_MakeCurrent(context);

    glEnable(GL_TEXTURE_2D);

    glClearColor(1.0f, 1.0f, 1.0f, 1.0f);
    glEnable(GL_CLEAR);

    glEnableClientState(GL_VERTEX_ARRAY);
    glEnableClientState(GL_TEXTURE_COORD_ARRAY);

    glMatrixMode(GL_PROJECTION);
    glDisable(GL_DEPTH_TEST);
    glDisable(GL_CULL_FACE);
    glViewport(0, 0, (GLint) SCREEN_WIDTH, (GLint) SCREEN_HEIGHT);
    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();
    glFrustumf(-200, 200, 120, -120, 50.0, 600);
    glMatrixMode(GL_MODELVIEW);
    glLoadIdentity();

    // Usual fog is too heavy - fake fog used instead
    //    GLfloat fogColor[4] = { 0.0f, 0.0f, 0.0f, 1.0f };
    //    glFogx(GL_FOG_MODE, GL_LINEAR); // Fog Mode
    //    glFogfv(GL_FOG_COLOR, fogColor); // Set Fog Color
    //    glFogf(GL_FOG_DENSITY, .1f); // How Dense Will The Fog Be
    //    glHint(GL_FOG_HINT, GL_FASTEST); // Fog Hint Value
    //    glFogf(GL_FOG_START, 100.0f); // Fog Start Depth
    //    glFogf(GL_FOG_END, 400.0f); // Fog End Depth
    //    glEnable(GL_FOG);
    //    const GLubyte *extensions = glGetString(GL_EXTENSIONS);
    //    fprintf(stderr, "GL extensions: %s\n", (const char*) extensions);

    return (TRUE);
}

GLfloat seconds;

void setupGlScene() {
    int pixelPerfect;

    clearR += (avgR - clearR) * 0.05;
    clearG += (avgR - clearG) * 0.05;
    clearB += (avgB - clearB) * 0.05;
    //    fprintf(stderr, "avgColor r=%f, g=%f, b=%f, clearColor r=%f, g=%f, b=%f\n", avgR, avgG, avgB, clearR, clearG, clearB);
    glClearColor(clearR, clearG, clearB, 1.0);

    glClear(GL_COLOR_BUFFER_BIT);

    glLoadIdentity();

    GLfloat scale = canvas.scale;
    if (fabs(canvas.rotx) > 0.1) {
        scale += fabs(canvas.rotx) / 35.0;
    }
    if (fabs(canvas.roty) > 0.1) {
        scale += fabs(canvas.roty) / 25.0;
    }
    glTranslatef(-(scale - 1.0) * SCREEN_WIDTH / 2 - SCREEN_WIDTH / 2, -(scale - 1.0) * SCREEN_HEIGHT / 2 - SCREEN_HEIGHT / 2, -100.0f);
    glScalef(scale, scale, scale);

    glTranslatef(SCREEN_WIDTH / 2, SCREEN_HEIGHT / 2, 0);
    if (fabs(canvas.rotx) < 0.01 && (fabs(canvas.rotz) < 0.01 || fabs(canvas.rotz + 90.0) < 0.01) && fabs(canvas.roty) < 0.01) {
        glRotatef(canvas.rotz, 0.0, 0.0, 1.0);
        pixelPerfect = TRUE;
    } else {
        glRotatef(canvas.rotx, 1.0, 0.0, 0.0);
        glRotatef(canvas.roty, 0.0, 1.0, 0.0);
        glRotatef(canvas.rotz, 0.0, 0.0, 1.0);
        pixelPerfect = FALSE;
    }
    glTranslatef(-SCREEN_WIDTH / 2, -SCREEN_HEIGHT / 2, 0);

    if (canvas.scale - 1.0 < 0.01 && pixelPerfect) {
        // for pixel perfect representation of tiles in 2d mode
        glTranslatef((int) canvas.x - TILE_SIZE, (int) canvas.y - TILE_SIZE, 0.0f);
    } else {
        glTranslatef(canvas.x - TILE_SIZE, canvas.y - TILE_SIZE, 0.0f);
    }

    glGetFloatv(GL_PROJECTION_MATRIX, projectionMatrix);
    glGetFloatv(GL_MODELVIEW_MATRIX, modelMatrix);
    glGetIntegerv(GL_VIEWPORT, viewPort);
}

void drawGLScene() {
    static GLint oldTicks = 0;
    static GLint frames = 0;
    int i, j;

    for (i = 0; i < TILES_X; i++) {
        for (j = 0; j < TILES_Y; j++) {
            if (tiles[i][j][currentTilesIdx] != NULL) {
                if (tiles[i][j][currentTilesIdx] -> visible == TRUE) {
                    glTranslatef((i - getCanvasShiftX()) * TILE_SIZE, (j - getCanvasShiftY()) * TILE_SIZE, 0);
                    drawTile(tiles[i][j][currentTilesIdx]);
                    glTranslatef(-(i - getCanvasShiftX()) * TILE_SIZE, -(j - getCanvasShiftY()) * TILE_SIZE, 0);
                }
            }
        }
    }

    drawBuildings();

    if (position -> status != UI_HIDDEN) {
        // Current position
        drawUiElement(position);

        // Arrow showing direction of movement
        if ((device -> fix -> fields | LOCATION_GPS_DEVICE_TRACK_SET) && device -> satellites_in_use > 0 && device -> fix -> speed > 1) {
            GLfloat track = device -> fix -> track - 180.0;
            glTranslatef(position -> x + position -> texture.size / 2 / canvas.scale, position -> y + position -> texture.size / 2 / canvas.scale, 0.0);
            glDisable(GL_TEXTURE_2D);
            glColor4f(0.09, 0.59, .94, 1.0);
            glRotatef(track, 0.0, 0.0, 1.0);
            directionVertices[1] = 18.0f / canvas.scale;
            directionVertices[3] = -6.0f / canvas.scale;
            directionVertices[4] = 6.0f / canvas.scale;
            directionVertices[6] = 6.0f / canvas.scale;
            directionVertices[7] = 6.0f / canvas.scale;

            glVertexPointer(3, GL_FLOAT, 0, directionVertices);
            glDrawArrays(GL_TRIANGLES, 0, 3);
            glRotatef(-track, 0.0, 0.0, 1.0);
            glTranslatef(-position -> x - position -> texture.size / 2, -position -> y - position -> texture.size / 2, 0.0);
        }
    }

    // Small grid of all tiles coloured by their state. Also a bounding trapezoid
    if (options.showDebugTiles) {
        glLoadIdentity();
        glTranslatef(300, 0, -100);
        glScalef(0.05, 0.05, 0.05);

        glEnable(GL_BLEND);
        glBlendFunc(GL_ONE, GL_ONE_MINUS_SRC_ALPHA);
        setQuadSize(TILE_SIZE, TILE_SIZE, quadVertices);
        glBindTexture(GL_TEXTURE_2D, 0);

        glTranslatef(-getCanvasShiftX() * TILE_SIZE, -getCanvasShiftY() * TILE_SIZE, 0);

        for (i = 0; i < TILES_X; i++) {
            for (j = 0; j < TILES_Y; j++) {
                glTranslatef(i * TILE_SIZE, j * TILE_SIZE, 0);
                drawTile_debug(tiles[i][j][currentTilesIdx]);
                glTranslatef(-i * TILE_SIZE, -j * TILE_SIZE, 0);
            }
        }

        for (i = 0; i < MAX_ZOOM_HELPER_TILES; i++) {
            glTranslatef(-3 * TILE_SIZE, i * TILE_SIZE, 0);
            drawTile_debug(currentPositionZoomHelperTiles[i]);
            glTranslatef(TILE_SIZE, 0, 0);
            drawTile_debug(destinationZoomHelperTiles[i]);
            glTranslatef(2 * TILE_SIZE, -i * TILE_SIZE, 0);
        }

        if (!debugTilesGridInitialized) {
            int idx = 0;
            // vertical lines for canvas grid
            for (i = 0; i < TILES_X + 1; i++) {
                debugTilesGrid[idx++] = i * TILE_SIZE;
                debugTilesGrid[idx++] = 0.0;
                debugTilesGrid[idx++] = 0.0;

                debugTilesGrid[idx++] = i * TILE_SIZE;
                debugTilesGrid[idx++] = TILES_Y * TILE_SIZE;
                debugTilesGrid[idx++] = 0.0;
            }

            // horizontal lines for canvas grid
            for (i = 0; i < TILES_Y + 1; i++) {
                debugTilesGrid[idx++] = 0.0;
                debugTilesGrid[idx++] = i * TILE_SIZE;
                debugTilesGrid[idx++] = 0.0;

                debugTilesGrid[idx++] = TILES_X * TILE_SIZE;
                debugTilesGrid[idx++] = i * TILE_SIZE;
                debugTilesGrid[idx++] = 0.0;
            }

            // vertical lines for zoom helper grid
            for (i = 0; i < 3; i++) {
                debugTilesGrid[idx++] = (i - 3) * TILE_SIZE;
                debugTilesGrid[idx++] = 0.0;
                debugTilesGrid[idx++] = 0.0;

                debugTilesGrid[idx++] = (i - 3) * TILE_SIZE;
                debugTilesGrid[idx++] = MAX_ZOOM_HELPER_TILES * TILE_SIZE;
                debugTilesGrid[idx++] = 0.0;
            }

            // horizontal lines for zoom helper grid
            for (i = 0; i < MAX_ZOOM_HELPER_TILES + 1; i++) {
                debugTilesGrid[idx++] = -3 * TILE_SIZE;
                debugTilesGrid[idx++] = i * TILE_SIZE;
                debugTilesGrid[idx++] = 0.0;

                debugTilesGrid[idx++] = -1 * TILE_SIZE;
                debugTilesGrid[idx++] = i * TILE_SIZE;
                debugTilesGrid[idx++] = 0.0;
            }

            debugTilesGridInitialized = TRUE;
        }
        glColor4f(0.4, 0.4, 0.4, .7);
        glVertexPointer(3, GL_FLOAT, 0, debugTilesGrid);
        glDrawArrays(GL_LINES, 0, (TILES_X + TILES_Y + 2 + MAX_ZOOM_HELPER_TILES * 2) * 2);

        drawBoundingBox();
        glScalef(20, 20, 20);
        glTranslatef(-150, -20, 0);
        sprintf(buf, "allocated tiles: %d, knot pos: %.2f, z%d", allCreatedTiles -> length, canvas.zoomKnotPosition, canvas.zoom);
        drawString(buf, .2, .0, 0.5, TRUE);
        glTranslatef(0, -20, 0);
        sprintf(buf, "scale: %f, destscale: %.2f", canvas.scale, canvas.destScale);
        drawString(buf, .2, .0, 0.5, TRUE);
    }

    glLoadIdentity();
    glTranslatef(-SCREEN_WIDTH / 2, -SCREEN_HEIGHT / 2, -100);

    glEnable(GL_BLEND);
    glEnable(GL_TEXTURE_2D);
    glBindTexture(GL_TEXTURE_2D, 0);
    glBlendFunc(GL_SRC_COLOR, GL_ZERO);

    // landscape fake fog
    if (fabs(canvas.rotx) > 0.01) {
        glTranslatef(0, -((LANDSCAPE_ROTATION_X - canvas.rotx) / LANDSCAPE_ROTATION_X) * 200, 0);
        glEnableClientState(GL_COLOR_ARRAY);
        glColorPointer(4, GL_FLOAT, 0, fakeFogColorsLandscape);
        setQuadSize(SCREEN_WIDTH, 200, quadVertices);
        glVertexPointer(3, GL_FLOAT, 0, quadVertices);
        glDrawArrays(GL_TRIANGLE_STRIP, 0, 4);
        glDisableClientState(GL_COLOR_ARRAY);
        glTranslatef(0, +((LANDSCAPE_ROTATION_X - canvas.rotx) / LANDSCAPE_ROTATION_X) * 200, 0);
    }

    // portrait fake fog
    if (fabs(canvas.roty) > 0.01) {
        GLfloat fogStart = ((PORTRAIT_ROTATION_Y + canvas.roty) / PORTRAIT_ROTATION_Y) * 250 - 80;
        if (fogStart < 0.0) {
            glColor4f(0.0, 0.0, 0.0, 1.0);
            setQuadSize(-fogStart, SCREEN_HEIGHT, quadVertices);
            glVertexPointer(3, GL_FLOAT, 0, quadVertices);
            glDrawArrays(GL_TRIANGLE_STRIP, 0, 4);
        }

        glTranslatef(-fogStart, 0, 0);
        glEnableClientState(GL_COLOR_ARRAY);
        glColorPointer(4, GL_FLOAT, 0, fakeFogColorsPortrait);
        setQuadSize(250, SCREEN_HEIGHT, quadVertices);
        glVertexPointer(3, GL_FLOAT, 0, quadVertices);
        glDrawArrays(GL_TRIANGLE_STRIP, 0, 4);
        glDisableClientState(GL_COLOR_ARRAY);
        glTranslatef(fogStart, 0, 0);
    }

    drawUiElems();
    drawZoomBar();
    drawSearchBar();
    //glLoadIdentity();
    drawStatusBar(LANDSCAPE);
    glLoadIdentity();
    glTranslatef(-SCREEN_WIDTH / 2, -SCREEN_HEIGHT / 2, -100);
    drawStatusBar(PORTRAIT);

    glLoadIdentity();
    glTranslatef(15, 15, -100);
    if (options.showCoordinates) {
        sprintf(buf, "%02.6f %02.6f", canvas.center.latitude, canvas.center.longitude);
        drawString(buf, 0.4, 0.4, 0.4, TRUE);
    }

    frames++;
    {
        GLint t = SDL_GetTicks();
        if (t - oldTicks >= 1000) {
            seconds = (t - oldTicks) / 1000.0;
            fps = frames / seconds;
            oldTicks = t;
            frames = 0;
            //            fprintf(stderr, "fps: %f\n", fps);
        }
    }
    SDL_GLES_SwapBuffers();
}
