#include "mainwindow.h"
#include "ui_mainwindow.h"
#include "ClassicPrintSettings.h"
#include "lensconfiguration.h"
#include "filmconfiguration.h"
#include "processingconfiguration.h"
#include "settingsdialog.h"

#include <QDomDocument>
#include <QDomElement>

#include <QTimer>
#include <QTime>
#include <QDir>
#include <QFileInfo>
#include <QFile>
#include <QPainter>
#include <QFuture>
#include <QtConcurrentRun>

MainWindow::MainWindow(QWidget *parent) :
    QMainWindow(parent),
    ui(new Ui::MainWindow)
{
	QIcon			film_icon;
	QIcon			lens_icon;
	QIcon			processing_icon;
	QIcon			close_icon;
	QIcon			settings_icon;

	m_maximum_to_process = 0;
	m_process_preview = true;
	m_camera_mp = 3;
	m_keep_negatives = true;

	// Initialise the ClassicPrint library
	ClassicPrint::init();

	ui->setupUi(this);

	// Set some default settings for the case where there is not configuration file
	m_negatives_folder = "/home/user/MyDocs/ClassicPhotos/Negatives/";
	m_developed_folder = "/home/user/MyDocs/ClassicPhotos/Developed/";

	// Load the initial settings
	load_settings();

	film_icon.addFile(ClassicPrintSettings::config_dir() + "/icons/Film.png", QSize(), QIcon::Normal);
	film_icon.addFile(ClassicPrintSettings::config_dir() + "/icons/Film_dimmed.png", QSize(), QIcon::Disabled);
	lens_icon.addFile(ClassicPrintSettings::config_dir() + "/icons/Lens.png", QSize(), QIcon::Normal);
	lens_icon.addFile(ClassicPrintSettings::config_dir() + "/icons/Lens_dimmed.png", QSize(), QIcon::Disabled);
	processing_icon.addFile(ClassicPrintSettings::config_dir() + "/icons/Processing.png", QSize(), QIcon::Normal);
	processing_icon.addFile(ClassicPrintSettings::config_dir() + "/icons/Processing_dimmed.png", QSize(), QIcon::Disabled);
	close_icon.addFile(ClassicPrintSettings::config_dir() + "/icons/camera_overlay_close.png", QSize(), QIcon::Normal);
	settings_icon.addFile(ClassicPrintSettings::config_dir() + "/icons/camera_camera_setting.png", QSize(), QIcon::Normal);

	ui->lens_button->setIcon(lens_icon);
	ui->film_button->setIcon(film_icon);
	ui->processing_button->setIcon(processing_icon);
	ui->close_button->setIcon(close_icon);
	ui->settings_button->setIcon(settings_icon);

	// Connect the camera signals
	connect(ui->camera, SIGNAL(photo_start()), this, SLOT(camera_photo_start()));
	connect(ui->camera, SIGNAL(preview(QImage*)), this, SLOT(camera_preview(QImage*)));
	connect(ui->camera, SIGNAL(photo_taken(QString)), this, SLOT(camera_photo_taken(QString)));

	// Hide the developing text until we are actually developing something
	ui->developing->setText(" ");

	// Setup the preview window
	QFont font;
	font.setPointSize(32);
	font.setBold(true);
	m_preview_window = new QLabel(this);
	m_preview_window->setAttribute(Qt::WA_Maemo5StackedWindow);
	m_preview_window->setWindowFlags(m_preview_window->windowFlags() | Qt::Window);
	m_preview_window->setAlignment(Qt::AlignHCenter | Qt::AlignVCenter);
	m_preview_window->setFont(font);
	m_preview_window->setText("Processing...");

	// Setup the camera modes
	ui->flash_button->set_flash_mode(ui->camera->get_flash_mode());
	ui->focus_button->set_focus_mode(ui->camera->get_focus_mode());
	connect(ui->flash_button, SIGNAL(flash_mode_changed(CAMERA_FLASH_MODE)), this, SLOT(set_flash_mode(CAMERA_FLASH_MODE)));
	connect(ui->focus_button, SIGNAL(focus_mode_changed(CAMERA_FOCUS_MODE)), this, SLOT(set_focus_mode(CAMERA_FOCUS_MODE)));

	// Ensure the output directories exists
	if (m_negatives_folder.size() > 0) {
		QDir d;
		d.mkpath(m_negatives_folder);
	}
	if (m_developed_folder.size() > 0) {
		QDir d;
		d.mkpath(m_developed_folder);
	}

	// Make sure the main window is a stacked window
	setAttribute(Qt::WA_Maemo5StackedWindow);

	QTimer::singleShot(500, this, SLOT(start_camera()));

	// Fire off the processing thread in-case there are any negatives to process
	process_negatives();
}

MainWindow::~MainWindow()
{
	delete m_preview_window;
    delete ui;
	printf("Terminating main window\n");
	ClassicPrintSettings::m_app_terminating = true;
}

void MainWindow::changeEvent(QEvent *e)
{
    QMainWindow::changeEvent(e);
    switch (e->type()) {
    case QEvent::LanguageChange:
        ui->retranslateUi(this);
        break;
    default:
        break;
    }
}

void MainWindow::start_camera() {
	ui->camera->start(m_negatives_folder, "cc");
}

void MainWindow::load_settings() {
	// Look in the home directory first
	QString dir = QDir::homePath();
	dir += "/.classicprint";

	QFileInfo info_dir(dir);
	QString cam_settings(dir);
	dir += "/settings.xml";
	cam_settings += "/classiccam.xml";

	QFileInfo info_file(dir);

	// See if the file exists
	if (info_file.exists()) {
		// Try to load it
		if (m_print.load(dir)) {
			// Attempt to load the camera settings from the same folder. Don't worry if
			// they're not there
			load(cam_settings);
			// Done
			return;
		}
		QFile::remove(dir);
	}
	// If here then the file is either invalid or doesn't exist
	m_print.load(ClassicPrintSettings::config_dir() + "/settings.xml");
	// Force a save to create the entry in the home directory
	save_settings();
}

void MainWindow::save_settings() {
	QString dir = QDir::homePath();
	dir += "/.classicprint";
	// Make the directory
	QDir d;
	d.mkpath(dir);

	// Save the file
	m_print.save(QString(dir + "/settings.xml"));

	// Save the camera settings
	save(QString(dir + "/classiccam.xml"));
}

void MainWindow::on_lens_button_clicked()
{
	ui->camera->stop();
	LensConfiguration lens_config(&m_print, this);
	if (lens_config.exec() == QDialog::Accepted) {
		save_settings();
	}
	ui->camera->start(m_negatives_folder, "cc");
}

void MainWindow::on_film_button_clicked()
{
	ui->camera->stop();
	FilmConfiguration film_config(&m_print, this);
	if (film_config.exec() == QDialog::Accepted) {
		save_settings();
	}
	ui->camera->start(m_negatives_folder, "cc");
}

void MainWindow::on_processing_button_clicked()
{
	ui->camera->stop();
	ProcessingConfiguration proc_config(&m_print, this);
	if (proc_config.exec() == QDialog::Accepted) {
		save_settings();
	}
	ui->camera->start(m_negatives_folder, "cc");
}


void MainWindow::paintEvent(QPaintEvent* evt) {
	Q_UNUSED(evt)

	QPainter paint(this);
	QImage img(ClassicPrintSettings::config_dir() + "/InstamaticBack.jpg");

	paint.drawImage(0, 0, img);
}

void MainWindow::on_close_button_clicked()
{
	close();
}

//--------------------------------------------------------------------------
/*!
** @brief		Signal that is emitted when a still image has been captured
**
** @param[in]	image	Preview image that can be displayed on screen
**
*/
void MainWindow::camera_preview(QImage* image) {
	// Display a preview image to the user
	QPixmap pix;
	if (m_process_preview) {
		QImage new_image;
		m_print.process(*image, image->width(), image->height(), new_image);
		new_image = new_image.scaledToWidth(500);
		pix = QPixmap::fromImage(new_image);
	}
	else {
		pix = QPixmap::fromImage(*image);
	}

	m_preview_window->setText("");
	m_preview_window->setPixmap(pix);
	m_preview_window->showNormal();

	QTime dieTime = QTime::currentTime().addSecs(5);
	while(QTime::currentTime() < dieTime) {
		QCoreApplication::processEvents(QEventLoop::AllEvents, 100);
		if (m_preview_window->isMinimized() ||
			m_preview_window->isHidden()) {
			break;
		}
	}

	m_preview_window->hide();
	m_preview_window->setPixmap(QPixmap());
	m_preview_window->setText("Processing...");

	repaint();
}

//--------------------------------------------------------------------------
/*!
** @brief		Signal that is emitted at the start of image capture
**
*/
void MainWindow::camera_photo_start() {
	// Show a message to the user to tell them the photo is being processed
	m_preview_window->showFullScreen();
}

//--------------------------------------------------------------------------
/*!
** @brief		Signal that is emitted after a photo is written to disk
**
** @param[in]	filename	Filename of image that has been written
**
*/
void MainWindow::camera_photo_taken(QString filename) {
	// Save the current settings along with the photo
	QFileInfo info(filename);
	QString		settings;

	settings = info.path() + QString("/") + info.baseName() + QString(".cps");
	printf("Saving settings: %s\n", settings.toAscii().constData());
	m_print.save(settings, true);

	// Trigger the negatives to be processed
	process_negatives();
}

void MainWindow::process_negative_thread() {
	// See if there are any negatives to process
	QDir negatives(m_negatives_folder);
	QStringList filter;
	filter << "*.cps";

	// Look for all the settings files. There should be a 1:1 correspondence
	// between settings and images
	QStringList files = negatives.entryList(filter);
	while (files.size() > 0) {
		QImage		print;
		QString		in_path(m_negatives_folder);
		QString		in_file;
		QString		in_settings;

		// Ensure the settings file exits
		in_settings = in_path + files[0];
		printf("Processing %s\n", in_settings.toAscii().constData());
		QFileInfo	info(in_settings);
		if (info.isFile()) {
			// Now read the image file with the same base name
			in_file = in_path + info.baseName() + QString(".jpg");
			QImage*		negative = new QImage(in_file);

			// Show the developing text
			QString s("Developing ");
			s += QVariant(m_maximum_to_process - files.size() + 1).toString() +
				 QString(" of ") + QVariant(m_maximum_to_process).toString();
			ui->developing->setText(s);

			// Make sure the maximum count is up to date
			if (files.size() > m_maximum_to_process) {
				m_maximum_to_process = files.size();
			}

			printf("Processing negative %s\n", files[0].toAscii().constData());
			if (!negative->isNull()) {
				int width = 2048;
				int height = 1536;
				// Configure the settings for this print
				ClassicPrint	print_processor;
				bool settings_valid = print_processor.load(in_settings);

				// Find the developed folder from the settings
				QString out_file(m_developed_folder);
				if (out_file.size() == 0) {
					// Cannot process if there is no output directory
					continue;
				}
				out_file += info.baseName() + QString(".jpg");

				if (settings_valid) {
					connect(this, SIGNAL(destroyed()), &print_processor, SLOT(on_cancel()));
					if (m_camera_mp == 5) {
						width = 2576;
						height = 1936;
						printf("5Mp\n");
					}
					else if (m_camera_mp == 1) {
						width = 1280;
						height = 960;
						printf("1Mp\n");
					}
					else {
						printf("3Mp\n");
					}
					if (print_processor.process(*negative, width, height, print)) {
						print.save(out_file);
					}
					disconnect(this, SIGNAL(destroyed()), &print_processor, SLOT(on_cancel()));
				}
				else {
					// Loaded settings were invalid. Use the current settings
					connect(this, SIGNAL(destroyed()), &m_print, SLOT(on_cancel()));
					if (m_camera_mp == 5) {
						width = 2576;
						height = 1936;
					}
					else if (m_camera_mp == 1) {
						width = 1280;
						height = 960;
					}
					if (m_print.process(*negative, width, height, print)) {
						print.save(out_file);
					}
					disconnect(this, SIGNAL(destroyed()), &m_print, SLOT(on_cancel()));
				}

				//disconnect(&m_print, SIGNAL(progress(int)), this, SLOT(negative_progress(int)));
			}
			// Delete the negative
			delete negative;
		}
		if (ClassicPrintSettings::m_app_terminating) {
			break;
		}
		if (!m_keep_negatives) {
			QFile::remove(in_file);
		}
		QFile::remove(in_settings);

		// Look to see if there are any files remaining
		files = negatives.entryList(filter);
	}
	// No images left in negative directory. Terminate the thread for now
	// Reset the maximum count to reset the progress bar
	m_maximum_to_process = 0;
	// Hide the developing text
	ui->developing->setText(" ");

}

void MainWindow::process_negatives() {
	// See how many negatives there are
	QDir negatives(m_negatives_folder);
	QStringList filter;
	filter << "*.cps";
	QStringList files = negatives.entryList(filter);
	if (files.size() > m_maximum_to_process) {
		m_maximum_to_process = files.size();
	}
	if (files.size() > 0) {
		// If the thread is not currently running then start it now
		if (!m_future.isRunning()) {
			// If m_future is already in use, i.e. the thread is already running
			// then this call will have no effect
			m_future = QtConcurrent::run(this, &MainWindow::process_negative_thread);
		}
	}
}

//--------------------------------------------------------------------------
/*!
** @brief		Emitted by worker thread when a negative has been processed
**
**
*/
void MainWindow::negative_processed() {
}

//--------------------------------------------------------------------------
/*!
** @brief		Progress report from negative processing
**
*/
void MainWindow::negative_progress(int percent) {
#if 0
	if (m_maximum_to_process > 0) {
		int total = (100 * (m_maximum_to_process - m_num_to_process) / m_maximum_to_process) +
					(percent / m_maximum_to_process);
		ui->progress->setValue(total);
	}
#endif
}

//--------------------------------------------------------------------------
/*!
** @brief		Set the button to display the current focus mode
**
** @param[In]	New focus mode
**
*/
void MainWindow::set_focus_mode(CAMERA_FOCUS_MODE new_focus_mode) {
	ui->camera->set_focus_mode(new_focus_mode);
	save_settings();
}

//--------------------------------------------------------------------------
/*!
** @brief		Set the button to display the current flash mode
**
** @param[In]	New flash mode
**
*/
void MainWindow::set_flash_mode(CAMERA_FLASH_MODE new_flash_mode) {
	ui->camera->set_flash_mode(new_flash_mode);
	save_settings();
}


void MainWindow::on_settings_button_clicked()
{
	ui->camera->stop();
	SettingsDialog settings(this, this);
	if (settings.exec() == QDialog::Accepted) {
		save_settings();
	}
	ui->camera->start(m_negatives_folder, "cc");
}

//---------------------------------------------------------------------------
/*!
** @brief   Save configuration to a file
**
** @param[In] filename  Filename to save to
**
** @return True/False
*/
bool MainWindow::save(QString filename) {
	QDomDocument    doc("ClassicCam");

	// Create the root element
	QDomElement     root = doc.createElement("ClassicCam");
	doc.appendChild(root);

	if (m_negatives_folder.size() > 0) {
		QDomElement	neg_dir = doc.createElement("NegativesFolder");
		QDomText	neg_text = doc.createTextNode(m_negatives_folder);
		neg_dir.appendChild(neg_text);
		root.appendChild(neg_dir);
	}
	if (m_developed_folder.size() > 0) {
		QDomElement	dev_dir = doc.createElement("DevelopedFolder");
		QDomText	dev_text = doc.createTextNode(m_developed_folder);
		dev_dir.appendChild(dev_text);
		root.appendChild(dev_dir);
	}
	QDomElement	elem_keep = doc.createElement("KeepNegatives");
	QDomText	text_keep = doc.createTextNode(m_keep_negatives ? "1" : "0");
	elem_keep.appendChild(text_keep);
	root.appendChild(elem_keep);

	QDomElement	elem_prev = doc.createElement("ProcessPreview");
	QDomText	text_prev = doc.createTextNode(m_process_preview ? "1" : "0");
	elem_prev.appendChild(text_prev);
	root.appendChild(elem_prev);

	QDomElement	elem_mp = doc.createElement("CameraMp");
	QDomText	text_mp = doc.createTextNode(QVariant(m_camera_mp).toString());
	elem_mp.appendChild(text_mp);
	root.appendChild(elem_mp);

	QDomElement elem_flash = doc.createElement("FlashMode");
	QDomText	text_flash = doc.createTextNode(QVariant((int)ui->camera->get_flash_mode()).toString());
	elem_flash.appendChild(text_flash);
	root.appendChild(elem_flash);

	QDomElement elem_focus = doc.createElement("FocusMode");
	QDomText	text_focus = doc.createTextNode(QVariant((int)ui->camera->get_focus_mode()).toString());
	elem_focus.appendChild(text_focus);
	root.appendChild(elem_focus);

	// Write it to the file
	QString xml(doc.toString());
	QFile file(filename);
	if (!file.open(QIODevice::ReadWrite | QIODevice::Truncate)) {
		return false;
	}
	file.write(xml.toUtf8());

	file.close();

	return true;
}

//---------------------------------------------------------------------------
/*!
** @brief   Load configuration from a file
**
** @param[In] filename  Filename to load from
**
** @return True/False
*/
bool MainWindow::load(QString filename) {
	QDomDocument doc("ClassicCam");
	QFile file(filename);
	if (!file.open(QIODevice::ReadOnly)) {
		return false;
	}
	if (!doc.setContent(&file)) {
		return false;
	}

	// Get the root element
	QDomElement root = doc.documentElement();
	if (root.tagName() != "ClassicCam") {
		return false;
	}
	// Iterate through all elements
	QDomNode n = root.firstChild();
	while (!n.isNull()) {
		// Try to convert the node to an element.
		QDomElement e = n.toElement();
		if(!e.isNull()) {
			if (e.tagName() == "NegativesFolder") {
				m_negatives_folder = e.text();
			}
			else if (e.tagName() == "DevelopedFolder") {
				m_developed_folder = e.text();
			}
			else if (e.tagName() == "KeepNegatives") {
				m_keep_negatives = e.text().toInt() ? true : false;
			}
			else if (e.tagName() == "ProcessPreview") {
				m_process_preview = e.text().toInt() ? true : false;
			}
			else if (e.tagName() == "CameraMp") {
				m_camera_mp = e.text().toInt();
			}
			else if (e.tagName() == "FlashMode") {
				ui->camera->set_flash_mode((CAMERA_FLASH_MODE)e.text().toInt());
			}
			else if (e.tagName() == "FocusMode") {
				ui->camera->set_focus_mode((CAMERA_FOCUS_MODE)e.text().toInt());
			}
		}
		n = n.nextSibling();
	}
	return true;
}
