/*!
** @file	camera.h
**
** @brief	Class that manages the camera on the N900
**
** Handles display of video preview as well as taking snapshots with
** the camera shutter button
*/
#ifndef __camera__h
#define __camera__h

/*---------------------------------------------------------------------------
** Includes
*/
#include <gst/gst.h>
#include <gst/interfaces/xoverlay.h>
#include <QX11EmbedContainer>
#include <QString>

/*---------------------------------------------------------------------------
** Defines and Macros
*/

/*---------------------------------------------------------------------------
** Typedefs
*/
class QObject;
class QWidget;
class QImage;

//!
//! DBus property type
//!
struct Property
{
	QString name;
	bool added;
	bool removed;
};

enum CAMERA_FLASH_MODE {
	CAMERA_FLASH_MODE_AUTO = 0,
	CAMERA_FLASH_MODE_OFF,
	CAMERA_FLASH_MODE_ON,
	CAMERA_FLASH_MODE_FILL_IN,
	CAMERA_FLASH_MODE_RED_EYE
};

enum CAMERA_FOCUS_MODE {
	CAMERA_FOCUS_MODE_AUTO,
 	CAMERA_FOCUS_MODE_FIXED
};

/*---------------------------------------------------------------------------
** Local function prototypes
*/

/*---------------------------------------------------------------------------
** Data
*/

class Camera : public QWidget {
	Q_OBJECT

public:
	//--------------------------------------------------------------------------
	/*!
	** @brief   	Constructor
	*/
	Camera(QWidget* parent = NULL);

	//--------------------------------------------------------------------------
	/*!
	** @brief   	Destructor
	*/
	virtual ~Camera();

	//-------------------------------------------------------------------------- 
	/*! 
	** @brief		Start the video preview and prepare for taking
	**				a snapshot
	**
	** @param[In]	image_directory		Folder to store photos in
	**
	** @param[In]	base_name			Base name for images
	**
	*/
	void start(QString image_directory, QString base_name);

	//--------------------------------------------------------------------------
	/*! 
	** @brief		Stop the video preview
	*/
	void stop();
	
	//-------------------------------------------------------------------------- 
	/*! 
	** @brief   	Set the X Window ID for rendering video overlay
	** 
	** @param [In] id		X Window ID
	** 
	*/
	void set_x_window_id(WId id);

	//-------------------------------------------------------------------------- 
	/*! 
	** @brief   	Capture a still image from the camera and emit
	**				snapshot signal
	*/
	void capture_image();

	//-------------------------------------------------------------------------- 
	/*! 
	** @brief   	Get the current flash mode for the camera
	**
	** @return 		Flash mode enumeration
	**
	*/
	CAMERA_FLASH_MODE get_flash_mode();

	//-------------------------------------------------------------------------- 
	/*! 
	** @brief   	Set the current flash mode for the camera
	**
	** @param[in]	mode	New flash mode
	**
	*/
	void set_flash_mode(CAMERA_FLASH_MODE mode);

	//-------------------------------------------------------------------------- 
	/*! 
	** @brief   	Get the current focus mode for the camera
	**
	** @return 		Focus mode enumeration
	**
	*/
	CAMERA_FOCUS_MODE get_focus_mode();

	//-------------------------------------------------------------------------- 
	/*! 
	** @brief   	Set the current focus mode for the camera
	**
	** @param[in]	mode	New focus mode
	**
	*/
	void set_focus_mode(CAMERA_FOCUS_MODE mode);

signals:
	//-------------------------------------------------------------------------- 
	/*! 
	** @brief		Signal that is emitted when a still image has been captured
	**
	** @param[in]	image	Preview image that can be displayed on screen
	**
	*/
	void preview(QImage* image);

	//-------------------------------------------------------------------------- 
	/*! 
	** @brief		Signal that is emitted at the start of image capture
	**
	*/
	void photo_start();
	
	//-------------------------------------------------------------------------- 
	/*! 
	** @brief		Signal that is emitted at the end of image capture
	**
	*/
	void photo_end();
	
	//-------------------------------------------------------------------------- 
	/*! 
	** @brief		Signal that is emitted after a photo is written to disk
	**
	** @param[in]	filename	Filename of image that has been written
	**
	*/
	void photo_taken(QString filename);
	
public slots:
	//--------------------------------------------------------------------------
	/*!
	** @brief		Slot called by DBus when the shutter button properties change
	**
	** This class uses the slot notification to determine when the
	** shutter button has been pressed
	**
	*/
	void shutter_property_modified(int num_updates, QList<Property> updates);

	//--------------------------------------------------------------------------
	/*!
	** @brief		Slot called by DBus when the focus properties change
	**
	** This class uses the slot notification to determine when the
	** shutter button is half pressed
	**
	*/
	void focus_property_modified(int num_updates, QList<Property> updates);
	
private:
	//-------------------------------------------------------------------------- 
	/*! 
	** @brief		Initialise the gstreamer pipeline
	**
	** @return		true/false
	**
	*/
	bool initialise_pipeline();

	//-------------------------------------------------------------------------- 
	/*! 
	** @brief		Destroy the gstreamer pipeline
	**
	*/
	void destroy_pipeline();

	//-------------------------------------------------------------------------- 
	/*! 
	** @brief		Perform autofocus operation
	**
	*/
	void focus();
	
	//--------------------------------------------------------------------------
	/*!
	** @brief		Set fixed focus range to infinity
	**
	*/
	void set_fixed_focus();
	
	//-------------------------------------------------------------------------- 
	/*! 
	** @brief		Receive element signals from pipeline bus
	**
	*/
	static void	element_msg_sync(GstBus* bus, GstMessage* msg, gpointer data);

	//--------------------------------------------------------------------------
	/*!
	** @brief   	Signal from camerabin when photo has been written to file
	**
	*/
	static gboolean image_done_callback(void* bin, gchar* filename, gpointer user_data);

private:

	//!
	//! Gstreamer pipe line
	//!
	GstElement*		m_pipeline;

	//!
	//! Handle for snapshot callback
	//!
	guint 			m_snapshot_callback;

	//!
	//! Streaming video sink that for screen
	//!
	GstElement*		m_video_sink;

	//!
	//! Last snapshot taken
	//!
	QImage 			m_photo;

	//!
	//! X Window ID for viewfinder
	//!
	WId 			m_id;

	//!
	//! Current flash mode
	//!
	CAMERA_FLASH_MODE m_flash_mode;

	//!
	//! Current focus mode
	//!
	CAMERA_FOCUS_MODE m_focus_mode;

	//!
	//! The filename of the next image to be taken
	//!
	QString			m_current_filename;

	//!
	//! Directory to store images in
	//!
	QString			m_image_directory;

	//!
	//! Base filename
	//!
	QString			m_base_filename;

	bool			m_started;
};

#endif
