#ifndef MAINWINDOW_H
#define MAINWINDOW_H

#include <QMainWindow>
#include <QLabel>
#include <QFuture>
#include <QFutureWatcher>
#include "ClassicPrint.h"
#include "camera.h"

namespace Ui {
    class MainWindow;
}

class MainWindow : public QMainWindow
{
    Q_OBJECT

public:
    explicit MainWindow(QWidget *parent = 0);
    ~MainWindow();

protected:
    void changeEvent(QEvent *e);

	void load_settings();
	void save_settings();

	virtual void paintEvent(QPaintEvent* evt);

	void process_negatives();
	void process_negative_thread();

public slots:
	void start_camera();

	//--------------------------------------------------------------------------
	/*!
	** @brief		Signal that is emitted when a still image has been captured
	**
	** @param[in]	image	Preview image that can be displayed on screen
	**
	*/
	void camera_preview(QImage* image);

	//--------------------------------------------------------------------------
	/*!
	** @brief		Signal that is emitted at the start of image capture
	**
	*/
	void camera_photo_start();

	//--------------------------------------------------------------------------
	/*!
	** @brief		Signal that is emitted after a photo is written to disk
	**
	** @param[in]	filename	Filename of image that has been written
	**
	*/
	void camera_photo_taken(QString filename);

	//--------------------------------------------------------------------------
	/*!
	** @brief		Emitted by worker thread when a negative has been processed
	**
	**
	*/
	void negative_processed();

private slots:
	//--------------------------------------------------------------------------
	/*!
	** @brief		Progress report from negative processing
	**
	*/
	void on_settings_button_clicked();
 void    negative_progress(int percent);

	//--------------------------------------------------------------------------
	/*!
	** @brief		Set the button to display the current focus mode
	**
	** @param[In]	New focus mode
	**
	*/
	void set_focus_mode(CAMERA_FOCUS_MODE new_focus_mode);

	//--------------------------------------------------------------------------
	/*!
	** @brief		Set the button to display the current flash mode
	**
	** @param[In]	New flash mode
	**
	*/
	void set_flash_mode(CAMERA_FLASH_MODE new_flash_mode);

	//---------------------------------------------------------------------------
	/*!
	** @brief   Save configuration to a file
	**
	** @param[In] filename					Filename to save to
	**
	** @return True/False
	*/
	bool    save(QString filename);

	//---------------------------------------------------------------------------
	/*!
	** @brief   Load configuration from a file
	**
	** @param[In] filename  Filename to load from
	**
	** @return True/False
	*/
	bool    load(QString filename);

private:
    Ui::MainWindow *ui;

	//!
	//! Main image processing object
	//!
	ClassicPrint		m_print;

	//!
	//! Photo preview window
	//!
	QLabel*				m_preview_window;

	//!
	//! Future object to manage processing thread
	//!
	QFuture<void>		m_future;

	//!
	//! Maximum number of negatives seen in this process run
	//!
	int					m_maximum_to_process;

public:
	// Classic Cam related settings
	QString				m_negatives_folder;
	QString				m_developed_folder;
	bool				m_keep_negatives;
	int					m_camera_mp;
	//!
	//! Flag that says whether to apply classic look to preview
	//!
	bool				m_process_preview;


private slots:
	void on_close_button_clicked();
 void on_processing_button_clicked();
	void on_film_button_clicked();
	void on_lens_button_clicked();
};

#endif // MAINWINDOW_H
