/* This file is part of Cinaest.
 *
 * Copyright (C) 2009 Philipp Zabel
 *
 * Cinaest is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Cinaest is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Cinaest. If not, see <http://www.gnu.org/licenses/>.
 */

using Gtk;
using Hildon;

public class MovieListMenu : AppMenu {
	public TreeSortable sortable;
	public MovieFilter filter;
	private MovieListWindow movie_list_window;
	private Hildon.Button filter_year;
	private Hildon.Button filter_rating;
	private Hildon.Button filter_genres;
	private Gtk.Button delete_movies;

	public signal void filter_changed ();

	public MovieListMenu (MovieListWindow window) {
		movie_list_window = window;
	}

	construct {
		// Add sort buttons as view menu filters
		var sort_by_title = new RadioButton.with_label (null, _("ABC"));
		var sort_by_year = new RadioButton.with_label_from_widget (sort_by_title, _("Year"));
		var sort_by_rating = new RadioButton.with_label_from_widget (sort_by_title, _("Rating"));

		// Draw them as toggle buttons, not as radio buttons
		sort_by_title.set_mode (false);
		sort_by_year.set_mode (false);
		sort_by_rating.set_mode (false);

		// TODO - get this from GConf
		sort_by_title.set_active (true);

		// Connect signals
		sort_by_title.toggled.connect (button => {
			if (button.get_active ())
				sortable.set_sort_column_id (MovieListStore.Columns.TITLE, Gtk.SortType.ASCENDING);
		});
		sort_by_year.toggled.connect (button => {
			if (button.get_active ())
				sortable.set_sort_column_id (MovieListStore.Columns.YEAR, Gtk.SortType.DESCENDING);
		});
		sort_by_rating.toggled.connect (button => {
			if (button.get_active ())
				sortable.set_sort_column_id (MovieListStore.Columns.RATING, Gtk.SortType.DESCENDING);
		});

		add_filter (sort_by_title);
		add_filter (sort_by_year);
		add_filter (sort_by_rating);

		// Add view menu buttons
		filter_year = new Hildon.Button.with_text (SizeType.FINGER_HEIGHT, ButtonArrangement.VERTICAL, _("Filter by year"), _("Off"));
		filter_rating = new Hildon.Button.with_text (SizeType.FINGER_HEIGHT, ButtonArrangement.VERTICAL, _("Filter by rating"), _("Off"));
		filter_genres = new Hildon.Button.with_text (SizeType.FINGER_HEIGHT, ButtonArrangement.VERTICAL, _("Filter by genres"), _("Off"));
		delete_movies = new Gtk.Button.with_label (_("Delete movies"));
		var settings = new Gtk.Button.with_label (_("Settings"));

		filter_year.set_style (ButtonStyle.PICKER);
		filter_rating.set_style (ButtonStyle.PICKER);
		filter_genres.set_style (ButtonStyle.PICKER);

		// Connect signals
		filter_year.clicked.connect (on_filter_year_clicked);
		filter_rating.clicked.connect (on_filter_rating_clicked);
		filter_genres.clicked.connect (on_filter_genres_clicked);
		delete_movies.clicked.connect (() => { movie_list_window.on_delete_movies_clicked (); });
		settings.clicked.connect (on_settings_clicked);

		append (filter_year);
		append (filter_rating);
		append (filter_genres);
		append (delete_movies);
		append (settings);

		show_all ();
	}

	public MovieSource source {
		set {
			if (value.get_editable ()) {
				delete_movies.show ();
			} else {
				delete_movies.hide ();
			}
		}
	}

	public void on_filter_year_clicked () {
		var dialog = new PickerDialog (movie_list_window);
		dialog.set_title (_("Filter by year"));

		var year_store = new ListStore (1, typeof (string));
		TreeIter iter;
		for (int i = 0; i < 128; i++) {
			year_store.append (out iter);
			year_store.set_value (iter, 0, "%d".printf (1888 + i));
		}

		var selector = new TouchSelector ();
		selector.append_text_column (year_store, true);
		selector.append_text_column (year_store, true);
		if (filter.year_min > 1888 && filter.year_min < 2015) {
			selector.set_active (0, filter.year_min - 1888);
		} else {
			selector.set_active (0, 0);
		}
		if (filter.year_max > 1888 && filter.year_max < 2015) {
			selector.set_active (1, filter.year_max - 1888);
		} else {
			selector.set_active (1, 127);
		}
		dialog.set_selector (selector);

		var res = dialog.run ();
		if (res == ResponseType.OK) {
			filter.year_min = 1888 + selector.get_active (0);
			if (filter.year_min <= 1888)
				filter.year_min = 0;
			filter.year_max = 1888 + selector.get_active (1);
			if (filter.year_max >= 2015)
				filter.year_max = 0;

			if (filter.year_min == 0 && filter.year_max == 0)
				filter_year.set_value (_("Off"));
			else if (filter.year_min == 0)
				filter_year.set_value (_("Until %d").printf (filter.year_max));
			else if (filter.year_max == 0)
				filter_year.set_value (_("Since %d").printf (filter.year_min));
			else
				filter_year.set_value ("%d - %d".printf (filter.year_min, filter.year_max));

			filter_changed ();
		}
		dialog.destroy ();
	}

	public void on_filter_rating_clicked () {
		var dialog = new PickerDialog (movie_list_window);
		dialog.set_title (_("Filter by rating"));

		var selector = new TouchSelector.text ();
		for (int i = 0; i < 10; i++) {
			selector.append_text ("%d.0".printf (9 - i));
		}
		selector.set_active (0, 9 - (filter.rating_min / 10));
		dialog.set_selector (selector);

		var res = dialog.run ();
		if (res == ResponseType.OK) {
			filter.rating_min = (9 - selector.get_active (0)) * 10;

			if (filter.rating_min == 0)
				filter_rating.set_value (_("Off"));
			else
				filter_rating.set_value (_("At least %d.0").printf (filter.rating_min / 10));

			filter_changed ();
		}
		dialog.destroy ();
	}

	public void on_filter_genres_clicked () {
		var dialog = new GenreFilterDialog (movie_list_window);
		string s;

		var res = dialog.run (filter);

		if (res == ResponseType.OK) {
			s = filter.genres.to_string ();
			if (s != null)
				filter_genres.set_value (s);
			else
				filter_genres.set_value (_("Off"));

			filter_changed ();
		}
	}

	public void on_settings_clicked (Gtk.Button button) {
		var dialog = new SettingsDialog (movie_list_window);

		dialog.run ();
	}
}
