/* This file is part of Cinaest.
 *
 * Copyright (C) 2009 Philipp Zabel
 *
 * Cinaest is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Cinaest is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Cinaest. If not, see <http://www.gnu.org/licenses/>.
 */

using Hildon;
using Gtk;

class GenreFilterDialog : Gtk.Dialog {
	private enum Response {
		CLEAR = 1,
	}
	private static const GenreType[] _genre_order = {
		GenreType.SHORT, GenreType.DRAMA, GenreType.COMEDY, GenreType.DOCUMENTARY,
		GenreType.ROMANCE, GenreType.ANIMATION, GenreType.ACTION, GenreType.CRIME,
		GenreType.THRILLER, GenreType.FAMILY, GenreType.ADVENTURE, GenreType.MUSIC,
		GenreType.HORROR, GenreType.FANTASY, GenreType.WESTERN, GenreType.MYSTERY,
		GenreType.SCIFI, GenreType.MUSICAL, GenreType.WAR, GenreType.SPORT,
		GenreType.BIOGRAPHY, GenreType.HISTORY, GenreType.FILMNOIR, GenreType.REALITYTV,
		GenreType.TALKSHOW, GenreType.GAMESHOW, GenreType.NEWS, GenreType.ADULT
	};

	private PannableArea pannable;
	private Gtk.Button button_clear;
	private ToggleButton[] button_genre;
	private Genres genres;

	public GenreFilterDialog (Gtk.Window window) {
		set_transient_for (window);
		set_title (_("Filter by genres"));
	}

	construct {
		var vbox = new VBox (true, 0);

		pannable = new PannableArea ();
		pannable.add_with_viewport (vbox);

		var content_area = (VBox) get_content_area ();
		content_area.set_size_request (-1, 5*70);
		content_area.pack_start (pannable, true, true, 0);

		HBox hbox;

		button_genre = new ToggleButton[28];
		int i;
		for (i = 0; i < 28; i++) {
			if ((i % 4) == 0) {
				hbox = new HBox (false, 0);
				vbox.pack_start (hbox, true, true, 0);
			}
			button_genre[i] = new ToggleButton.with_label (Genres.genre_string (_genre_order[i]));
			hbox.pack_start (button_genre[i], true, true, 0);
			Hildon.gtk_widget_set_theme_size (button_genre[i], SizeType.FINGER_HEIGHT);
			button_genre[i].toggled.connect (on_genre_toggled);
		}

		button_clear = new Gtk.Button.with_label (_("Clear"));
		Hildon.gtk_widget_set_theme_size (button_clear, SizeType.FINGER_HEIGHT);

		add_action_widget (button_clear, Response.CLEAR);

		add_button(_("Done"), ResponseType.OK);
	}

	public new int run (MovieFilter filter) {
		int res = 0;
		int i;

		genres = filter.genres;
		for (i = 0; i < 28; i++) {
			if ((1 << _genre_order[i]) in genres.field)
				button_genre[i].set_active (true);
		}

		show_all ();

		do {
			res = base.run ();
			switch (res) {
			case Response.CLEAR:
				for (i = 0; i < 28; i++) {
					button_genre[i].set_active (false);
				}
				break;
			}
		} while (res > 0);
		if (res == ResponseType.OK) {
			if (filter.genres.field == genres.field)
				res = ResponseType.CANCEL;
			else
				filter.genres = genres;
		}
		destroy ();
		return res;
	}

	void on_genre_toggled (ToggleButton button) {
		int i;

		for (i = 0; i < 28; i++) {
			if (button_genre[i] == button)
				break;
		}
		genres.set_bit (_genre_order[i], button.get_active ());
		if (genres.field != 0)
			button_clear.show ();
		else
			button_clear.hide ();

		update_title ();
	}

	private void update_title () {
		string s = _("Filter by genres");

		if (genres.field != 0) {
			s += " - " + genres.to_string ();
		}

		set_title (s);
	}
}

