/*
 *  charmap - GTK+ based simple character map
 *
 *  Copyright (C) 2014 Peter Pichler (this application)
 *  Copyrught (C) 2010 Gabriel Schulhof (the author of scv-reader,
 *                     a Maemo package this app was partially based on)
 *  Copyright (C) 2010 The GNOME Charcter Map project (the icon)
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to Free Software Foundation Inc.,
 *  59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <gucharmap/gucharmap.h>

#include "maemo-gucharmap-chartable.h"

struct _MaemoGucharmapChartable
{
  GucharmapChartable __parent_instance__;

  GtkAdjustment *my_vadj;
  GtkAdjustment *foreign_vadj;
};

struct _MaemoGucharmapChartableClass
{
  GucharmapChartableClass __parent_class__;
};

static void maemo_gucharmap_chartable_init(MaemoGucharmapChartable *chart);
static void maemo_gucharmap_chartable_class_init(MaemoGucharmapChartableClass *chart_class);

G_DEFINE_TYPE(MaemoGucharmapChartable, maemo_gucharmap_chartable, GUCHARMAP_TYPE_CHARTABLE);

double pixel_size = 60.0; // default for 8 chars column

static double set_up_sync (const GtkAdjustment *p_src,
                           GtkAdjustment **p_dst,
                           GtkAdjustment *my_adj,
                           GtkAdjustment *foreign_adj)
{
  if (p_src == foreign_adj)
  {
    *p_dst = my_adj;
    return 1.0 / pixel_size;
  }
  else
  {
    *p_dst = foreign_adj;
    return pixel_size;
  }
}

static void sync_adj (const GtkAdjustment *src,
                      MaemoGucharmapChartable *chart)
{
  GtkAdjustment *dst = NULL;
  const double factor = set_up_sync(src, &dst, chart->my_vadj, chart->foreign_vadj);

  if (!src || !dst)
    return;

  if (  dst->upper          != factor * src->upper
     || dst->lower          != factor * src->lower
     || dst->step_increment != factor * src->step_increment
     || dst->page_increment != factor * src->page_increment
     || dst->page_size      != factor * src->page_size )
  {
    dst->upper          = factor * src->upper;
    dst->lower          = factor * src->lower;
    dst->step_increment = factor * src->step_increment;
    dst->page_increment = factor * src->page_increment;
    dst->page_size      = factor * src->page_size;
    gtk_adjustment_changed(dst);
  }
}

static void sync_adj_value (const GtkAdjustment *src,
                            MaemoGucharmapChartable *chart)
{
  GtkAdjustment *dst = NULL;
  const double factor = set_up_sync(src, &dst, chart->my_vadj, chart->foreign_vadj);

  if (!src || !dst)
    return;

  const double new_val = src->value * factor;
  if (dst->value != new_val)
  {
    dst->value = new_val;
    gtk_adjustment_value_changed(dst);
  }
}

static void move_to_new_adj (MaemoGucharmapChartable *chart,
                             GtkAdjustment **p_adj,
                             GtkAdjustment *my_adj,
                             GtkAdjustment *new_adj)
{
  if (*p_adj)
  {
    g_signal_handlers_disconnect_by_func(G_OBJECT(*p_adj), sync_adj,       chart);
    g_signal_handlers_disconnect_by_func(G_OBJECT(*p_adj), sync_adj_value, chart);
    g_signal_handlers_disconnect_by_func(G_OBJECT(my_adj), sync_adj,       chart);
    g_signal_handlers_disconnect_by_func(G_OBJECT(my_adj), sync_adj_value, chart);
    g_object_unref(*p_adj);
    *p_adj = NULL;
  }

  if (new_adj)
  {
    *p_adj = g_object_ref(new_adj);
    g_signal_connect(G_OBJECT(new_adj), "changed",       (GCallback)sync_adj,       chart);
    g_signal_connect(G_OBJECT(new_adj), "value-changed", (GCallback)sync_adj_value, chart);
    g_signal_connect(G_OBJECT(my_adj),  "changed",       (GCallback)sync_adj,       chart);
    g_signal_connect(G_OBJECT(my_adj),  "value-changed", (GCallback)sync_adj_value, chart);
  }
}

static void set_scroll_adjustments (GucharmapChartable *chart,
                                    GtkAdjustment *hadjustment,
                                    GtkAdjustment *vadjustment)
{
  MaemoGucharmapChartable *maemo_chart = MAEMO_GUCHARMAP_CHARTABLE(chart);
  move_to_new_adj(maemo_chart, &maemo_chart->foreign_vadj, maemo_chart->my_vadj, vadjustment);
}

static void maemo_gucharmap_chartable_init (MaemoGucharmapChartable *chart)
{
  chart->foreign_vadj = NULL;
  chart->my_vadj = g_object_new(GTK_TYPE_ADJUSTMENT, NULL);
  GUCHARMAP_CHARTABLE_CLASS(maemo_gucharmap_chartable_parent_class)->set_scroll_adjustments(GUCHARMAP_CHARTABLE(chart), NULL, chart->my_vadj);

  g_signal_connect(G_OBJECT(chart->my_vadj), "changed",       (GCallback)sync_adj,       chart);
  g_signal_connect(G_OBJECT(chart->my_vadj), "value-changed", (GCallback)sync_adj_value, chart);
}

static void maemo_gucharmap_chartable_class_init (MaemoGucharmapChartableClass *chart_class)
{
  GtkWidgetClass *gtkwidget_class = GTK_WIDGET_CLASS(chart_class);
  GucharmapChartableClass *gucc_class = GUCHARMAP_CHARTABLE_CLASS(chart_class);

  gucc_class->set_scroll_adjustments = set_scroll_adjustments;
  gtkwidget_class->button_press_event = GTK_WIDGET_CLASS(maemo_gucharmap_chartable_parent_class)->button_press_event;
}

void maemo_gucharmap_chartable_set_font(MaemoGucharmapChartable *chart, const char *name, int size)
{
  char *str = g_strdup_printf("%s %d", name, size);
  PangoFontDescription *desc = pango_font_description_from_string(str);

  gucharmap_chartable_set_font_desc(GUCHARMAP_CHARTABLE(chart), desc);

  pango_font_description_free(desc);
  g_free(str);
}

void maemo_gucharmap_chartable_set_scroll_adjustments (MaemoGucharmapChartable *chart, double size)
{
  pixel_size = size;
}
