// case - file manager for N900
// Copyright (C) 2010 Lukas Hrazky <lukkash@email.cz>
// 
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.


#ifndef OPERATIONTHREAD_H
#define OPERATIONTHREAD_H

#include <QThread>
#include <QFileInfo>
#include <QMap>
#include <QSet>
#include <QFSFileEngine>
#include <QDir>
#include <QMutex>
#include <QWaitCondition>


class OperationThread : public QThread {
    Q_OBJECT

public:
    // DONT_ASK_ONCE is a hackish way to avoid asking twice to overwrite the same directory when moving
    enum Response{NONE, ABORT, RETRY, IGNORE, KEEP, OVERWRITE, SKIP_DIR, ASK, DONT_ASK_ONCE};

    explicit OperationThread(const QFileInfoList files, QDir dest = QDir());
    void setResponse(const Response response, const bool appyToAll = false, const int err = 0);

    void wake();

    // the new name entered from the overwrite dialog
    QString newNameFromDialog;
    // flags to abort/pause the operation
    bool abort, pause;

protected:
    void processFiles(const QFileInfoList &files);
    virtual void perform(const QFileInfo &file) = 0;

    bool remove(QString &fileName, const bool doUpdates = false);
    bool remove(const QFileInfoList &files, const bool doUpdates = false);
    bool remove(const QFileInfo &file, const bool doUpdates = false);

    void copy(const QFileInfo &file);

    unsigned int calculateFileSize(const QFileInfoList &files,
        const bool count = false,
        const bool addSize = false);

    QFileInfoList listDirFiles(const QString &dirPath);

    void setTotalSize(unsigned int size);
    void updateProgress(int value);
    void updateFile(const QString &name);

    void waitOnCond();

    bool checkSequentialFile(const QFSFileEngine &engine);

    QMutex mutex;
    QWaitCondition waitCond;

    // files to process by the operation
    const QFileInfoList files;
    // destination for files - changes as the operation recurses into directories
    QDir dest;

    // responses from the dialog prompts (error and overwrite)
    Response response;
    Response overwriteAll;
    // an array indicating whether to always ignore the error of index errno
    bool ignoreAll[256];

    // set of files that won't be deleted by the remove(...) functions
    // used when move(...) would not overwrite target file to ensure the source file doesn't get deleted
    QSet<QString> removeExcludeFiles;

    // A map of file paths to their size. Not the actual size, but what is calculated for the
    // purpose of the progressbar for the given operation. So either fileSize/BLOCK_SIZE or simply
    // 1 for a file and file count for dirs (or both for copy&delete)
    QMap<QString, qint64> fileSizeMap;

    // progress information of the bar and for the current file
    unsigned int totalSize, totalValue, fileValue;

signals:
    void showErrorPrompt(OperationThread*, const QString&, const QString&, const int);
    void showOverwritePrompt(OperationThread*, const QString&, const bool);
    void showInputFilenamePrompt(OperationThread*, const QFileInfo&, const bool);
    void finished(OperationThread*);

    void totalSizeChanged(int);
    void progressUpdate(int);
    void fileNameUpdated(QString);
    void operationStarted(time_t);
    void operationPaused();
    void operationResumed(time_t);
    // special signal emitted when operation has to copy files between partitions
    // to notify the user it is deleting files after succesful copy (which can take some time)
    void removeAfterCopy();
};


class DeleteThread : public OperationThread {
    Q_OBJECT

public:
    explicit DeleteThread(const QFileInfoList &files) : OperationThread(files) {}

protected:
    void run();
    virtual void perform(const QFileInfo &file);
};


class CopyThread : public OperationThread {
    Q_OBJECT

public:
    explicit CopyThread(const QFileInfoList &files, QDir &dest) : OperationThread(files, dest) {}

protected:
    void run();
    virtual void perform(const QFileInfo &file);
};


class MoveThread : public OperationThread {
    Q_OBJECT

public:
    explicit MoveThread(const QFileInfoList &files, QDir &dest) : OperationThread(files, dest) {}

protected:
    void run();
    virtual void perform(const QFileInfo &file);
    void rename(const QFileInfoList &files, const QDir &dest);
};

#endif // OPERATIONTHREAD_H
