// case - file manager for N900
// Copyright (C) 2010 Lukas Hrazky <lukkash@email.cz>
// 
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.


#include "fileoperator.h"

#include <QtGui>
#include <QMessageBox>
#include <QVBoxLayout>
#include <QMaemo5InformationBox>

#include "dialog.h"
#include "utils.h"


FileOperator::FileOperator(QWidget *parent) : QWidget(parent) {
    QVBoxLayout *layout = new QVBoxLayout;
    layout->setContentsMargins(0, 0, 0, 0);
    layout->setSpacing(1);
    topRow = new QHBoxLayout;
    topRow->setContentsMargins(0, 0, 0, 0);
    topRow->setSpacing(1);
    layout->addLayout(topRow);
    bottomRow = new QHBoxLayout;
    bottomRow->setContentsMargins(0, 0, 0, 0);
    bottomRow->setSpacing(1);
    layout->addLayout(bottomRow);
    setLayout(layout);

    qRegisterMetaType<QFileInfo>("QFileInfo");
    qRegisterMetaType<time_t>("time_t");

    loadOperationIcons(palette(), "delete_small", deleteIcon, inverseDeleteIcon);
    loadOperationIcons(palette(), "copy_small", copyIcon, inverseCopyIcon);
    loadOperationIcons(palette(), "move_small", moveIcon, inverseMoveIcon);
}


void FileOperator::deleteFiles(const QFileInfoList &files) {
    if (checkMaxOpsNumber()) return;

    QString title, desc;
    if (files.size() == 1) {
        title = tr("Delete file");
        desc = tr("Are you sure you want to delete %1?")
            .arg(shortenPath(files[0].absoluteFilePath()));
    } else {
        title = tr("Delete files");
        desc = tr("You are about to delete %1 files. Are you sure you want to continue?").arg(files.size());
    }

    int confirm = QMessageBox::warning(
        0,
        title,
        desc,
        QMessageBox::Yes,
        QMessageBox::No
    );

    if(confirm == QMessageBox::Yes) {
        ProgressBar *bar = new ProgressBar(deleteIcon, inverseDeleteIcon);
        initOperation(new DeleteThread(files), bar);
    }
}


void FileOperator::copyFiles(const QFileInfoList &files, QDir &destination) {
    if (checkMaxOpsNumber()) return;

    QString title, desc;
    if (files.size() == 1) {
        title = tr("Copy file");
        desc = tr("Are you sure you want to copy %1 to %2?")
            .arg(shortenPath(files[0].absoluteFilePath()))
            .arg(shortenPath(destination.absolutePath()));
    } else {
        title = tr("Copy files");
        desc = tr("You are about to copy %1 files to %2. Are you sure you want to continue?")
            .arg(files.size()).arg(shortenPath(destination.absolutePath()));
    }

    int confirm = QMessageBox::warning(
        0,
        title,
        desc,
        QMessageBox::Yes,
        QMessageBox::No
    );

    if(confirm == QMessageBox::Yes) {
        ProgressBar *bar = new ProgressBar(copyIcon, inverseCopyIcon);
        bar->setBottomTexts(shortenPath(files[0].absolutePath()), shortenPath(destination.absolutePath()));
        initOperation(new CopyThread(files, destination), bar);
    }
}


void FileOperator::moveFiles(const QFileInfoList &files, QDir &destination) {
    if (checkMaxOpsNumber()) return;

    // for move we don't wanna move to the same dir
    if (files[0].absolutePath() == destination.absolutePath()) return;

    QString title, desc;
    if (files.size() == 1) {
        title = tr("Move file");
        desc = tr("Are you sure you want to move %1 to %2?")
            .arg(shortenPath(files[0].absoluteFilePath()))
            .arg(shortenPath(destination.absolutePath()));
    } else {
        title = tr("Move files");
        desc = tr("You are about to move %1 files to %2. Are you sure you want to continue?")
            .arg(files.size()).arg(shortenPath(destination.absolutePath()));
    }

    int confirm = QMessageBox::warning(
        0,
        title,
        desc,
        QMessageBox::Yes,
        QMessageBox::No
    );

    if(confirm == QMessageBox::Yes) {
        ProgressBar *bar = new ProgressBar(moveIcon, inverseMoveIcon);
        bar->setBottomTexts(shortenPath(files[0].absolutePath()), shortenPath(destination.absolutePath()));
        initOperation(new MoveThread(files, destination), bar);
    }
}


void FileOperator::showErrorPrompt(OperationThread* op,
    const QString &message,
    const QString &fileName,
    const int err)
{
    QMessageBox msgBox;
    QAbstractButton *cancelButton = msgBox.addButton(QMessageBox::Cancel);
    QAbstractButton *abortButton = msgBox.addButton(tr("Abort"), QMessageBox::DestructiveRole);
    QAbstractButton *retryButton = msgBox.addButton(QMessageBox::Retry);
    QAbstractButton *ignoreButton = msgBox.addButton(QMessageBox::Ignore);
    QAbstractButton *ignoreAllButton = msgBox.addButton(tr("Ignore All"), QMessageBox::AcceptRole);
    msgBox.setText(message.arg(shortenPath(fileName)));

    msgBox.exec();

    if (msgBox.clickedButton() == cancelButton) {
        op->pause = true;
        op->setResponse(OperationThread::RETRY);
    } else if (msgBox.clickedButton() == abortButton) {
        op->setResponse(OperationThread::ABORT);
    } else if (msgBox.clickedButton() == retryButton) {
        op->setResponse(OperationThread::RETRY);
    } else if (msgBox.clickedButton() == ignoreButton) {
        op->setResponse(OperationThread::IGNORE);
    } else if (msgBox.clickedButton() == ignoreAllButton) {
        op->setResponse(OperationThread::IGNORE, true, err);
    }
}


void FileOperator::showOverwritePrompt(
    OperationThread* op,
    const QString &fileName,
    const bool dirOverDir)
{
    Dialog msgBox;
    QAbstractButton *yesButton = msgBox.addButtonFirst(QDialogButtonBox::Yes);
    QAbstractButton *yesToAllButton = msgBox.addButtonFirst(QDialogButtonBox::YesToAll);
    QAbstractButton *noButton = msgBox.addButtonSecond(QDialogButtonBox::No);
    QAbstractButton *noToAllButton = msgBox.addButtonSecond(QDialogButtonBox::NoToAll);
    QAbstractButton *abortButton = msgBox.addButtonSecond(tr("Abort"), QDialogButtonBox::DestructiveRole);
    QAbstractButton *newNameButton = msgBox.addButtonFirst(tr("New Name"), QDialogButtonBox::AcceptRole);
    QAbstractButton *askButton = 0;
    QAbstractButton *skipDirButton = 0;

    if (dirOverDir) {
        msgBox.setText(tr("Directory %1 already exists. Overwrite the files inside?")
            .arg(shortenPath(fileName)));
        askButton = msgBox.addButtonFirst(tr("Ask"), QDialogButtonBox::AcceptRole);
        skipDirButton = msgBox.addButtonSecond(tr("Skip"), QDialogButtonBox::NoRole);
    } else {
        msgBox.setText(tr("File %1 already exists. Overwrite?").arg(shortenPath(fileName)));
    }

    msgBox.exec();

    if (msgBox.clickedButton == 0) {
        op->pause = true;
        op->setResponse(OperationThread::NONE);
    } else if (msgBox.clickedButton == abortButton) {
        op->setResponse(OperationThread::ABORT);
    } else if (msgBox.clickedButton == yesButton) {
        op->setResponse(OperationThread::OVERWRITE);
    } else if (msgBox.clickedButton == yesToAllButton) {
        op->setResponse(OperationThread::OVERWRITE, true);
    } else if (msgBox.clickedButton == noButton) {
        op->setResponse(OperationThread::KEEP);
    } else if (msgBox.clickedButton == noToAllButton) {
        op->setResponse(OperationThread::KEEP, true);
    } else if (msgBox.clickedButton == askButton) {
        op->setResponse(OperationThread::ASK);
    } else if (msgBox.clickedButton == newNameButton) {
        op->setResponse(OperationThread::NONE);
    } else if (msgBox.clickedButton == skipDirButton) {
        op->setResponse(OperationThread::SKIP_DIR);
    }
}


void FileOperator::showInputFilenamePrompt(OperationThread* op,
    const QFileInfo &file,
    const bool dir)
{
    bool ok;
    QString prompt, error;

    if (dir) {
        prompt = tr("Enter the new directory name.");
    } else {
        prompt = tr("Enter the new file name.");
    }

    op->newNameFromDialog = "";
    QString text = file.fileName();

    while (true) {
        text = QInputDialog::getText(this, QString(), prompt + error, QLineEdit::Normal, text, &ok);

        if (!ok) break;

        error = "";
        if (text.contains(QRegExp("[\"*/:<>?\\\\|]"))) {
            error = "<small><br/><font color = 'red'>" +
                tr("The name cannot contain any of the following characters: ") +
                "\"*/:&lt;&gt;?\\|</font></small>";
        } else if (ok && !text.isEmpty()) {
            QFileInfo info(file.path() + "/" + text);
            op->newNameFromDialog = info.absoluteFilePath();
            break;
        }
    }

    op->wake();
}


void FileOperator::remove(OperationThread* op) {
    op->wait();
    ProgressBar *bar = get(op);
    removeBarFromLayout(bar);
    opList.removeAll(qMakePair(op, bar));
    delete op;
    delete bar;
}


void FileOperator::togglePauseOperation(ProgressBar* bar) {
    OperationThread *op = get(bar);

    if (op->pause) {
        op->wake();
    } else {
        op->pause = true;
    }
}


void FileOperator::abortOperation(ProgressBar* bar) {
    OperationThread *op = get(bar);

    int confirm = QMessageBox::warning(
        0,
        tr("Abort operation"),
        tr("Are you sure you want to abort the operation?"),
        QMessageBox::Yes,
        QMessageBox::No
    );

    if(confirm == QMessageBox::Yes) {
        op->abort = true;
        op->pause = false;
        op->wake();
    }
}


void FileOperator::initOperation(OperationThread *thread, ProgressBar *bar) {
    addBarToLayout(bar);
    opList.append(qMakePair(thread, bar));

    connect(thread, SIGNAL(showErrorPrompt(OperationThread*, const QString&, const QString&, const int)),
        this, SLOT(showErrorPrompt(OperationThread*, const QString&, const QString&, const int)));
    connect(thread, SIGNAL(showOverwritePrompt(OperationThread*, const QString&, bool)),
        this, SLOT(showOverwritePrompt(OperationThread*, const QString&, bool)));
    connect(thread, SIGNAL(showInputFilenamePrompt(OperationThread*, const QFileInfo&, bool)),
        this, SLOT(showInputFilenamePrompt(OperationThread*, const QFileInfo&, bool)));
    connect(thread, SIGNAL(finished(OperationThread*)),
        this, SLOT(remove(OperationThread*)));

    connect(thread, SIGNAL(totalSizeChanged(int)), bar, SLOT(setMaximum(int)));
    connect(thread, SIGNAL(progressUpdate(int)), bar, SLOT(updateProgress(int)));
    connect(thread, SIGNAL(fileNameUpdated(QString)), bar, SLOT(updateMainText(QString)));
    connect(thread, SIGNAL(operationStarted(time_t)), bar, SLOT(setStartTime(time_t)));
    connect(thread, SIGNAL(operationPaused()), bar, SLOT(pause()));
    connect(thread, SIGNAL(operationResumed(time_t)), bar, SLOT(resume(time_t)));
    connect(thread, SIGNAL(removeAfterCopy()), bar, SLOT(showRemoveNotice()));

    connect(bar, SIGNAL(togglePauseOperation(ProgressBar*)), this, SLOT(togglePauseOperation(ProgressBar*)));
    connect(bar, SIGNAL(abortOperation(ProgressBar*)), this, SLOT(abortOperation(ProgressBar*)));

    thread->start(QThread::LowestPriority);
}


ProgressBar *FileOperator::get(OperationThread *op) const {
    for (OperationList::const_iterator it = opList.begin(); it != opList.end(); ++it) {
        if (it->first == op) return it->second;
    }
    return 0;
}


OperationThread *FileOperator::get(ProgressBar *bar) const {
    for (OperationList::const_iterator it = opList.begin(); it != opList.end(); ++it) {
        if (it->second == bar) return it->first;
    }
    return 0;
}


void FileOperator::addBarToLayout(ProgressBar *bar) {
    switch (opList.size()) {
    case 0:
    case 1:
    case 2:
        topRow->addWidget(bar);
        break;
    case 4:
        topRow->addItem(bottomRow->takeAt(0));
        bottomRow->addWidget(bar);
        break;
    case 3:
        bottomRow->addItem(topRow->takeAt(2));
    default:
        bottomRow->addWidget(bar);
    }
}


void FileOperator::removeBarFromLayout(ProgressBar *bar) {
    int index = topRow->indexOf(bar);
    if (index != -1) {
        topRow->takeAt(index);
        switch (opList.size()) {
        case 4:
            topRow->addItem(bottomRow->takeAt(0));
        case 6:
            topRow->addItem(bottomRow->takeAt(0));
            break;
        }
    } else {
        bottomRow->removeWidget(bar);
        switch (opList.size()) {
        case 4:
            topRow->addItem(bottomRow->takeAt(0));
            break;
        case 5:
            bottomRow->insertWidget(0, topRow->takeAt(2)->widget());
        }
    }
}


bool FileOperator::checkMaxOpsNumber() {
    if (opList.size() == 6) {
        QMaemo5InformationBox::information(this, tr("The maximum number of file operations is %1.").arg(6));
        return true;
    }
    return false;
}
