// case - file manager for N900
// Copyright (C) 2010 Lukas Hrazky <lukkash@email.cz>
// 
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.


#include "case.h"

#include <QHBoxLayout>
#include <QVBoxLayout>


Case::Case(QWidget *parent) :
    QMainWindow(parent),
    leftPane(new Pane(this)),
    rightPane(new Pane(this)),
    activePane(leftPane),
    inactivePane(rightPane),
    cloneBtn(new Button("clone", 0, 60, 70)),
    swapBtn(new Button("swap", 0, 60, 70)),
    copyBtn(new Button("copy", 0, 60, 70)),
    moveBtn(new Button("move", 0, 60, 70)),
    delBtn(new Button("delete", 0, 60, 70)),
    fileOperator(new FileOperator(this))
{
    QVBoxLayout *layout = new QVBoxLayout;
    layout->setContentsMargins(0, 0, 0, 0);
    layout->setSpacing(0);

    QHBoxLayout *paneLayout = new QHBoxLayout;
    paneLayout->setContentsMargins(0, 0, 0, 0);
    paneLayout->setSpacing(1);

    QWidget *central = new QWidget;
    setCentralWidget(central);
    central->setLayout(layout);
    layout->addLayout(paneLayout);

    paneLayout->addWidget(leftPane);
    leftPane->toggleActive();

    QVBoxLayout *middleButtons = new QVBoxLayout;
    paneLayout->addLayout(middleButtons);
    middleButtons->setSpacing(10);
    middleButtons->setContentsMargins(0, 0, 0, 0);

    cloneBtn->setContentsMargins(0, 0, 0, 0);

    middleButtons->addWidget(cloneBtn);
    middleButtons->addWidget(swapBtn);
    middleButtons->addWidget(copyBtn);
    middleButtons->addWidget(moveBtn);
    middleButtons->addWidget(delBtn);

    paneLayout->addWidget(rightPane);

    layout->addWidget(fileOperator);

    connect(this, SIGNAL(activePaneSwitched()), leftPane, SLOT(toggleActive()));
    connect(this, SIGNAL(activePaneSwitched()), rightPane, SLOT(toggleActive()));

    connect(cloneBtn, SIGNAL(pressed()), this, SLOT(clonePane()));
    connect(swapBtn, SIGNAL(pressed()), this, SLOT(swapPanes()));
    connect(copyBtn, SIGNAL(pressed()), this, SLOT(copyFiles()));
    connect(moveBtn, SIGNAL(pressed()), this, SLOT(moveFiles()));
    connect(delBtn, SIGNAL(pressed()), this, SLOT(deleteFiles()));
}


void Case::switchActivePane() {
    Pane *tmpPane = activePane;
    activePane = inactivePane;
    inactivePane = tmpPane;

    cloneBtn->swapIcon();
    swapBtn->swapIcon();
    copyBtn->swapIcon();
    moveBtn->swapIcon();
    delBtn->swapIcon();

    emit activePaneSwitched();
}


void Case::keyPressEvent(QKeyEvent *e) {
    if(e->key() == Qt::Key_H && e->modifiers() == Qt::ControlModifier) {
        activePane->toggleShowHiddenFiles();
    } else {
        QMainWindow::keyPressEvent(e);
    }
}


void Case::clonePane() {
    inactivePane->changePath(activePane->path());
}


void Case::swapPanes() {
    QString tmpPath = activePane->path();
    activePane->changePath(inactivePane->path());
    inactivePane->changePath(tmpPath);
}


void Case::copyFiles() {
    if (activePane->selection().size()) {
        QDir dest = inactivePane->path();
        fileOperator->copyFiles(activePane->selection(), dest);
    }
}


void Case::moveFiles() {
    if (activePane->selection().size()) {
        QDir dest = inactivePane->path();
        fileOperator->moveFiles(activePane->selection(), dest);
    }
}


void Case::deleteFiles() {
    if (activePane->selection().size()) {
        fileOperator->deleteFiles(activePane->selection());
    }
}
