#
# This file is part of Canola
# Copyright (C) 2007-2009 Instituto Nokia de Tecnologia
# Contact: Renato Chencarek <renato.chencarek@openbossa.org>
#          Eduardo Lima (Etrunko) <eduardo.lima@openbossa.org>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# Additional permission under GNU GPL version 3 section 7
#
# The copyright holders grant you an additional permission under Section 7
# of the GNU General Public License, version 3, exempting you from the
# requirement in Section 6 of the GNU General Public License, version 3, to
# accompany Corresponding Source with Installation Information for the
# Program or any work based on the Program. You are still required to comply
# with all other Section 6 requirements to provide Corresponding Source.
#

import evas

from terra.core.manager import Manager
from terra.core.terra_object import TerraObject
from terra.ui.base import EdjeWidget, Widget, PluginEdjeWidget
from terra.ui.kinetic import KineticMouse

mger = Manager()
KnobWidget = mger.get_class("Widget/Knob")
ScrollbarWidget = mger.get_class("Widget/Scrollbar")


class ScrollableTextBlock(TerraObject, Widget, evas.ClippedSmartObject):
    """Scrollable text view using kinetics.

    Events will not be handled to row renderers, they will be handled by
    event_area, an object that will be laid out on top of every renderer.

    Behavior is based on B{click_constant}: maximum vertical motion to
    consider as click, values greater that (in absolute terms) are considered
    drag.
    """
    terra_type = "Widget/ScrollableTextBlock"
    click_constant = 20
    move_constant = 10
    plugin = None
    group = "description"

    def __init__(self, canvas, parent, theme=None):
        Widget.__init__(self, canvas, parent)
        evas.ClippedSmartObject.__init__(self, canvas)

        self._bg = EdjeWidget(canvas, "scrolled_window", parent, theme)
        self.member_add(self._bg)
        self._bg.show()
        self._bg.signal_emit("scrollbar,transition,in", "")

        self.view = self._bg.part_object_get("contents")

        knob = KnobWidget(parent, group="panel_list_scrollbar")
        self._scroll = ScrollbarWidget(self, knob)
        self._bg.part_swallow("scrollbar", self._scroll)

        if self.plugin is not None:
            self.contents = PluginEdjeWidget(canvas, self.group, parent, theme,
                                             self.plugin)
        else:
            self.contents = EdjeWidget(canvas, self.group, parent, theme)

        self.member_add(self.contents)
        self._bg.part_swallow("contents", self.contents)

        self.event_area = self.Rectangle(color=(0, 0, 0, 0))
        self.event_area.show()
        self.event_area.on_mouse_down_add(self._cb_on_mouse_down)
        self.event_area.on_mouse_up_add(self._cb_on_mouse_up)
        self.event_area.on_mouse_move_add(self._cb_on_mouse_move)
        self.kinetic = KineticMouse(self.move_offset)
        self.is_drag = False
        self.mouse_down_pos = None

    def text_set(self, value):
        self.contents.part_text_set("text", value)

        # Reswallow so that edje knows the size
        self._bg.part_unswallow(self.contents)
        self._bg.part_swallow("contents", self.contents)

        self.stored_size = self.contents.size_min_calc()
        self.contents.size = self.stored_size

        self._refresh_scrollbar()

    def emit_clicked(self):
        pass

    def move_offset(self, offset):
        # XXX: Why is this needed? bug?
        self.contents.size = self.stored_size

        if self.contents.size <= self.view.size:
            return

        if self.contents.pos[1] + offset > self.view.pos[1]:
            self.contents.pos = self.view.pos
        elif self.contents.pos[1] + self.contents.size[1] + offset \
            < self.view.pos[1] + self.view.size[1]:
            end_pos = (self.contents.pos[0], \
                       self.view.pos[1] + self.view.size[1] \
                       - self.contents.size[1])
            self.contents.pos = end_pos
        else:
            self.contents.move_relative(0, offset)

        pos = self.view.pos[1] - self.contents.pos[1]
        percent = pos / float(self.contents.size[1] - self.view.size[1])
        self._scroll.value_set(percent)

    def resize(self, w, h):
        self.event_area.resize(w, h)
        self._bg.resize(w, h)
        self._refresh_scrollbar()

    def _refresh_scrollbar(self):
        if not hasattr(self, "stored_size"):
            return

        self.contents.size = self.stored_size

        if self.contents.size <= self.view.size:
            self._bg.signal_emit("scrollbar,transition,out", "")
        else:
            self._scroll.scale_set(self.view.size[1] / float(self.contents.size[1]))
            self._bg.signal_emit("scrollbar,transition,in", "")

    def _cb_on_mouse_down(self, obj, event):
        if event.button == 1:
            y = event.position.canvas.y
            self.mouse_down_pos = y
            self.is_drag = not self.kinetic.mouse_down(y)

    def _is_click_possible(self, y):
        if self.is_drag or self.mouse_down_pos is None:
            return False
        else:
            return abs(y - self.mouse_down_pos) <= self.click_constant

    def _cb_on_mouse_up(self, obj, event):
        if event.button == 1:
            y = event.position.canvas.y
            if self._is_click_possible(y):
                self.emit_clicked()
                self.kinetic.mouse_cancel()
            else:
                self.kinetic.mouse_up(y)

    def _cb_on_mouse_move(self, obj, event):
        if event.buttons == 1:
            y = event.position.canvas.y
            if not self._is_click_possible(y):
                self.is_drag = True
            if self.mouse_down_pos is None:
                return
            if abs(self.mouse_down_pos - y) > self.move_constant:
                self.kinetic.mouse_move(y)

    @evas.decorators.del_callback
    def _destroy_contents(self):
        self._bg.delete()
        self.contents.delete()
