#
# This file is part of Canola
# Copyright (C) 2007-2009 Instituto Nokia de Tecnologia
# Contact: Renato Chencarek <renato.chencarek@openbossa.org>
#          Eduardo Lima (Etrunko) <eduardo.lima@openbossa.org>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# Additional permission under GNU GPL version 3 section 7
#
# The copyright holders grant you an additional permission under Section 7
# of the GNU General Public License, version 3, exempting you from the
# requirement in Section 6 of the GNU General Public License, version 3, to
# accompany Corresponding Source with Installation Information for the
# Program or any work based on the Program. You are still required to comply
# with all other Section 6 requirements to provide Corresponding Source.
#

from terra.core.controller import Controller, get_controller_for_model
from terra.core.model import Model
from terra.core.terra_object import TerraObject
from terra.ui.screen import Screen

from list import ListPanelController
from base import ModalController, ModalMessageController


import logging
log = logging.getLogger("plugins.canola-core.settings")


class MainPanelController(Controller):
    def __init__(self, model, canvas, parent, end_callback=None,
                 leave_callback=None):
        Controller.__init__(self, model, canvas, parent)
        self.window = parent.view
        self.view = None
        self.animating = False
        self.panels = []
        self.modal = None
        self.callback_leave = leave_callback

    def _last_panel_get(self):
        if self.panels:
            return self.panels[-1]
        else:
            return None

    last_panel = property(_last_panel_get)

    def cb_on_animation_ended(self, *ignored):
        self.animating = False

    def back_area_clicked(self):
        if self.animating:
            return

        # dispatch to subcontroller, so they can do actions before back()
        p = self.panels[-1]
        p.back()

    def back(self, end_callback=None):
        if self.animating:
            return

        # XXX: focus management, misplaced UI requests
        if self.modal:
            if end_callback:
                end_callback()
            self.modal.delete()
            self.modal = None
            self.panels[-1].view.focus = True
            return

        p = self.panels.pop()
        def end(*ignored):
            self.animating = False
            p.delete()
            if not self.panels:
                self.parent.back_options(self.cb_on_animation_ended)
                if self.callback_leave:
                    self.callback_leave()
            if end_callback:
                end_callback()
        self.animating = True
        self.parent.panel_unstack(end)

    def message(self, message, leave_callback=None):
        mmc = ModalMessageController(None, self.evas, self, message)
        mmc.callback_leave = leave_callback
        self.use(mmc)

    def use(self, obj, end_callback=None):
        if self.animating:
            return

        if isinstance(obj, Model):
            controller = get_controller_for_model(obj, self.evas, self)
        elif isinstance(obj, Controller):
            controller = obj
        else:
            log.error("unknown object type %r (expected Model "
                      "or Controller)", obj.__class__.__name__)
            return

        if isinstance(controller, ModalController):
            # XXX
            if self.modal:
                log.debug_error("we can't have two settings modals")
                return
            self.modal = controller
            if end_callback:
                end_callback(self, controller.model)
            self.modal.view.focus = True
            return

        if isinstance(controller, ActionController):
            if end_callback:
                end_callback(self, controller.model)
            return

        # XXX: the default should be stack, really? shouldn't the controller
        # ask to be stacked if it wants?
        self.panels.append(controller)
        def cb(view, panel):
            self.animating = False
            if end_callback:
                end_callback(self, controller.model)

        self.animating = True
        self.parent.panel_stack(self.panels[-1].view, cb)

    def killall(self):
        self.parent.killall()

    def delete(self):
        pass


class SettingsController(MainPanelController):
    terra_type = "Controller/Folder/Settings"

    def __init__(self, model, canvas, parent, end_callback=None,
                 leave_callback=None):
        MainPanelController.__init__(self, model, canvas, parent,
                                     end_callback)
        self.use(ListPanelController(model, canvas, self), end_callback)


class OptionsController(MainPanelController):
    terra_type = "OptionsController"

    def __init__(self, model, canvas, parent, screen_controller, end_callback=None,
                 leave_callback=None):
        MainPanelController.__init__(self, model, canvas, parent,
                                     end_callback, leave_callback)
        self.screen_controller = screen_controller
        model.screen_controller = self.screen_controller
        self.use(model, end_callback)


class ActionController(Controller):
    terra_type = "Controller/Action"

    def __init__(self, model, canvas, parent):
        Controller.__init__(self, model, canvas, parent)
        if model.execute():
            parent.back()


class SettingsActionController(ActionController):
    terra_type = "Controller/Settings/Action"


class OptionsActionController(ActionController):
    terra_type = "Controller/Options/Action"


class OptionsControllerMixin(TerraObject):
    """Mixin to be used by a Controller of a Screen with Options button.

    Add the mixin to your parent list and be sure to call the
    OptionsControllerMixin implementations of __init__() and delete() in
    your own. The model used by OptionsController will be whatever
    the function options_model_get() returns, so override this method too.

    Override options() if you need a custom behaviour, like using
    leave_callback when calling OptionsController.
    """
    terra_type = "OptionsControllerMixin"

    def __init__(self):
        try:
            if self.view is None or not isinstance(self.view, Screen):
                raise TypeError("You must have a Screen typed view.")
        except AttributeError:
            raise TypeError("You must have a Screen typed view.")

        self.view.options = self.options
        self.view.has_options = self.has_options
        self.options_model = self.options_model_get()

    def has_options(self):
        if not self.options_model:
            return False
        if not self.options_model.is_loaded:
            self.options_model.load()
        return bool(self.options_model.children)

    def options(self, leave_callback=None):
        if not self.options_model:
            return

        def cb(controller, *ignored):
            def cb_finished(*ignored):
                self.view.parent.signal_emit("events,unblock", "")
            self.parent.use_options(controller, cb_finished)

        self.view.parent.signal_emit("events,block", "")
        oc = OptionsController(self.options_model, self.evas,
                               self.parent, self, end_callback=cb,
                               leave_callback=leave_callback)

    def options_model_get(self):
        return None

    def delete(self):
        if self.options_model and self.options_model.is_loaded:
            self.options_model.unload()
            self.options_model = None
