#
# This file is part of Canola
# Copyright (C) 2007-2009 Instituto Nokia de Tecnologia
# Contact: Renato Chencarek <renato.chencarek@openbossa.org>
#          Eduardo Lima (Etrunko) <eduardo.lima@openbossa.org>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# Additional permission under GNU GPL version 3 section 7
#
# The copyright holders grant you an additional permission under Section 7
# of the GNU General Public License, version 3, exempting you from the
# requirement in Section 6 of the GNU General Public License, version 3, to
# accompany Corresponding Source with Installation Information for the
# Program or any work based on the Program. You are still required to comply
# with all other Section 6 requirements to provide Corresponding Source.
#

from terra.ui.panel import PanelRichList
from terra.core.model import ModelFolder
from terra.core.manager import Manager
from terra.core.controller import get_controller_for_model

mger = Manager()
OnOffItemRenderer = mger.get_class("Renderer/EtkList/OnOffItem")
FeedItemRenderer = mger.get_class("Renderer/EtkList/FeedItem")
PanelController = mger.get_class("Controller/Panel")
MultipleItemRenderer = mger.get_class("Renderer/EtkList/MultipleItem")


class MixedListItemOnOff(ModelFolder):
    terra_type = "Model/Settings/Folder/MixedList/Item/OnOff"
    title = ""

    def __init__(self, parent=None):
        ModelFolder.__init__(self, self.title, parent)

        self.callback_use = None
        self.callback_update = None
        self.callback_killall = None

        self.__create_renderer()

    def __create_renderer(self):
        def _get_state(row):
            return row.get_state()

        def _on_clicked(row, list):
            row.on_clicked()

        self.renderer = OnOffItemRenderer(ui_func=_get_state,
                                          item_click=_on_clicked)

    def get_state(self):
        raise NotImplementedError("must be implemented by subclasses")

    def on_clicked(self):
        raise NotImplementedError("must be implemented by subclasses")

    def do_load(self):
        pass


class MixedListItemDual(ModelFolder):
    terra_type = "Model/Settings/Folder/MixedList/Item/Dual"
    title = ""

    def __init__(self, parent=None):
        ModelFolder.__init__(self, self.title, parent)
        self.left_button_text = "Edit"
        self.right_button_text = "Delete"

        self.callback_use = None
        self.callback_update = None
        self.callback_killall = None

        self.__create_renderer()

    def __create_renderer(self):
        def _on_item_clicked(row, list):
            row.on_clicked()

        def _on_edit_clicked(row, list):
            row.on_left_button_clicked()

        def _on_delete_clicked(row, list):
            row.on_right_button_clicked()

        def _get_title_text(row):
            return row.get_title()

        def _get_left_button_text(row):
            return row.get_left_button_text()

        def _get_right_button_text(row):
            return row.get_right_button_text()

        self.renderer = FeedItemRenderer(text_func=_get_title_text,
                                         item_click=_on_item_clicked,
                                         edit_click=_on_edit_clicked,
                                         delete_click=_on_delete_clicked,
                                         bt_left_func=_get_left_button_text,
                                         bt_right_func=_get_right_button_text)

    def get_title(self):
        raise NotImplementedError("must be implemented by subclasses")

    def get_left_button_text(self):
        raise NotImplementedError("must be implemented by subclasses")

    def get_right_button_text(self):
        raise NotImplementedError("must be implemented by subclasses")

    def on_clicked(self):
        raise NotImplementedError("must be implemented by subclasses")

    def on_left_button_clicked(self):
        raise NotImplementedError("must be implemented by subclasses")

    def on_right_button_clicked(self):
        raise NotImplementedError("must be implemented by subclasses")

    def do_load(self):
        pass


class MixedListPanel(PanelRichList):
    def __init__(self, main_window, title, elements, theme=None):
        PanelRichList.__init__(self, main_window, title, None, theme)

        self.list.animated_changes = True

        self.callback_text_activated = None

        self.list.freeze()
        for e in elements:
            self.list_model.append(e)
        self.list.thaw()

    def list_columns_get(self):
        return [(100, MultipleItemRenderer(), True)]

    def _on_text_activated(self, e, *ignored):
        if self.callback_text_activated:
            self.callback_text_activated(e.text)


class MixedList(PanelController):
    terra_type = "Controller/Settings/Folder/MixedList"

    def __init__(self, model, canvas, parent):
        PanelController.__init__(self, model, canvas, parent)

        self.model = model
        self.model.load()

        for c in self.model.children:
            c.callback_use = self._cb_use
            c.callback_update = self._cb_update_view
            c.callback_killall = self.killall

        self.view = MixedListPanel(parent.window, model.title,
                                   self.model.children)
        self.view.callback_escape = self.back

    def _cb_use(self, model):
        controller = get_controller_for_model(model, self.evas, self)
        self.use(controller)

    def _cb_update_view(self, model):
        self.view.list.redraw_queue()

    def killall(self):
        self.parent.killall()

    def delete(self):
        self.view.delete()
        self.view = None
        self.model.unload()
