#
# This file is part of Canola
# Copyright (C) 2007-2009 Instituto Nokia de Tecnologia
# Contact: Renato Chencarek <renato.chencarek@openbossa.org>
#          Eduardo Lima (Etrunko) <eduardo.lima@openbossa.org>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# Additional permission under GNU GPL version 3 section 7
#
# The copyright holders grant you an additional permission under Section 7
# of the GNU General Public License, version 3, exempting you from the
# requirement in Section 6 of the GNU General Public License, version 3, to
# accompany Corresponding Source with Installation Information for the
# Program or any work based on the Program. You are still required to comply
# with all other Section 6 requirements to provide Corresponding Source.
#

import etk
import ecore
import logging
import evas.decorators
import lightmediascanner as lms

from terra.core.controller import Controller, get_controller_for_model
from terra.core.manager import Manager
from terra.ui.modal import ModalSlider, SliderContentList, Modal
from terra.ui.throbber import EtkThrobber
from terra.ui.modal import ModalMessage, ModalMessageConfirm

from renderers import DirItemRenderer

log = logging.getLogger("plugins.canola-core.settings.media_library")

mger = Manager()
SettingsFolderController = mger.get_class("Controller/Settings/Folder")
ModalController = mger.get_class("Controller/Modal")

RESCAN_TIMEOUT = 120.0
STATUS_INTERVAL = 0.2


class SettingsMediaLibraryView(ModalSlider):
    def __init__(self, parent, title):
        ModalSlider.__init__(self, parent.last_panel.view, title,
                             parent.last_panel.view.theme, hborder=16,
                             vborder=20)

        self.button_box_show()
        self.ok_button = self.button_add("OK")
        self.back_button = self.button_add("  Back  ", show=False)

        self.callback_ok_clicked = None
        self.callback_back_clicked = None

        self._init_message()

    def _init_message(self):
        self.embed = etk.Embed(self.evas)
        self.throbber = EtkThrobber("  Updating media library...")
        self.throbber.show()
        self.throbber_align = \
            etk.Alignment(0.5, 0.4, 0.0, 0.0, child=self.throbber)
        self.throbber_align.show()
        self.embed.add(self.throbber_align)

    def message(self, hide_cb=None):
        self.header_area_hide()
        self.button_box_hide()
        self.title = ""
        self.throbber.callback_animate_hide_finished = hide_cb
        self.throbber.start()
        self.contents_set(self.embed.object)
        self.slider_contents.hide()
        self.embed.show()
        self.half_expand()

    def _on_button_clicked(self, bt):
        if bt == self.ok_button:
            if self.callback_ok_clicked:
                self.callback_ok_clicked()
        elif bt == self.back_button:
            if self.callback_back_clicked:
                self.callback_back_clicked()

    @evas.decorators.del_callback
    def _destroy_message(self):
        self.embed.destroy()
        self.embed = None


class SettingsMediaLibraryController(ModalController):
    terra_type = "Controller/Settings/MediaFolder"

    def __init__(self, model, canvas, parent):
        ModalController.__init__(self, model, canvas, parent)
        self.children = []

        controller = SettingsDirController(model, self.evas, self)
        self.children.append(controller)

        self.view = SettingsMediaLibraryView(parent, model.title)
        self.view.center_contents_set(controller.view.object)
        self.view.callback_ok_clicked = self.close
        self.view.callback_back_clicked = self.back

        self.model.callback_rescan = self._cb_rescan
        self._updating_timer = None
        self._will_rescan = False

        self.view.show()

    def escape(self):
        self.back()

    def really_close(self):
        def cb():
            self.parent.back()
        self.view.hide(cb)

    def close(self):
        self.delete_children()
        if self._will_rescan:
            self._updating_timer = ecore.timer_add(RESCAN_TIMEOUT,
                                                   self._cb_unlocked)
            self._status_waiting = False
            self._status_timer = ecore.timer_add(STATUS_INTERVAL, self._status)

            def hide_cb():
                self.really_close()
            self.view.message(hide_cb)
            self._will_rescan = False
        else:
            self.really_close()

    def _status(self):
        if self._status_waiting:
            return True
        self._status_waiting = True
        mger.canola_daemon.scan_status(self._status_reply)
        return True

    def _status_reply(self, counts):
        self._status_waiting = False
        n_process = counts[lms.LMS_PROGRESS_STATUS_PROCESSED]
        if (n_process > 0):
            self.view.throbber.text_set("  %d files processed..." % n_process)

    def _cb_rescan(self, scanned_folders, media_type):
        self._will_rescan = True
        self.parent.killall()

        daemon = mger.canola_daemon
        daemon.callbacks_locked.insert(0, lambda: False)
        daemon.callbacks_unlocked.insert(0, self._cb_unlocked)
        daemon.rescan_by_type(scanned_folders, media_type)

    def _cb_unlocked(self):
        if self._status_timer is not None:
            self._status_timer.stop()
            self._status_timer = None
        if self._updating_timer is None:
            return False
        self._updating_timer.stop()
        self._updating_timer = None
        self.view.throbber.stop()

    def _load_dir(self):
        controller = self.children[-1]
        controller.load_model()

    def delete_children(self):
        while self.children:
            controller = self.children.pop()
            controller.delete()

    def delete(self):
        self.model.callback_rescan = None
        self.view.delete()
        self.view = None

    def back(self):
        if len(self.children) < 2:
            self.close()
            return

        controller = self.children.pop()
        left_view = self.children[-1].view
        title = self.children[-1].model.title

        self.view.left_contents_set(left_view.object)
        def cb():
            controller.delete()
        self.view.move_right(cb)
        self.view.title_set(title)

        if len(self.children) == 1:
            self.view.back_button.hide()

    def use(self, model, end_callback=None):
        controller = get_controller_for_model(model, self.evas, self)
        self.children.append(controller)
        self.view.right_contents_set(controller.view.object)
        self.view.move_left(end_callback)
        self.view.title_set(model.title)
        self.view.back_button.show()


class SettingsDirContent(SliderContentList):
    def __init__(self, evas_obj, item_clicked=None, toggle_clicked=None):
        SliderContentList.__init__(self, evas_obj)
        self.callback_item_clicked = item_clicked
        self.callback_toggle_clicked = toggle_clicked

    def columns_get(self):
        dir_item_renderer = \
            DirItemRenderer(ui_data_func=self._get_ui_data,
                            item_click=self._on_item_clicked,
                            toggle_click=self._on_toggle_clicked)
        return [(100, dir_item_renderer, True)]

    def _get_ui_data(self, row):
        return (row.has_subdirs(), row.name, row.scan_state)

    def _on_item_clicked(self, row, list):
        if self.callback_item_clicked:
            self.callback_item_clicked(row)

    def _on_toggle_clicked(self, row, list):
        if self.callback_toggle_clicked:
            self.callback_toggle_clicked(row)


class SettingsDirController(Controller):
    terra_type = "Controller/Settings/Dir"

    def __init__(self, model, canvas, parent):
        Controller.__init__(self, model, canvas, parent)

        self.view = SettingsDirContent(self.evas,
                                       self._on_item_clicked,
                                       self._on_toggle_clicked)
        self.view.callback_escape = self.parent.escape
        self.animating = False

        self.view.show()
        self.model.changed_callback_add(self._add_item)
        self.load_model()

    def load_model(self):
        ecore.idler_add(self.model.load)

    def _add_item(self, model):
        self.view.item_append(model.children[-1])

    def _on_item_clicked(self, model):
        if self.animating or not self.can_use(model):
            return

        self.animating = True
        def cb():
            self.animating = False
        self.use(model, end_callback=cb)

    def _on_toggle_clicked(self, model):
        if self.animating:
            return

        if model.scan_state == model.STATE_NONE:
            model.scan_state = model.STATE_ALL
        else:
            model.scan_state = model.STATE_NONE

    def delete(self):
        self.model.changed_callback_del(self._add_item)
        self.view.destroy()
        self.view = None
        self.model.update_scanned()
        self.model.unload()

    def can_use(self, model):
        return model.has_subdirs() and model.scan_state != model.STATE_ALL

    def use(self, model, end_callback=None):
        self.parent.use(model, end_callback)


class SettingsRootMediaLibraryController(SettingsFolderController):
    terra_type = "Controller/Settings/Folder/Root/MediaLibrary"


class MediaLibraryForceRescanView(Modal):
    def __init__(self, parent, title):
        Modal.__init__(self, parent.last_panel.view, title,
                       parent.last_panel.view.theme, hborder=16,
                       vborder=20)

        self.title = ""
        self.embed = etk.Embed(self.evas)
        self.throbber = EtkThrobber("  Updating media library...")
        self.throbber.show()
        self.throbber_align = \
            etk.Alignment(0.5, 0.4, 0.0, 0.0, child=self.throbber)
        self.throbber_align.show()
        self.embed.add(self.throbber_align)
        self.embed.show()
        self.contents = self.embed.object

    def message(self, hide_cb=None):
        self.throbber.callback_animate_hide_finished = hide_cb
        self.throbber.start()
        self.show()
        self.half_expand()

    @evas.decorators.del_callback
    def _destroy_message(self):
        self.embed.destroy()
        self.embed = None


class MediaLibraryForceRescanController(ModalController):
    terra_type = "Controller/Settings/MediaFolder/ForceRescan"

    def __init__(self, model, canvas, parent):
        ModalController.__init__(self, model, canvas, parent)
        model.callback_locked = self.start
        model.callback_unlocked = self.stop
        model.callback_killall = self.parent.killall

        self.view = MediaLibraryForceRescanView(parent, model.name)
        self._updating_timer = ecore.timer_add(RESCAN_TIMEOUT, self.stop, True)
        self._status_waiting = False
        self._status_timer = ecore.timer_add(STATUS_INTERVAL, self._status)
        self.model.execute()

    def _status(self):
        if self._status_waiting:
            return True
        self._status_waiting = True
        mger.canola_daemon.scan_status(self._status_reply)
        return True

    def _status_reply(self, counts):
        self._status_waiting = False
        n_process = counts[lms.LMS_PROGRESS_STATUS_PROCESSED]
        if (n_process > 0):
            self.view.throbber.text_set("  %d files processed..." % n_process)

    def start(self):
        self.view.message(hide_cb=self.stop_finished)
        self.model.callback_locked = None

    def stop(self, timeout=False):
        if timeout:
            log.info("Rescan timeout, finishing before unlock signal.")

        if self._updating_timer is None:
            return False
        else:
            self._updating_timer.stop()
            self._updating_timer = None
            self._status_timer.stop()
            self._status_timer = None

        self.model.callback_unlocked = None
        self.view.throbber.stop()

    def stop_finished(self):
        def cb():
            self.parent.back()
        self.view.hide(cb)

    def delete(self):
        self.view.delete()
        self.view = None
        if self._updating_timer is not None:
            self._updating_timer.stop()
            self._updating_timer = None
        self.model.callback_locked = None
        self.model.callback_unlocked = None
        self.model.callback_killall = None
        self.model = None
        if self._status_timer is not None:
            self._status_timer.stop()
            self._status_timer = None

class MediaLibraryResetDBController(ModalController):
    terra_type = "Controller/Settings/MediaFolder/ResetDB"
    message_confirm = "<small>This action will erase your entire database. " \
                      "Proceed anyway?</small>"
    message = "<small>Database erased. You may now hit \"Refresh all\"</small>"

    def __init__(self, model, canvas, parent):
        ModalController.__init__(self, model, canvas, parent)

        self.callback_leave = None
        model.callback_ready = self._finish
        model.callback_killall = self.parent.killall

        self.view = ModalMessageConfirm(
            parent.last_panel.view, self.message_confirm, hborder=60,
            vborder=120)
        self.view.callback_escape = self.close
        self.view.callback_clicked_yes = self._start
        self.view.callback_clicked_no = self.close

        self.view.show()

    def close(self):
        def cb(*ignored):
            self.back(self.callback_leave)
        self.view.hide(end_callback=cb)

    def delete(self):
        self.view.delete()

    def _start(self):
        self.view.hide()
        self.model.execute()

    def _finish(self):
        self.view.delete()
        self.view = ModalMessage(
            self.parent.last_panel.view, self.message, hborder=60, vborder=120)
        self.view.callback_escape = self.close
        self.view.callback_clicked = self.close
        self.view.show()

