#
# This file is part of Canola
# Copyright (C) 2007-2009 Instituto Nokia de Tecnologia
# Contact: Renato Chencarek <renato.chencarek@openbossa.org>
#          Eduardo Lima (Etrunko) <eduardo.lima@openbossa.org>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# Additional permission under GNU GPL version 3 section 7
#
# The copyright holders grant you an additional permission under Section 7
# of the GNU General Public License, version 3, exempting you from the
# requirement in Section 6 of the GNU General Public License, version 3, to
# accompany Corresponding Source with Installation Information for the
# Program or any work based on the Program. You are still required to comply
# with all other Section 6 requirements to provide Corresponding Source.
#

import os.path

import ecore
from terra.core.model import Model, ModelFolder
from terra.core.manager import Manager
from terra.core.plugin_prefs import PluginPrefs

mger = Manager()
CheckListPanelController = mger.get_class("Controller/Settings/CheckedFolder")
CheckListPanel = mger.get_class("Widget/CheckListPanel")
CheckListRenderer = mger.get_class("Widget/CheckListRenderer")
PanelButtonWidget = mger.get_class("Widget/PanelButton")
SystemProps = mger.get_class("SystemProperties")

system_props = SystemProps()

class DownloadFolderModel(Model):
    def __init__(self, name, dir, parent=None):
        Model.__init__(self, name, parent)
        self.dir = dir


class DownloadFolderModelFolder(ModelFolder):
    terra_type = "Model/Settings/Folder/InternetMedia/DownloadFolder"
    title = "My download folder"

    def __init__(self, parent=None):
        ModelFolder.__init__(self, self.title, parent)

    def do_load(self):
        for dir, name in system_props.download_dirs_get():
            DownloadFolderModel(name, dir, self)

    def do_unload(self):
        self.current = None
        ModelFolder.do_unload(self)


class DownloadFolderItemRenderer(CheckListRenderer):
    def _is_selected(self, v):
        parent = v.parent
        return parent.current is not None and \
            parent.children[parent.current] is v


class DownloadFolderPanel(CheckListPanel):
    def __init__(self, main_window, title, elements, theme=None):
        header_text = \
            "Choose the folder where you want to store your downloads:"
        CheckListPanel.__init__(self, main_window, title, elements,
                                DownloadFolderItemRenderer,
                                theme=theme, header_text=header_text)
        self.callback_ok = None
        self._setup_buttons()

    def _setup_buttons(self):
        ok = PanelButtonWidget(self.evas, "OK",
                               clicked_callback=self._cb_ok_clicked,
                               parent=self)
        self._setup_button_box(right=ok)

    def _cb_ok_clicked(self, *ignored):
        if self.callback_ok is not None:
            self.callback_ok()


class DownloadFolderController(CheckListPanelController):
    terra_type = "Controller/Settings/Folder/InternetMedia/DownloadFolder"

    def __init__(self, model, canvas, parent):
        CheckListPanelController.__init__(self, model, canvas, parent)

        def mark_selected(*args, **kargs):
            settings = PluginPrefs("settings")
            dpath = settings.get('download_path', None)
            if dpath is None:
                return False

            dpath = os.path.realpath(dpath)
            for i, m in enumerate(self.model.children):
                if dpath == m.dir:
                    self.model.current = i
                    break

            if self.model.current is None:
                DownloadFolderModel(dpath, dpath, self.model)
                self.model.current = len(self.model.children) - 1

            self.view.redraw_item(self.model.current)
            return False

        ecore.idler_add(mark_selected)


    def _setup_view(self):
        title = self.model.name
        self.view = DownloadFolderPanel(self.parent.window, title,
                                        self.model.children)
        self.view.callback_clicked = self.cb_on_clicked
        self.view.callback_escape = self.back
        self.view.callback_ok = self.cb_on_ok

    def cb_on_clicked(self, view, index):
        old_index = self.model.current
        self.model.current = index

        if old_index is not None:
            self.view.redraw_item(old_index)
        self.view.redraw_item(index)

    def _save_selected(self):
        model = self.model.children[self.model.current]
        settings = PluginPrefs("settings")
        settings['download_path'] = model.dir
        settings.save()

    def cb_on_ok(self):
        self._save_selected()
        self.back()
