#
# This file is part of Canola
# Copyright (C) 2007-2009 Instituto Nokia de Tecnologia
# Contact: Renato Chencarek <renato.chencarek@openbossa.org>
#          Eduardo Lima (Etrunko) <eduardo.lima@openbossa.org>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# Additional permission under GNU GPL version 3 section 7
#
# The copyright holders grant you an additional permission under Section 7
# of the GNU General Public License, version 3, exempting you from the
# requirement in Section 6 of the GNU General Public License, version 3, to
# accompany Corresponding Source with Installation Information for the
# Program or any work based on the Program. You are still required to comply
# with all other Section 6 requirements to provide Corresponding Source.
#

import logging

from terra.core.manager import Manager
from terra.core.terra_object import TerraObject
from terra.core.controller import Controller


log = logging.getLogger("plugins.canola_core.player")


class PlayerHook(TerraObject):
    terra_type = "Hook/Player"

    def setup(self, player):
        pass

    def delete(self):
        pass

    def media_changed(self, model):
        pass

    def playing(self):
        pass

    def paused(self):
        pass

    def volume_changed(self, volume):
        pass

    def position_changed(self, position):
        pass

    def duration_updated(self, duration):
        pass


class PlayerState(object):
    def __init__(self):
        self.model = None
        self.is_playing = False
        self.volume = None
        self.position = None
        self.duration = None


class Player(Controller):
    terra_type = "Player"
    player_type = "Player"
    hooks_type = "Hook/Player"

    def __init__(self, model, evas, parent):
        Controller.__init__(self, model, evas, parent)
        self.player_state = PlayerState()
        self.player_state.model = model
        self._setup_hooks()

    def delete(self):
        self._delete_hooks()
        self.player_state = None

    # hooks handling

    def _setup_hooks(self):
        try:
            hooks_cls = Manager().get_classes(self.hooks_type)
        except:
            log.debug("no hooks with type %r for player %r",
                      self.hooks_type, self)
            self._hooks = []
            return

        hooks = []
        app = hooks.append
        for cls in hooks_cls:
            h = cls()
            h.setup(self)
            app(h)
        self._hooks = hooks

    def _delete_hooks(self):
        for h in self._hooks:
            h.delete()
        self._hooks = []

    def add_hook(self, hook):
        hook.setup(self)
        self._hooks.append(hook)

    def remove_hook(self, hook):
        hook.delete()
        self._hooks.remove(hook)

    # informing hooks of events

    def _call_hooks(self, methname, *args):
        for h in self._hooks:
            meth = getattr(h, methname)
            meth(*args)

    def inform_media_changed(self, model):
        self.player_state.model = model
        self._call_hooks("media_changed", model)

    def inform_playing(self):
        self.player_state.is_playing = True
        self._call_hooks("playing")

    def inform_paused(self):
        self.player_state.is_playing = False
        self._call_hooks("paused")

    def inform_volume_changed(self, volume):
        self.player_state.volume = volume
        self._call_hooks("volume_changed", volume)

    def inform_position_changed(self, position):
        self.player_state.position = position
        self._call_hooks("position_changed", position)

    def inform_duration(self, duration):
        self.player_state.duration = duration
        self._call_hooks("duration_updated", duration)

    # way to hold the player (now playing)

    def must_hold(self):
        raise NotImplementedError("must be implemented by subclasses")
