# -*- coding: utf-8 -*-
#
# This file is part of Canola
# Copyright (C) 2007-2009 Instituto Nokia de Tecnologia
# Contact: Renato Chencarek <renato.chencarek@openbossa.org>
#          Eduardo Lima (Etrunko) <eduardo.lima@openbossa.org>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# Additional permission under GNU GPL version 3 section 7
#
# The copyright holders grant you an additional permission under Section 7
# of the GNU General Public License, version 3, exempting you from the
# requirement in Section 6 of the GNU General Public License, version 3, to
# accompany Corresponding Source with Installation Information for the
# Program or any work based on the Program. You are still required to comply
# with all other Section 6 requirements to provide Corresponding Source.
#

import os
from logging import getLogger

from terra.core.manager import Manager
from terra.core.plugin_prefs import PluginPrefs

__all__ = ("PodcastModelFolder", "AudioOnDemandFolder",)


log = getLogger("plugins.canola-core.podcast")
mger = Manager()
db = mger.canola_db

AudioModel = mger.get_class("Model/Media/Audio")
FatalError = mger.get_class("Model/Notify/Error/Fatal")
DefaultIcon = mger.get_class("Icon")
OptionsModelFolder = mger.get_class("Model/Options/Folder")
OptionsActionModel = mger.get_class("Model/Options/Action")
AudioFeedListOptionsModel = \
    mger.get_class("Model/Options/Folder/Media/Feeds/Audio")
OnDemandFolderOptionsModel = \
    mger.get_class("Model/Options/Folder/Media/OnDemand")

DownloadManager = mger.get_class("DownloadManager")
download_mger = DownloadManager()

(
    STATE_INITIAL,
    STATE_DOWNLOAD_DIALOG_OPENING,
    STATE_DOWNLOADING,
    STATE_PAUSED,
    STATE_DOWNLOADED,
    STATE_QUEUED,
) = range(6)


class PodcastIcon(DefaultIcon):
    terra_type = "Icon/Folder/Task/Audio/OnDemand"
    icon = "icon/main_item/music_podcast"


OnDemandMixin = mger.get_class("Model/Media/Generic/OnDemand")
class AudioOnDemandModel(OnDemandMixin, AudioModel):
    terra_type = "Model/Media/Audio/OnDemand"
    item_table = "episodes"
    feed_table = "feeds"
    download_group = "podcast"

    def __init__(self, parent):
        AudioModel.__init__(self)
        OnDemandMixin.__init__(self, parent)
        self.parent = parent # reset parent here as AudioModel annuls it.


OnDemandFolder = mger.get_class("Model/Folder/Media/OnDemand")
class AudioOnDemandFolder(OnDemandFolder):
    terra_type = "Model/Folder/Media/OnDemand/Audio"
    item_table = "episodes"
    feed_table = "feeds"
    model_cls = AudioOnDemandModel

    def __get_id(self):
        return self._id

    def __set_id(self, value):
        self._id = int(value)
        path = PluginPrefs("settings").get("cover_path")
        self.cover = os.path.join(path, "podcasts/%d/cover.jpg" % int(value))

    id = property(__get_id, __set_id)

    def is_acceptable_format(self, uri):
        uri = uri.split("?")[0]
        ext = uri.split(".")[-1]
        for ext in ['mp3', 'm4a', 'ogg', 'wma', 'avi', 'mov']:
            if uri.lower().endswith(ext):
                return True
        return False

    def options_model_get(self, controller):
        return AudioFeedListOptionsModel(None, controller)


TodaysEpisodesGenericOnDemandFolder = \
    mger.get_class("Model/Folder/Media/OnDemand/Generic/TodaysEpisodes")
class TodaysEpisodesAudioOnDemandFolder(TodaysEpisodesGenericOnDemandFolder):
    terra_type = "Model/Folder/Media/OnDemand/Audio/TodaysEpisodes"
    model_cls = AudioOnDemandModel
    feed_cls = AudioOnDemandFolder


TopRatedGenericOnDemandFolder = \
    mger.get_class("Model/Folder/Media/OnDemand/Generic/TopRated")
class TopRatedAudioOnDemandFolder(TopRatedGenericOnDemandFolder):
    terra_type = "Model/Folder/Media/OnDemand/Audio/TopRated"
    model_cls = AudioOnDemandModel


NewEpisodesGenericOnDemandFolder = \
    mger.get_class("Model/Folder/Media/OnDemand/Generic/NewEpisodes")
class NewEpisodesAudioOnDemandFolder(NewEpisodesGenericOnDemandFolder):
    terra_type = "Model/Folder/Media/OnDemand/Audio/NewEpisodes"
    model_cls = AudioOnDemandModel


CurrentDownloadsGenericOnDemandFolder = \
    mger.get_class("Model/Folder/Media/OnDemand/Generic/CurrentDownloads")
class CurrentDownloadsAudioOnDemandFolder(CurrentDownloadsGenericOnDemandFolder):
    terra_type = "Model/Folder/Media/OnDemand/Generic/CurrentDownloads/Audio"
    model_cls = AudioOnDemandModel
    feed_cls = AudioOnDemandFolder
    watched_groups = ['podcast']


FeedFolder = mger.get_class("Model/Folder/Media/Feeds")
class AudioFeedFolder(FeedFolder):
    terra_type = "Model/Folder/Media/Feeds/Audio"
    title = "My feeds"
    feed_cls = AudioOnDemandFolder
    model_cls = AudioOnDemandModel


OnDemandTaskFolder = mger.get_class("Model/Folder/Task/Generic/OnDemand")
class PodcastTaskFolder(OnDemandTaskFolder):
    terra_task_type = "Task/Podcast"
    terra_type = "Model/Folder/Task/Audio/OnDemand"
    title = "Podcasts"
    preset_section = "Podcasts"
    feed_cls = AudioOnDemandFolder
    model_cls = AudioOnDemandModel

    def __init__(self, parent):
        OnDemandTaskFolder.__init__(self, parent)

    def do_load(self):
        AudioFeedFolder(self)
        TodaysEpisodesAudioOnDemandFolder(self)
        NewEpisodesAudioOnDemandFolder(self)
        CurrentDownloadsAudioOnDemandFolder(self)
        TopRatedAudioOnDemandFolder(self)
