#
# This file is part of Canola
# Copyright (C) 2007-2009 Instituto Nokia de Tecnologia
# Contact: Renato Chencarek <renato.chencarek@openbossa.org>
#          Eduardo Lima (Etrunko) <eduardo.lima@openbossa.org>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# Additional permission under GNU GPL version 3 section 7
#
# The copyright holders grant you an additional permission under Section 7
# of the GNU General Public License, version 3, exempting you from the
# requirement in Section 6 of the GNU General Public License, version 3, to
# accompany Corresponding Source with Installation Information for the
# Program or any work based on the Program. You are still required to comply
# with all other Section 6 requirements to provide Corresponding Source.
#

import logging

from terra.core.controller import Controller, get_controller_class_for_model
from terra.core.model import Model
from terra.core.task import TaskController
from terra.core.manager import Manager

mger = Manager()
Player = mger.get_class("Player")

log = logging.getLogger("canola.multitask.task_controller")


class BaseTaskController(TaskController):
    terra_type = "TaskController"

    def __init__(self, task, parent, multitask):
        TaskController.__init__(self, task, parent, multitask)
        self.view = parent.view
        self._evas = parent.ecore_evas.evas
        self._player_state = None
        self._sub_controllers = []
        self._sub_controller = None
        self._foreground = False
        self._notify = None
        self._must_hold = True

    def _get_player(self):
        try:
            return self._player_state[-1]
        except:
            return None

    player = property(_get_player)

    def _save_player_state(self):
        state = []
        for c in self._sub_controllers:
            c.ref()
            state.append(c)
        self._player_state = state

    def _use_player_state(self):
        self._unref_sub_controllers()
        self._sub_controllers = self._player_state
        self._sub_controller = self._sub_controllers[-1]
        self._player_state = None

    def _unref_player_state(self):
        while self._player_state:
            c = self._player_state.pop()
            c.unref()
        self._player_state = None

    def _remove_nowplaying_state(self):
        player_type = self._player_state[-1].player_type
        self.multitask.remove_player(player_type, self)

    def _remove_player_state(self):
        self._remove_nowplaying_state()
        self._unref_player_state()

    def _unref_sub_controllers(self):
        while self._sub_controllers:
            c = self._sub_controllers.pop()
            c.model.state_reason = None
            c.unref()
        self._sub_controller = None
        self._sub_controllers = []

    def run(self, end_callback=None):
        self._foreground = True
        self.use(self.task, end_callback)

    def suspend(self):
        # this method is called after a transition to a new task has been
        # made by the main controller, so we're not in foreground anymore!
        assert self._sub_controller is not None or \
            self._player_state is not None

        if isinstance(self._sub_controller, Player):
            if not self._sub_controller.must_hold():
                last = self._sub_controllers.pop()
                last.suspend()
                last.unref()
                self._sub_controller = self._sub_controllers[-1]
            else:
                last = self._sub_controller
                self._save_player_state()
                self.multitask.add_player(last.player_type, self)

        self._foreground = False
        self._sub_controller.suspend()

    def resume(self, end_callback=None):
        if self._sub_controller is None:
            # empty browsed state, it's like running again
            self.run(end_callback)
            return

        def end(*ignored):
            if end_callback is not None:
                end_callback(self, self._sub_controller.model)

        if self._player_state == self._sub_controllers:
            self._remove_player_state()

        self._foreground = True
        self.parent.change_view_to(self._sub_controller, end)

    def go_to_nowplaying(self, end_callback=None):
        self._remove_nowplaying_state()

        def end(*ignored):
            self._use_player_state()
            if end_callback is not None:
                end_callback(self)

        self._foreground = True
        self.parent.change_view_to(self._player_state[-1], end)

    def remove_nowplaying(self):
        assert self._player_state is not None
        assert not self._foreground

        if self._sub_controllers == self._player_state:
            self._unref_sub_controllers()

        # player was already removed from multitask
        self._unref_player_state()

    def do_back(self):
        self._sub_controller.back()

    def do_go_home(self):
        self._sub_controller.go_home()

    def show_notify(self, model):
        self.parent.show_notify(model)

    def can_use(self, obj):
        return self.parent.can_use(obj)

    def use(self, obj, end_callback=None):
        cont_cls = None

        if isinstance(obj, Model):
            cont_cls = get_controller_class_for_model(obj)

        if cont_cls and issubclass(cont_cls, Player):
            if self._player_state is None:
                # others might be using the same player, stop them!
                self.multitask.stop_other_player(cont_cls.player_type)
            else:
                # I'm using this player, so deal with it here ;-)
                player = self._player_state[-1]
                self._remove_nowplaying_state()

                if obj == player.model:
                    # trying to play the same media
                    def end(*ignored):
                        self._sub_controller.suspend()
                        self._use_player_state()
                        if end_callback is not None:
                            end_callback(self, self._sub_controller.model)

                    self.parent.change_view_to(self._player_state[-1], end)
                    return
                else:
                    # trying to play a different media
                    self._unref_player_state()

        if isinstance(obj, Model):
            controller = cont_cls(obj, self._evas, self)
        elif isinstance(obj, Controller):
            controller = obj
        else:
            log.error("unknown object type %r (expected Model or Controller)",
                      obj.__class__.__name__)
            return

        last_controller = self._sub_controller
        self._sub_controllers.append(controller)
        self._sub_controller = controller
        self._sub_controller.ref()

        def end(*ignored):
            if last_controller is not None:
                last_controller.suspend()
            if end_callback is not None:
                end_callback(self, controller.model)

        self.parent.change_view_to(controller, end)

    def back(self, end_callback=None):
        last = self._sub_controllers.pop()

        try:
            self._sub_controller = self._sub_controllers[-1]
        except IndexError:
            self._sub_controller = None

        if isinstance(last, Player) and last.must_hold():
            def end(*ignored):
                last.suspend()
                self._save_player_state()
                self._player_state.append(last)
                self.multitask.add_player(last.player_type, self)

                if end_callback is not None:
                    end_callback(self)
        else:
            def end(*ignored):
                last.suspend()
                last.unref()
                if end_callback is not None:
                    end_callback(self)

        if self._sub_controller is not None:
            self.parent.change_view_to(self._sub_controller, end)
        else:
            self._foreground = False
            self._must_hold = self._player_state is not None
            self.parent.back(end)

    def go_home(self, end_callback=None):
        def end(*ignored):
            self.suspend()
            if end_callback is not None:
                end_callback(self)

        self._foreground = False
        self._must_hold = True
        self.parent.go_home(end)

    def propagate_error(self, controller, notify):
        if self._notify is not None:
            # only one notification per task
            assert notify is self._notify
            return
        self._notify = notify

        remove_curr_state = False
        if controller in self._sub_controllers:
            remove_curr_state = True
            idx = self._sub_controllers.index(controller)
            invalidate = self._sub_controllers[idx+1:]
            for c in invalidate:
                if c.model.is_valid:
                    c.model.state_reason = notify
            to_die = len(invalidate) + 1

        remove_player_state = False
        if self._player_state is not None and \
           controller in self._player_state:
            remove_player_state = True
            for c in self._player_state:
                if c != controller and c.model.is_valid:
                    c.model.state_reason = notify

        def cleanup():
            if remove_player_state:
                self._remove_player_state()

            if remove_curr_state:
                while self._sub_controllers:
                    c = self._sub_controllers.pop()
                    c.unref()
                    if c == controller:
                        break
                try:
                    self._sub_controller = self._sub_controllers[-1]
                except IndexError:
                    self._sub_controller = None

            self._notify = None

        if self._foreground and remove_curr_state:
            def end(*ignored):
                cleanup()

            old_cb = notify.answer_callback
            def answer_callback(*ignored):
                if old_cb is not None:
                    old_cb(*ignored)

                notify.answer_callback = None # don't call me twice
                if to_die == len(self._sub_controllers):
                    self._foreground = False
                    self._must_hold = self._player_state is not None and \
                        not remove_curr_state
                    self.parent.back(end)
                else:
                    new_controller = self._sub_controllers[idx-1]
                    self.parent.change_view_to(new_controller, end)

            notify.answer_callback = answer_callback
            self.parent.show_notify(notify)
        else:
            # we're in background or current state wasn't invalidated
            cleanup()

    def killtree(self, notify):
        if self._player_state is None and \
           self._sub_controller is None:
            # fast path, empty task
            return

        if self._notify is not None:
            # only one notification per task
            assert notify is self._notify
            return
        self._notify = notify

        if self._player_state is not None:
            for c in self._player_state:
                c.model.state_reason = notify

        for c in self._sub_controllers:
            c.model.state_reason = notify

        def cleanup():
            if self._player_state is not None:
                self._remove_player_state()
            self._unref_sub_controllers()
            self._notify = None

        if self._foreground:
            def end(*ignored):
                cleanup()

            old_cb = notify.answer_callback
            def answer_callback(*ignored):
                if old_cb is not None:
                    old_cb(*ignored)

                notify.answer_callback = None # don't call me twice
                self._foreground = False
                self._must_hold = False
                self.parent.back(end)

            notify.answer_callback = answer_callback
            self.parent.show_notify(notify)
        else:
            cleanup()

    def must_hold(self):
        return self._must_hold

    def theme_changed(self):
        subs = set(self._sub_controllers)
        if self._player_state is not None:
            subs.update(set(self._player_state))

        for c in subs:
            c.view.theme_changed()
            c.view.hide()

    def panel_stack(self, *args, **kargs):
        self.parent.panel_stack(*args, **kargs)

    def panel_unstack(self, *args, **kargs):
        self.parent.panel_unstack(*args, **kargs)

    def use_options(self, *args, **kargs):
        self.parent.use_options(*args, **kargs)

    def back_options(self, *args, **kargs):
        self.parent.back_options(*args, **kargs)

