#
# This file is part of Canola
# Copyright (C) 2007-2009 Instituto Nokia de Tecnologia
# Contact: Renato Chencarek <renato.chencarek@openbossa.org>
#          Eduardo Lima (Etrunko) <eduardo.lima@openbossa.org>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# Additional permission under GNU GPL version 3 section 7
#
# The copyright holders grant you an additional permission under Section 7
# of the GNU General Public License, version 3, exempting you from the
# requirement in Section 6 of the GNU General Public License, version 3, to
# accompany Corresponding Source with Installation Information for the
# Program or any work based on the Program. You are still required to comply
# with all other Section 6 requirements to provide Corresponding Source.
#

import ecore
import edje.decorators
import evas.decorators
import logging

from terra.ui.screen import Screen
from terra.core.controller import Controller
from terra.core.manager import Manager
from terra.core.terra_object import TerraObject
from efl_utils.animations import SinusoidalTimelineAnimation as TimelineAnimation
from efl_utils.callbacks import generate_multiple_wait

log = logging.getLogger("canola.list.ui")

mger = Manager()
KineticListWidget = mger.get_class("Widget/KineticList")
ScrollbarWidget = mger.get_class("Widget/Scrollbar")
KnobWidget = mger.get_class("Widget/Knob")


class ListScreen(Screen, TerraObject):
    terra_type = "Widget/ListScreen"
    view_name = "list"
    empty_msg = "No items found."
    SCROLLBAR_TIMEOUT = 0.7
    SCROLLBAR_FIRST_TIMEOUT = 1.5

    def __init__(self, canvas, main_window, title, elements,
                 row_renderer, group=None, theme=None):
        Screen.__init__(self, canvas, group, main_window, title, theme)
        self.elements = elements
        self.callback_clicked = None
        self._row_renderer = row_renderer
        self._hide_scroll_timer = None
        self._setup_gui_scrollbar()
        self._setup_gui_list()
        self._scrollbar_anim = None
        self._bottom_msg_timer = None
        self._start_loading = ecore.idler_add(self._start_loading_cb)

    def _setup_gui_scrollbar(self):
        self._scroll = ScrollbarWidget(self, KnobWidget(self))
        self.part_swallow("scrollbar", self._scroll)

    def _setup_gui_list(self):
        def renderer_new(canvas):
            return self._row_renderer(self)
        self._list = KineticListWidget(self, renderer_new, self.elements)
        self._list.on_resize_add(self._cb_list_resized)
        self._list.position_changed_cb_set(self._cb_position_changed)
        self._list.clicked_cb_set(self._cb_clicked)
        self.part_swallow("contents", self._list)

    def _cb_list_resized(self, lst):
        self._scroll.scale_set(lst.visible_rows_scale_get())

    def _cb_position_changed(self, lst, percent):
        self._scroll.value_set(percent)
        if self._hide_scroll_timer is not None:
            self._hide_scroll_timer.delete()
        self._hide_scroll_timer = ecore.timer_add(self.SCROLLBAR_TIMEOUT,
                                                  self._hide_scroll_cb)
        self.signal_emit("scrollbar,transition,in", "")

    def _hide_scroll_cb(self):
        self.signal_emit("scrollbar,transition,out", "")
        self._hide_scroll_timer = None
        return False

    def _cb_clicked(self, lst, index):
        self.callback_clicked(self, index)

    @edje.decorators.signal_callback("scrollbar,activate", "*")
    def _cb_scrollbar_activate(self, name, value):
        if self._scrollbar_anim is None:
            if self._list.visible_rows_count() >= len(self.elements):
                return
            def goto_pos(values):
                pos, prog = values
                self._list.position_set(pos)
                if prog >= 1.0:
                    def anim():
                        p = self._scrollbar_anim_pos
                        self._scroll.tooltip_set("%d%%" % (p * 100))
                        self._list.position_set(p)
                        return True
                    self._scroll.tooltip_show()
                    self._scrollbar_anim = ecore.animator_add(anim)

            value = float(value)
            start = (self._list.position_get(), 0.0)
            end = (value, 1.0)
            self._scrollbar_anim_pos = value
            self._scrollbar_anim = TimelineAnimation(start, end, 0.2, goto_pos)

    @edje.decorators.signal_callback("scrollbar,deactivate", "*")
    def _cb_scrollbar_deactivate(self, *ignored):
        if self._scrollbar_anim is not None:
            anim = self._scrollbar_anim
            anim.delete()
            # deleting TimelineAnimation will still call it with progress
            # being 1.0 and then it will schedule ecore.animator_add(anim),
            # so check if this happened and delete the new Animator
            if anim is not self._scrollbar_anim:
                self._scrollbar_anim.delete()
            self._scrollbar_anim = None
            self._scroll.tooltip_hide()

    @edje.decorators.signal_callback("scrollbar,move", "*")
    def _cb_scrollbar(self, name, value):
        self._scrollbar_anim_pos = float(value)

    def model_updated(self):
        self._list.model_updated()
        self._check_has_elements()

    def delete(self):
        self._list.delete()
        self._scroll.delete()
        self.elements = None
        Screen.delete(self)

    def append(self, child):
        self._list.append(child)

    def extend(self, children):
        self._list.extend(children)

    def theme_changed(self):
        Screen.theme_changed(self)
        self._list.theme_changed()
        self._scroll.theme_changed()

    def transition_from(self, old_view, end_callback=None):
        for r in self._list.renderers:
            r.state_default()
            r.unselect()
        Screen.transition_from(self, old_view, end_callback)

        if self._list.visible_rows_count() < len(self.elements):
            self._hide_scroll_timer = \
                ecore.timer_add(self.SCROLLBAR_FIRST_TIMEOUT,
                                self._hide_scroll_cb)
            self.signal_emit("scrollbar,transition,in", "")

    def select_item(self, index, end_callback=None):
        self._list.select_item(index, end_callback)

    def loaded(self):
        self._check_has_elements()
        self.throbber_stop()
        if self._start_loading is not None:
            self._start_loading.delete()
            self._start_loading = None

    def _check_has_elements(self):
        if self.elements:
            self.signal_emit("message,hide", "")
        else:
            self.part_text_set("message", self.empty_msg)
            self.signal_emit("message,show", "")

    def bottom_message_show(self, msg, timeout):
        self.part_text_set("bottom_msg", msg)
        self.signal_emit("bottom,message,show", "")

        def _bottom_msg_hide():
            self.signal_emit("bottom,message,hide", "")
            return False

        if self._bottom_msg_timer:
            self._bottom_msg_timer.delete()
            self._bottom_msg_timer = None

        self._bottom_msg_timer = ecore.timer_add(timeout, _bottom_msg_hide)

    def _start_loading_cb(self):
        if self._start_loading is not None:
            self._start_loading.delete()
        self._start_loading = None
        self.throbber_start()

    def throbber_start(self):
        t = self.part_swallow_get("throbber")
        t.signal_emit("throbber,start", "")

    def throbber_stop(self):
        t = self.part_swallow_get("throbber")
        t.signal_emit("throbber,stop", "")

    @evas.decorators.del_callback
    def _on_delete(self, *ignored):
        self.elements = None


class OnTheGoListScreen(ListScreen):
    terra_type = "Widget/ListScreen/OnTheGo"
    empty_msg = "Click and hold on a song or folder to add it to this list."


def row_value_set(self, v):
    self.part_text_set("text", v.name)

RowRendererFactory = mger.get_class("Widget/SimpleRowRendererFactory")
class ListController(Controller):
    terra_type = "Controller/Folder"
    row_renderer = RowRendererFactory(row_value_set, "list_item_text")
    list_group = "list"
    view_class = ListScreen
    empty_msg = None

    def __init__(self, model, canvas, parent):
        Controller.__init__(self, model, canvas, parent)
        self.animating = False
        self.model.load()
        self._setup_view()

        # should be after setup UI
        self.model.changed_callback_add(self._update_ui)
        self.model.callback_state_changed = self._model_state_changed
        self._check_model_loaded()

    def _setup_view(self):
        title = self.model.name
        self.view = self.view_class(self.evas, self.parent.view, title,
                                    self.model.children, self.row_renderer,
                                    self.list_group)
        if self.empty_msg:
            self.view.empty_msg = self.empty_msg
        self.view.callback_clicked = self.cb_on_clicked

    def _update_ui(self, model):
        self.view.model_updated()

    def _check_model_loaded(self):
        if self.model.is_loaded:
            self.view.loaded()
        else:
            self.model.callback_loaded = self._model_loaded

    def _model_loaded(self, model):
        self.view.loaded()
        model.callback_loaded = None

    def _model_state_changed(self, model):
        if not model.is_valid:
            self.parent.propagate_error(self, model.state_reason)

    def delete(self):
        self.model.callback_state_changed = None

        # should be before deleting UI
        self.model.changed_callback_del(self._update_ui)

        self.model.callback_loaded = None
        self.view.delete()
        self.model.unload()

    def back(self):
        if self.animating:
            log.info("click ignored: was animating")
            return

        def end(*ignored):
            self.animating = False

        self.animating = True
        self.parent.back(end)

    def cb_on_animation_ended(self, *ignored):
        self.animating = False

    def cb_on_clicked(self, view, index):
        if self.animating:
            log.info("click ignored: was animating")
            return

        model = self.model.children[index]
        if hasattr(model, "activatable"):
            if not model.activatable:
                return

        if not self.parent.can_use(model):
            self.parent.show_notify(model.state_reason)
            return

        end = generate_multiple_wait(2, self.cb_on_animation_ended)
        self.animating = True
        self.view.select_item(index, end)

        self.model.current = index
        self.parent.use(model, end)

    def go_home(self):
        self.parent.go_home()
