#! /usr/bin/python
# - coding: utf-8 -*-

# Copyright (c) 2007-2008 Renato Chencarek. All rights reserved.
#
# Author: Renato Chencarek <renato.chencarek@gmail.com>
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED,
# THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
# PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR
# CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
# EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
# PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
# PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
# LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
# NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
# SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.


import sys, os, time
import dbus
import dbus.service
import gobject
import dbus.glib

import urllib
import feedparser

import cPickle
from threading import Thread

import gtk
import gobject

error_msg = ""
timeout = None
main_loop = gobject.MainLoop()
rfd, wfd = os.pipe()

try:
    import hildon
except:
    pass

try:
    canola_data = cPickle.load(open(os.path.expanduser('~/.canola/prefs/settings')))
except Exception, e:
    print "Settings Error! Run Canola to fix this."
    print e
    sys.exit()

try:
    from sqlite3 import dbapi2 as sqlite
except ImportError:
    from pysqlite2 import dbapi2 as sqlite

def to_utf8(text):
    """Best-effort converting an arbitrary string to byte string in UTF-8"""

    if type(text) is unicode:
        return text.encode("utf-8")

    encoding_lst = [("utf-8",), ("latin-1",), ("utf-8", "replace")]
    for encoding in encoding_lst:
        try:
            return text.decode(*encoding).encode("utf-8")
        except:
            pass
    return "???"   # Return something indicating we had problem w/ encoding

def open_db_connection(db_path):
    try:
        con = sqlite.connect(db_path)
        cur = con.cursor()
    except Exception, e:
        print "Database Error!"
        print e
        sys.exit()

    return con, cur

def check_if_exist(cur, sql, uri):
    try:
        cur.execute(sql, (uri, ))
    except Exception, e:
        return False

    return cur.fetchone()

def insert_new_feed(cur, create, sql, values):
    try:
        cur.execute(create)
        cur.execute(sql, values)
    except Exception, e:
        print "Insert: Database Error!"
        print e, sql, values
        return -1

    return cur.lastrowid

def download_cover(url, id):
    path = os.path.join(canola_data["cover_path"], "podcasts", id)
    if not os.path.exists(path):
        os.makedirs(path)
    cover = os.path.join(path, "cover.jpg")

    try:
        data = urllib.urlopen(url).read()
    except:
        return

    if not data:
        return

    f = open(cover, "wr")
    f.write(data)
    f.close()


class FeedDialog(gtk.Dialog):

    feeds = ['Canola Podcast', 'Canola Internet Radio',  'Canola Photocast', 'RSS feed reader']

    def __init__(self, parent=None):
        gtk.Dialog.__init__(self, 'Feed Handler', parent)
        self.set_has_separator(False)
        self.vbox.set_spacing(6)

        label = gtk.Label()
        label.set_markup('<b>Add this feed to:</b>' + ' ' * 12)
        label.set_justify(gtk.JUSTIFY_LEFT)
        label.set_padding(8, 4)
        self.vbox.pack_start(label, expand=False)

        alignment = gtk.Alignment(xalign=0.2)
        vbox = gtk.VBox(spacing=6)
        vbox.set_border_width(10)

        self.radios = {}
        last = None
        for feed in FeedDialog.feeds:
            last = gtk.RadioButton(last, feed)
            self.radios[feed] = last
            vbox.pack_start(last)

        alignment.add(vbox)
        self.vbox.pack_start(alignment)

        self.add_buttons(gtk.STOCK_CANCEL, gtk.RESPONSE_CANCEL,
                         gtk.STOCK_OK, gtk.RESPONSE_OK)

        self.show_all()

    def set_feed(self, feed_id):
        self.radios[feed[feed_id]].set_active(True)

    def get_feed(self):
        for radio in self.radios.keys():
            if self.radios[radio].get_active():
                return radio

        return None


class ProgressDialog(gtk.Dialog):

    def __init__(self, parent=None):
        gtk.Dialog.__init__(self, 'Downloading feed', parent)
        self.vbox.set_spacing(6)
        self.running = False

        self.progress = gtk.ProgressBar()
        self.progress.pulse()
        self.progress.set_size_request(360, -1)

        self.vbox.pack_start(self.progress)

        self.add_buttons(gtk.STOCK_CANCEL, gtk.RESPONSE_CANCEL)
        self.action_area.set_layout(0)
        self.show_all()

    def __pulse(self):
        self.progress.pulse()
        return self.running

    def stop(self):
        self.running = False

    def run(self):
        self.running = True
        gobject.timeout_add(100, self.__pulse)
        return gtk.Dialog.run(self)

def emit_global_error_msg(msg):
    global error_msg
    error_msg = msg
    print "Error:", msg
    os.write(wfd, "1")

def stop_loading_dialog(fd, cond, fhandler):
    r = os.read(fd, 1)
    fhandler.loading.stop()
    fhandler.loading.destroy()
    return False

def create_progress_dialog(window, cancel_cb=None):
    dialog = ProgressDialog(window)

    def _distract_response(dialog, response_id):
        global timeout
        timeout = 0.00001
        dialog.stop()
        dialog.destroy()

    if not cancel_cb:
        dialog.connect('response', _distract_response)
    else:
        dialog.connect('response', cancel_cb)

    return dialog

def show_feed_type_dialog(window):
    dialog = FeedDialog(window)
    response = dialog.run()
    dialog.destroy()

    if response == gtk.RESPONSE_OK:
        return dialog.get_feed()
    else:
        return None


class FeedHandler(dbus.service.Object):
    DBUS_SERVICE_NAME = "org.maemo.garage.feedhandler"
    DBUS_OBJ_PATH     = "/org/maemo/garage/feedhandler"
    DBUS_IFACE        = "org.maemo.garage.feedhandler"

    podcast_table  = "feeds"
    podcast_create = ("CREATE TABLE IF NOT EXISTS %s (id INTEGER PRIMARY KEY, "
                      "uri VARCHAR UNIQUE, last_updated VARCHAR, title VARCHAR, "
                      "desc VARCHAR, epoch INTEGER NOT NULL)") % podcast_table
    podcast_insert = ("INSERT INTO %s (uri, title, desc, last_updated, epoch) "
                      "values (?, ?, ?, ?, ?)") % podcast_table
    podcast_exists = "SELECT * FROM %s WHERE uri == ?" % podcast_table
    photocast_table  = "photocast_feeds"
    photocast_create = ("CREATE TABLE IF NOT EXISTS %s (id INTEGER "
                        "PRIMARY KEY, uri VARCHAR UNIQUE, "
                        "title VARCHAR, desc VARCHAR, author VARCHAR, "
                        "epoch INTEGER NOT NULL)") % photocast_table
    photocast_insert = ("INSERT INTO %s (uri, title, desc, author, epoch) "
                        "values (?, ?, ?, ?, ?)") % photocast_table
    photocast_exists = "SELECT * FROM %s WHERE uri == ?" % photocast_table



    iradio_table  = "iradio_feeds"
    iradio_create = ("CREATE TABLE IF NOT EXISTS %s (id INTEGER PRIMARY KEY, "
                     "uri VARCHAR UNIQUE, mimetype VARCHAR, title VARCHAR, "
                     "desc VARCHAR, epoch INTEGER NOT NULL)") % iradio_table
    iradio_insert = ("INSERT INTO %s (uri, mimetype, title, desc, epoch) "
                     "values (?, ?, ?, ?, ?)") % iradio_table
    iradio_exists = "SELECT * FROM %s WHERE uri == ?" % iradio_table


    def __init__(self):
        self.session_bus = dbus.SessionBus()
        self.bn = dbus.service.BusName(self.DBUS_SERVICE_NAME,
                                       bus=self.session_bus)

        dbus.service.Object.__init__(self, self.bn, self.DBUS_OBJ_PATH)

    def add_feed(self, feed_type, url):
        self.con, self.cur = open_db_connection(os.path.expanduser("~/.canola/canola.db"))

        if "PODCAST" in feed_type.upper():
            self.add_podcast(url)
        elif "PHOTOCAST" in feed_type.upper():
            self.add_photocast(url)
        elif "RADIO" in feed_type.upper():
            self.add_iradio(url)
        else:
            emit_global_error_msg("")

        self.con.close()


    def add_podcast(self, url):
        if check_if_exist(self.cur, FeedHandler.podcast_exists, url):
            emit_global_error_msg("Feed already inserted into Canola DB.")
            return

        contents = feedparser.parse(url)
        data = contents.feed
        last_update = None

        if not data.has_key("title"):
            emit_global_error_msg("Invalid feed format")
            return

        title = to_utf8(data.title)
        if data.has_key("description"):
            desc = to_utf8(data.description)
        else:
            desc = ""

        values = (url, title, desc, last_update, int(time.time()))
        item_id = insert_new_feed(self.cur,
                                  FeedHandler.podcast_create,
                                  FeedHandler.podcast_insert,
                                  values)
        if item_id == -1:
            emit_global_error_msg("Error adding feed to Canola DB.")
            return

        if data.has_key("image"):
            download_cover(data.image.href, str(item_id))

        emit_global_error_msg("")
        self.con.commit()


    def add_photocast(self, url):
        if check_if_exist(self.cur, FeedHandler.photocast_exists, url):
            emit_global_error_msg("Feed already inserted into Canola DB.")
            return

        contents = feedparser.parse(url)
        data = contents.feed

        if not data.has_key("title"):
            emit_global_error_msg("Invalid feed format")
            return

        title = to_utf8(data.title)
        if data.has_key("description"):
            desc = to_utf8(data.description)
        else:
            desc = ""

        if data.has_key("author_detail") and data.author_detail.has_key("name"):
            author = data.author_detail.name
        elif data.has_key("author"):
            author = data.author
        else:
            author = ""

        values = (url, title, desc, author, int(time.time()))
        item_id = insert_new_feed(self.cur,
                                  FeedHandler.photocast_create,
                                  FeedHandler.photocast_insert,
                                  values)
        if item_id == -1:
            emit_global_error_msg("Error adding feed to Canola DB.")
            return

        emit_global_error_msg("")
        self.con.commit()

    def add_iradio(self, url):
        if check_if_exist(self.cur, FeedHandler.iradio_exists, url):
            emit_global_error_msg("Feed already inserted into Canola DB.")
            return

        contents = feedparser.parse(url)
        data = contents.feed

        if not data.has_key("title"):
            title = url
        else:
            title = to_utf8(data.title)

        if data.has_key("description"):
            desc = to_utf8(data.description)
        else:
            desc = ""

        mimetype = contents.headers['content-type']

        values = (url, mimetype, title, desc, int(time.time()))
        item_id = insert_new_feed(self.cur,
                                  FeedHandler.iradio_create,
                                  FeedHandler.iradio_insert,
                                  values)
        if item_id == -1:
            emit_global_error_msg("Error adding feed to Canola DB.")
            return

        emit_global_error_msg("")
        self.con.commit()

    @dbus.service.method(DBUS_IFACE)
    def mime_open(self, url):
        global error_msg, timeout, main_loop

        try:
            self.window = hildon.Window()
        except:
            self.window = gtk.Window()

        self.window.connect('destroy', gtk.main_quit)
        feed_type = show_feed_type_dialog(self.window)

        if not feed_type:
            main_loop.quit()
            return
        elif "RSS" in feed_type.upper():
            service_name = "com.nokia.osso_rss_feed_reader_refresh"
            obj_path = "/com/nokia/osso_rss_feed_reader_refresh"
            session_bus = dbus.SessionBus()
            obj = session_bus.get_object(service_name, obj_path)
            iface = dbus.Interface(obj, service_name)
            iface.mime_open(url)
            main_loop.quit()
            return

        handler = Thread(target=self.add_feed, args=(feed_type, url))

        self.loading = create_progress_dialog(self.window, None)
        tag = gobject.io_add_watch(rfd, gobject.IO_IN, stop_loading_dialog, self)

        handler.setDaemon(True)
        handler.start()

        self.loading.run()

        gobject.source_remove(tag)

        if not timeout:
            if error_msg:
                msg = gtk.MessageDialog(self.window, gtk.DIALOG_MODAL,
                                        gtk.MESSAGE_ERROR, gtk.BUTTONS_CLOSE,
                                        error_msg)
            else:
                msg = gtk.MessageDialog(self.window, gtk.DIALOG_MODAL,
                                        gtk.MESSAGE_INFO, gtk.BUTTONS_CLOSE,
                                        "Feed Added")

            msg.run()
            msg.destroy()

        handler.join(timeout)
        main_loop.quit()

if __name__ == "__main__":
    gtk.gdk.threads_init()

    f = FeedHandler()

    if len(sys.argv) == 2:
        gobject.timeout_add(100, f.mime_open, sys.argv[1])

    main_loop.run()


