#include "flickcharm.h"

#include <QApplication>
#include <QAbstractScrollArea>
#include <QMouseEvent>
#include <QScrollBar>

#include <QDebug>

FlickCharm::FlickCharm(QAbstractScrollArea *scrollArea, QObject *parent): QObject(parent) {

  scrollArea->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
  scrollArea->setVerticalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
    
  addFilter(scrollArea);

  m_scrollArea = scrollArea;
  m_steady = true;
}

FlickCharm::~FlickCharm() {
}

void FlickCharm::scrollTo(const QPoint &newPosition) {
  const QPoint delta = newPosition - m_lastPos;
  const int x = m_scrollArea->horizontalScrollBar()->value();
  const int y = m_scrollArea->verticalScrollBar()->value();
  m_scrollArea->horizontalScrollBar()->setValue(x - delta.x());
  m_scrollArea->verticalScrollBar()->setValue(y - delta.y());
}

void FlickCharm::addFilter(QWidget *widget) {
  widget->installEventFilter(this);

  const QObjectList& childList = widget->children();
  for(QObjectList::const_iterator it = childList.begin(); it != childList.end(); ++it) 
    if(*it && (*it)->isWidgetType()) 
      addFilter(static_cast<QWidget *>(*it));
}

bool FlickCharm::eventFilter(QObject *, QEvent *event) {
  QEvent::Type type = event->type();

  // make sure that all child widgets added afterwards also get
  // the event filter installed
  if(type == QEvent::ChildAdded) {
    QChildEvent *childEvent = static_cast<QChildEvent*>(event);
    QObject *child = childEvent->child();
    if(child->isWidgetType()) 
      addFilter(static_cast<QWidget *>(child));

    return false;
  }

  if(type == QEvent::MouseButtonDblClick)
    return true; 

  if((type != QEvent::MouseButtonPress) &&
     (type != QEvent::MouseButtonRelease) &&
     (type != QEvent::MouseMove))
    return false;
  
  QMouseEvent *mouseEvent = static_cast<QMouseEvent*>(event);
  if (type == QEvent::MouseMove && mouseEvent->buttons() != Qt::LeftButton)
    return false;
  
  if (mouseEvent->modifiers() != Qt::NoModifier)
    return false;

  const QPoint mousePos = mouseEvent->globalPos();
  
  bool consumed = false;
  if(m_steady) {
    // just ignore press and release events in this state ...
    if (type == QEvent::MouseMove) {
      if ((mousePos - m_lastPos).manhattanLength() > 5) {
	scrollTo(mousePos);
	m_steady = false;
      }
      consumed = true;
    }
  } else {
    if (type == QEvent::MouseMove) {
      scrollTo(mousePos);
      consumed = true;
    } else if (type == QEvent::MouseButtonRelease) {
      m_steady = true;
    }
  }

  m_lastPos = mousePos;
  return consumed;
}

