/*
  cacheprovider, class answering requests for caches

  The cacheprovider handles requests asynchronously and 
  internally uses a queue to store requests. It also drops
  unhandled pending requests if a newer request of the same
  time comes in
*/

#include <QDebug>

// to load plugins
#include <QApplication>
#include <QPluginLoader>
#include <QDir>
#include <QSettings>

#include "cacheproviderplugin.h"
#include "cacheprovider.h"

CacheProvider::RequestEntry::RequestEntry(const RequestType &type, const QString &cache) {

  Q_ASSERT((type == Info) || (type == Detail));

  this->m_type = type;
  this->m_cache = cache;
}

CacheProvider::RequestEntry::RequestEntry(const RequestType &type, const QGeoBoundingBox &bbox) {

  Q_ASSERT(type == Overview);

  this->m_type = type;
  this->m_bbox = bbox;
}

CacheProvider::RequestEntry::RequestType CacheProvider::RequestEntry::type() {
  return this->m_type;
}

QGeoBoundingBox CacheProvider::RequestEntry::bbox() {
  return this->m_bbox;
}

QString CacheProvider::RequestEntry::cache() {
  return this->m_cache;
}

void CacheProvider::RequestEntry::set(const QString &cache) {
  this->m_cache = cache;
}

void CacheProvider::RequestEntry::set(const QGeoBoundingBox &bbox) {
  this->m_bbox = bbox;  
}

CacheProvider::RequestQueue::RequestQueue(CacheProvider *cacheProvider) {
  this->m_cacheProvider = cacheProvider;
}

bool CacheProvider::RequestQueue::add(const RequestEntry::RequestType &type, const QString &cache) {

  // ignore detail requests if there's already one being processed. This
  // prevents multiple stacked windows from opening on maemo5
  if(size() && (type == RequestEntry::Detail) && 
     (head()->type() == RequestEntry::Detail)) {
    qDebug() << __FUNCTION__ << "supressing addional Detail request";
    return false;
  }

  // check if there's already the same type of request in
  // queue and overwrite that one
  if(size() > 1) {
    for(int i=1;i<size();i++) {
      if(at(i)->type() == type) {
	at(i)->set(cache);
	return false;
      }      
    }
  }

  enqueue(new RequestEntry(type, cache));
  return true;
}

bool CacheProvider::RequestQueue::add(const RequestEntry::RequestType &type, const QGeoBoundingBox &bbox) {

  // check if there's already the same type of request in
  // queue and overwrite that one
  if(size() > 1) {
    for(int i=1;i<size();i++) {
      if(at(i)->type() == type) {
	at(i)->set(bbox);
	return false;
      }      
    }
  }

  enqueue(new RequestEntry(type, bbox));
  return true;
}

CacheProvider::RequestEntry::RequestType CacheProvider::RequestQueue::type() {
  return head()->type();
}

void CacheProvider::RequestQueue::done() {
  qDebug() << __FUNCTION__;
  delete dequeue();
}

void CacheProvider::RequestQueue::next() {

  if(!empty()) {

    // process topmost entry
    switch(type()) {
    case RequestEntry::Overview:
      qDebug() << __FUNCTION__ << "Pending Overview request";
      m_cacheProvider->processRequestOverview(head()->bbox()); 
      break;

    case RequestEntry::Info:
      qDebug() << __FUNCTION__ << "Pending Info request";
      m_cacheProvider->processRequestInfo(head()->cache()); 
      break; 

    case RequestEntry::Detail:
      qDebug() << __FUNCTION__ << "Pending Detail request";
      m_cacheProvider->processRequestDetail(head()->cache()); 
      break; 

    default:
      Q_ASSERT(0);
      break;
    }    
  } else
    qDebug() << __FUNCTION__ << "No pending request";
}

void CacheProvider::RequestQueue::restart() {
  // restart if queue holds one entry and if the cache provider
  // is not busy doing its own stuff (e.g. logging in)
  if((size() == 1) && !m_cacheProvider->busy()) 
    next();
}

void CacheProvider::loadPluginsInDir(QDir &pluginsDir) {

  if(!pluginsDir.cd("plugins")) return;
  if(!pluginsDir.cd("cacheprovider")) return;
  
  qDebug() << __FUNCTION__ << "Loading plugins from:" << pluginsDir;

  // load all available plugins
  foreach (QString fileName, pluginsDir.entryList(QDir::Files)) {
    if(QLibrary::isLibrary(fileName)) {
      qDebug() << __FUNCTION__ << fileName;
      QPluginLoader loader(pluginsDir.absoluteFilePath(fileName));
      QObject *plugin = loader.instance();
      if(!loader.isLoaded()) 
	qDebug() << __FUNCTION__ << loader.errorString();
      else if (plugin) {
	CacheProviderPlugin *cacheProviderPlugin = 
	  qobject_cast<CacheProviderPlugin *>(plugin);
	
	m_cacheProviderPluginList.append(cacheProviderPlugin);
	
	qDebug() << __FUNCTION__ << "Plugin name:" << 
	  cacheProviderPlugin->name();
      }
    }
  }
}
  
CacheProvider::CacheProvider(QWidget *parent) : 
  m_currentPlugin(NULL), m_parent(parent) {
  QDir pluginsDir = QDir(QApplication::applicationDirPath());

  // try to load plugins from local path first
#if defined(Q_OS_WIN)
  if (pluginsDir.dirName().toLower() == "debug" || 
      pluginsDir.dirName().toLower() == "release")
    pluginsDir.cdUp();
#endif

  loadPluginsInDir(pluginsDir);

#ifdef LIBDIR
  if(m_cacheProviderPluginList.isEmpty()) {
    // if no local plugins were found -> try the intallation path
    pluginsDir = QDir(LIBDIR);  
    loadPluginsInDir(pluginsDir);
  }
#endif

  this->m_pending = new RequestQueue(this);

  QSettings settings;
  settings.beginGroup("CacheProvider");
  QString name = settings.value("name").toString();
  settings.endGroup();

  // try to load preset plugin 
  foreach(CacheProviderPlugin *plugin, m_cacheProviderPluginList) {
    if(plugin->name() == name) {
      this->m_currentPlugin = plugin;
      return;
    }
  }
  
  // activate first default capable plugin if available
  foreach(CacheProviderPlugin *plugin, m_cacheProviderPluginList) {
    if(plugin->canBeDefault()) {
      this->m_currentPlugin = plugin;
      return;
    }
  }
}

CacheProvider::~CacheProvider() {
  qDebug() << __FUNCTION__;

#if 0
  foreach(CacheProviderPlugin *plugin, m_cacheProviderPluginList) 
    delete plugin;
#endif

  delete this->m_pending;
}

void CacheProvider::requestOverview(const QGeoBoundingBox &area) {
  qDebug() << __PRETTY_FUNCTION__ << "from" << area.topLeft() << 
    " to " << area.bottomRight();

  if(m_pending->add(RequestEntry::Overview, area))
    m_pending->restart();
}

void CacheProvider::requestInfo(const QString &cache) {
  qDebug() << __PRETTY_FUNCTION__ << cache;

  if(m_pending->add(RequestEntry::Info, cache))
    m_pending->restart();
}

void CacheProvider::requestDetail(const QString &cache) {
  qDebug() << __PRETTY_FUNCTION__ << cache;

  if(m_pending->add(RequestEntry::Detail, cache))
    m_pending->restart();
}

void CacheProvider::next() {
  this->m_pending->next();
}

void CacheProvider::done() {
  this->m_pending->done();
}

CacheProvider::RequestEntry::RequestType CacheProvider::type() {
  return this->m_pending->type();
}

// the following functions are called by the plugin in order to 
// let the CacheProvider emit a signal
void CacheProvider::emitReplyOverview(const CacheList &cacheList) {
  emit replyOverview(cacheList);
}

void CacheProvider::emitReplyInfo(const Cache &cache) {
  emit replyInfo(cache);
}

void CacheProvider::emitReplyDetail(const Cache &cache) {
  emit replyDetail(cache);
}

void CacheProvider::emitReplyError(const QString &msg) {
  emit replyError(msg);
}

void CacheProvider::emitReload() {
  emit reload();
}

void CacheProvider::emitNotifyBusy(bool on) {
  emit notifyBusy(on);
}

QString CacheProvider::name() {
  if(!m_currentPlugin) return("<none>");
  return m_currentPlugin->name();
}

bool CacheProvider::busy() {
  if(!m_currentPlugin) return(true);
  return m_currentPlugin->busy();
}
 
void CacheProvider::processRequestOverview(const QGeoBoundingBox &bbox) {
  if(m_currentPlugin)
    m_currentPlugin->processRequestOverview(bbox);
}

void CacheProvider::processRequestInfo(const QString &id) {
  if(m_currentPlugin)
    m_currentPlugin->processRequestInfo(id);
}

void CacheProvider::processRequestDetail(const QString &id) {
  if(m_currentPlugin)
    m_currentPlugin->processRequestDetail(id);
}

void CacheProvider::start(QWidget *parent) {
  foreach(CacheProviderPlugin *cplugin, m_cacheProviderPluginList) {
    QObject *plugin = cplugin->object();

    // connect to plugin
    connect(plugin, SIGNAL(replyOverview(const CacheList &)), 
	    this, SLOT(emitReplyOverview(const CacheList &)));

    connect(plugin, SIGNAL(replyInfo(const Cache &)), 
	    this, SLOT(emitReplyInfo(const Cache &)));

    connect(plugin, SIGNAL(replyDetail(const Cache &)), 
	    this, SLOT(emitReplyDetail(const Cache &)));

    connect(plugin, SIGNAL(replyError(const QString &)), 
	    this, SLOT(emitReplyError(const QString &)));
    
    connect(plugin, SIGNAL(reload()), 
	    this, SLOT(emitReload()));
    
    connect(plugin, SIGNAL(notifyBusy(bool)), 
	    this, SLOT(emitNotifyBusy(bool)));
    
    connect(plugin, SIGNAL(done()), this, SLOT(done()));
    
    connect(plugin, SIGNAL(next()), this, SLOT(next()));
  }

  // call active plugins init function
  if(this->m_currentPlugin) 
    this->m_currentPlugin->init(parent);
}

void CacheProvider::pluginConfig(QDialog *parent, QVBoxLayout *box) {
  foreach(CacheProviderPlugin *plugin, m_cacheProviderPluginList)
    plugin->createConfig(parent, box);
}

void CacheProvider::providerSelected(QAction *action) {
  CacheProviderPlugin *plugin = NULL;

  // figure out which entry was selected
  foreach(CacheProviderPlugin *cp, m_cacheProviderPluginList) 
    if(cp->name() == action->text())
      plugin = cp;

  Q_ASSERT(plugin);

  qDebug() << __FUNCTION__ << plugin->name();  

  this->m_currentPlugin = plugin;
  this->m_currentPlugin->init(this->m_parent);

  QSettings settings;
  settings.beginGroup("CacheProvider");
  settings.setValue("name", plugin->name());
  settings.endGroup();
}

void CacheProvider::createMenu(QMenuBar *menuBar) {

  QMenu *cacheProvider = menuBar->addMenu(tr("&Cache Provider"));
  QActionGroup *filterGroup = new QActionGroup(this);
  filterGroup->setExclusive(true);

  foreach(CacheProviderPlugin *plugin, m_cacheProviderPluginList) {
    QAction *action = new QAction(plugin->name(), filterGroup);
    action->setCheckable(true);

    if(m_currentPlugin == plugin)
      action->setChecked(true);
  }

  connect(filterGroup, SIGNAL(triggered(QAction *)), 
	  this, SLOT(providerSelected(QAction *)));
  
  cacheProvider->addActions(filterGroup->actions());        
}
