// Filename: mainwindow.cpp

#include <QDebug>
#include <QMessageBox>

#include "mainwindow.h"
#include "mapwidget.h"
#include "settingsdialog.h"
#include "iconloader.h"  
#include "cachewindow.h"

#include <QIcon>
#include <QSettings>

#ifndef Q_WS_MAEMO_5
#include <QVBoxLayout>
#else
#include <QMaemo5InformationBox>
#endif

void MainWindow::createMenu() {
  addMenuEntry(tr("&About"), this, SLOT(about()));
  addMenuEntry(tr("&Settings"), this, SLOT(settingsdialog()));

  /* create a submenu for the plugin selection */  
  m_cacheProvider->createMenu(menuBar());
}

void MainWindow::settingsdialog() {
  qDebug() << __FUNCTION__;

  SettingsDialog dialog(m_mapWidget, m_cacheProvider, this);
  dialog.exec(); 
}

void MainWindow::about() {
  QMessageBox::about(this, tr("About CacheMe"),
	     tr("CacheMe") + " " + tr("Version") + " " + VERSION + "\n" + 
	     tr("Build date:") + " " + __DATE__ + "\n" +
	     tr("(c) 2011 by Till Harbaum <till@harbaum.org>") + "\n\n" +
	     tr("CacheMe is an online geocaching application.")
	     );
}


MainWindow::MainWindow(LocationProvider *locationProvider, QWidget *parent) : 
  CustomWindow("MainWindow", parent), m_locationProvider(locationProvider) {

  qDebug() << __FUNCTION__;
  m_cacheProvider = new CacheProvider(this);

  setWindowTitle(tr("CacheMe"));

  IconLoader iconLoader(48);
  QPixmap *icon = iconLoader.load(APPNAME);
  if(icon) setWindowIcon(QIcon(*icon));

  createMenu();

  // search for service providers
  QStringList list = QGeoServiceProvider::availableServiceProviders();
  for(int i=0;i<list.size();i++)
    qDebug() << __FUNCTION__ << "Service provider: " << list[i];

  // there needs to be at least one geo service provider
  Q_ASSERT(list.size() > 0);

  QSettings settings;
  settings.beginGroup("Map");
  QString serviceProviderStr = settings.value("Service", "nokia").toString();
  settings.endGroup();

  this->m_serviceProvider = new QGeoServiceProvider(serviceProviderStr);
  if(!this->m_serviceProvider->mappingManager())
    this->m_serviceProvider = new QGeoServiceProvider("nokia");

  this->m_mapWidget = new MapWidget(this->m_serviceProvider->mappingManager());

  // make sure cache updates are processed
  QObject::connect(this->m_locationProvider, SIGNAL(positionUpdated(const QGeoPositionInfo &)), 
		   this->m_mapWidget, SLOT(positionUpdated(const QGeoPositionInfo &)));

  QObject::connect(this->m_cacheProvider, SIGNAL(replyOverview(const CacheList &)), 
		   this, SLOT(updateCaches(const CacheList &)));

  QObject::connect(this->m_cacheProvider, SIGNAL(replyInfo(const Cache &)), 
		   this->m_mapWidget, SLOT(showBubble(const Cache &)));

  QObject::connect(this->m_cacheProvider, SIGNAL(replyDetail(const Cache &)), 
		   this, SLOT(showDetail(const Cache &)));

  QObject::connect(this->m_cacheProvider, SIGNAL(replyError(const QString &)), 
		   this, SLOT(displayError(const QString &)));

  QObject::connect(this->m_cacheProvider, SIGNAL(reload()),
		   this, SLOT(reloadCaches()));

  QObject::connect(this->m_cacheProvider, SIGNAL(notifyBusy(bool)),
		   this, SLOT(setBusy(bool)));

  QObject::connect(this->m_mapWidget, SIGNAL(mapChanged()),
		   this, SLOT(reloadCaches()));

  QObject::connect(this->m_mapWidget, SIGNAL(showMessage(const QString &)),
		   this, SLOT(statusMessage(const QString &)));

  QObject::connect(this, SIGNAL(zoomInPressed()),
		   this->m_mapWidget, SLOT(zoomIn()));

  QObject::connect(this, SIGNAL(zoomOutPressed()),
		   this->m_mapWidget, SLOT(zoomOut()));

  QObject::connect(this->m_mapWidget, SIGNAL(fullscreen()),
		   this, SLOT(toggleFullscreen()));

  // connect to map to handle clicks on cache icons ...
  QObject::connect(this->m_mapWidget, SIGNAL(cacheClicked(const QString &)),
		   this, SLOT(selectCache(const QString &)));

  // ... and the detail button
  QObject::connect(this->m_mapWidget, SIGNAL(detailClicked(const QString &)),
		   this, SLOT(detailCache(const QString &)));

  // build a graphics scene based on the mapwidget
  QGraphicsScene *scene = new QGraphicsScene(this);
  m_view = new QGraphicsView( scene );
  m_view->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
  m_view->setVerticalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
  m_view->setVisible(true);
  m_view->setInteractive(true);
  scene->addItem(m_mapWidget);
  setCentralWidget(m_view);

  m_cacheProvider->start();

#ifdef EXPIRATION_DATE
  // create expiration timer if required
  if(QDate::fromString(EXPIRATION_DATE, "ddMMyyyy") <=
     QDate::currentDate()) {
    statusMessage(tr("This copy of CacheMe has expired and will close in one minute!"));
    QTimer::singleShot(60000, this, SLOT(close()));
  } else {
    statusMessage(tr("This copy of CacheMe will expire in %1 days!").
		  arg(QDate::currentDate().daysTo(
		  QDate::fromString(EXPIRATION_DATE, "ddMMyyyy"))));
  }
#endif  
}

MainWindow::~MainWindow() {
  qDebug() << __FUNCTION__;

  delete m_cacheProvider;
  delete m_serviceProvider;
}

void MainWindow::resizeEvent(QResizeEvent*) {
  m_view->setSceneRect(QRect(QPoint(0,0), m_view->contentsRect().size()));
  m_mapWidget->resize(m_view->contentsRect().size());
}

void MainWindow::selectCache(const QString &name) {
  // ask CacheProvider for info about this cache
  this->m_cacheProvider->requestInfo(name);
}

void MainWindow::detailCache(const QString &name) {
  // ask CacheProvider for details about this cache
  this->m_cacheProvider->requestDetail(name);
}

void MainWindow::reloadCaches() {
  // request new caches for the maps new viewport
  m_cacheProvider->requestOverview(this->m_mapWidget->viewport());
}

void MainWindow::updateCaches(const CacheList &cacheList) {
  m_message = "";
  this->m_mapWidget->updateCaches(cacheList);
}

void MainWindow::statusMessage(const QString &message) {
#ifdef Q_WS_MAEMO_5
  QMaemo5InformationBox::information(this, message);
#else
  this->m_mapWidget->addBanner(message);
#endif
}

void MainWindow::displayError(const QString &message) {
  if(m_message != message) {
    statusMessage(message);
    m_message = message;
  }
}

void MainWindow::showDetail(const Cache &cache) {
  qDebug() << __FUNCTION__;

  CacheWindow *cacheWindow = new CacheWindow(cache, m_locationProvider, this);
  cacheWindow->show();
}

void MainWindow::setBusy(bool on) {
  static int busyCount = 0;

  if((!on && busyCount == 1) || ( on && busyCount <= 0)) {
#ifdef Q_WS_MAEMO_5
    setAttribute(Qt::WA_Maemo5ShowProgressIndicator, on);
#else
    m_mapWidget->setBusy(on);
#endif
  }

  if(on)                        busyCount++;
  else if(!on && busyCount > 0) busyCount--;
}
