#include "iconloader.h"
#include "filefinder.h"

#include <QDebug>
#include <QPainter>
#include <QLabel>

bool IconLoader::Pixmap::load(QSvgRenderer *renderer, const QString &name) {
  QString filename = FileFinder::find("icons", name);
  if(filename.isNull()) return false;

  renderer->load(filename);
  return true;
}

IconLoader::Pixmap::Pixmap(QSvgRenderer *renderer, 
			   const QStringList &name, int size) :
  m_pixmap(NULL) {

  if(load(renderer, name[0] + ".svg")) {
    // if no explicit size is given, then use default size of first image
    if(size > 0) 
      this->m_pixmap = new QPixmap(renderer->defaultSize().width()*size/
				   renderer->defaultSize().height(), size);
    else 
      this->m_pixmap = new QPixmap(renderer->defaultSize());
  } else 
    this->m_pixmap = new QPixmap(size, size);
 
  // fill with transparency and saved icon name
  this->m_pixmap->fill(Qt::transparent);
  this->m_name = name.join(",");
  
  QPainter painter(this->m_pixmap);
  painter.setRenderHint(QPainter::Antialiasing, true);

  // draw all given images on top of each other
  QStringList::const_iterator i;
  for(i=name.begin();i!=name.end();i++) {
    if(load(renderer, *i + ".svg"))
      renderer->render(&painter);
    else {
      // unable to load image, draw some kind of "broken" icon
      painter.setPen(QPen(QBrush(Qt::red), size/5) );
      painter.drawEllipse(QRect(QPoint(size/8,size/8), this->m_pixmap->size() - 
				QSize(size/4, size/4)));
    }
  }
}

IconLoader::Pixmap::~Pixmap() {
  if(this->m_pixmap)
    delete this->m_pixmap;
}

QPixmap* IconLoader::Pixmap::pixmap() {
  return m_pixmap;
}

bool IconLoader::Pixmap::is(const QString &name) {
  return m_name == name;
}

IconLoader::IconLoader(int size) {
  m_renderer = new QSvgRenderer();
  this->m_size = size;
}

IconLoader::~IconLoader() {
  qDeleteAll(pixmapList);
  delete m_renderer;
}

QPixmap *IconLoader::load(const QStringList &name, int size) {
  QString joinedName = name.join(",");

  // check if there's already a matching pixmap in the list
  QList<Pixmap*>::const_iterator i;
  for(i = pixmapList.constBegin(); i != pixmapList.constEnd(); ++i) 
    if( (*i)->is(joinedName) )
      return (*i)->pixmap();
  
  // nothing matching in pixmap list, create a new entry
  Pixmap *pix = new Pixmap(m_renderer, name, size?size:m_size); 
  pixmapList.append(pix);
  
  return pix->pixmap();
}

QPixmap *IconLoader::load(const QString &name, int size) {
  return load(QStringList(name), size);
}

QPixmap *IconLoader::load(const Cache &cache, int size) {
  QStringList names;
  names << cache.typeIconFile();
  QString overlay = cache.overlayIconFile();
  if(!overlay.isNull())
    names << overlay;
  
  return load(names, size);
}

QPixmap *IconLoader::load(const Container &container, int size) {
  return load(container.iconFile(), size);
}

int IconLoader::size() const {
  return m_size;
}

QWidget *IconLoader::newWidget(const QString &name, int size) {
  QLabel *widget = new QLabel;
  widget->setPixmap(*load(name, size));
  return widget;
}

QWidget *IconLoader::newWidget(const QStringList &names, int size) {
  QLabel *widget = new QLabel;
  widget->setPixmap(*load(names, size));
  return widget;
}

QWidget *IconLoader::newWidget(const Cache &cache, int size) {
  QLabel *widget = new QLabel;
  widget->setPixmap(*load(cache, size));
  return widget;
}
