// Filename: cachewindow.cpp

#include <QDebug>

#include <QVBoxLayout>
#include <QHBoxLayout>
#include <QLabel>
#include <QPainter>
#include <QKeyEvent>
#include <QPushButton>
#include <QFontInfo>

#include "htmlview.h"
#include "cachewindow.h"
#include "descriptionwindow.h"
#include "navigationwindow.h"
#include "logwindow.h"
#include "attributewindow.h"
#include "iconlabel.h"

#include "swapbox.h"

class HintWindow : public CustomWindow {
public:
  HintWindow(const Cache &cache, QWidget *parent) :
    CustomWindow("HintWindow", parent) {
    setWindowTitle(tr("Hint"));

    setCentralWidget(new HtmlView(cache.hint(), this));
  }
};

void CacheWindow::showHint() {
  HintWindow *hintWindow = new HintWindow(m_cache, this);
  hintWindow->show();
}

#define HACK

// a qpushbutton with a small icon on the right
class Button : public QPushButton {
public:
  Button(const QString &text, QWidget *parent = 0) : 
    QPushButton(text, parent), m_pixmap(NULL) {
  };

  void addIcon(IconLoader *loader, const QString &name) {
#ifndef HACK
    if(!m_pixmap) {
      QMargins margins = contentsMargins();
      qDebug() << __FUNCTION__ << 
	margins.left() << margins.right();
      margins.setRight(margins.right() + iconSize().width());
      margins.setLeft(0);
      setContentsMargins(margins);
      qDebug() << __FUNCTION__ << 
	margins.left() << margins.right();
    }
#endif
    m_pixmap = loader->load("button_" + name, iconSize().width());
#ifdef HACK
    // append some spaces to the text to shift it a little bit
    // to the left
    setText(text()+"           ");
#endif
  }

#ifndef HACK // now done by the additional spaces
  QSize sizeHint() const {
    QSize size = QPushButton::sizeHint();

    // request additional space for icon
    if(m_pixmap)
      size += QSize(m_pixmap->width(),0);

    return size;
  }
#endif

  void paintEvent(QPaintEvent *event) {
    // draw original button
    QPushButton::paintEvent(event);

    if(m_pixmap) {
      // add pixmap on top
      QRect rect = contentsRect();
      QPainter painter(this);
#ifdef HACK
      painter.drawPixmap(rect.x()+rect.width()-1.3*m_pixmap->width(), 
      			 rect.y()+(rect.height()-m_pixmap->height())/2, 
			 *m_pixmap);
#else
      painter.drawPixmap(rect.x(), 
			 rect.y()+(rect.height()-m_pixmap->height())/2, 
			 *m_pixmap);
      painter.drawPixmap(rect.x()+rect.width(), 
			 rect.y()+(rect.height()-m_pixmap->height())/2, 
			 *m_pixmap);
#endif
    }
  }

private:
  QPixmap *m_pixmap;
};

CacheWindow::CacheWindow(const Cache &cache, 
	 LocationProvider *locationProvider, QWidget *parent) : 
  CustomWindow("CacheWindow", parent),  m_cache(cache), m_iconLoader(24),
  m_locationProvider(locationProvider) {

  setWindowTitle(cache.name());

  qDebug() << __FUNCTION__ << cache;

  QWidget *vbox = new QWidget;
  QVBoxLayout *layout = new QVBoxLayout;
  layout->setContentsMargins(0,0,0,0);
  layout->setSpacing(0);
  vbox->setLayout(layout);
  setCentralWidget(vbox);

  // ---------- general info on top -------------
  
  // ------------- title with icon -------------
  layout->addWidget(new IconLabel(&m_iconLoader, cache, 
				  cache.description(), 1.5));

  // if there's a short description, then use a swapbox to place
  // the description in landscape in the left window half

  // ------------- short description -------------
  SwapBox *sbox = NULL;
  HtmlView *htmlView = NULL;
  if(cache.shortDescription().isSet()) {
    sbox = new SwapBox();

    // html view below
    htmlView = new HtmlView(cache.shortDescription());
    connect(this, SIGNAL(zoomInPressed()), htmlView, SLOT(zoomIn()));
    connect(this, SIGNAL(zoomOutPressed()), htmlView, SLOT(zoomOut()));

    sbox->addWidget(htmlView, 1);
    layout->addWidget(sbox);

    // create a new vbox for the right (button) half
    vbox = new QWidget;
    layout = new QVBoxLayout;
    layout->setContentsMargins(0,0,0,0);
    layout->setSpacing(0);
    vbox->setLayout(layout);
    sbox->addWidget(vbox);
  }

  // ------------- button to open long description -------------
  if(cache.longDescription().isSet()) {
    Button *button = new Button(tr("Description"));
    button->addIcon(&m_iconLoader, "new_window");    

    connect(button, SIGNAL(clicked()), this, SLOT(showDescription()));

    layout->addWidget(button);
  }

  // ------------- button to open navigation window -------------
  if(cache.coordinate().isValid()) {
    Button *button = new Button(tr("Navigation"));
    button->addIcon(&m_iconLoader, "new_window");    
    
    connect(button, SIGNAL(clicked()), this, SLOT(showNavigation()));
    
    layout->addWidget(button);
  }

  // ------------- button to display hint -------------
  if(cache.hint().isSet()) {
    Button *button = new Button(tr("Hint"));
    button->addIcon(&m_iconLoader, "new_window");    

    connect(button, SIGNAL(clicked()), this, SLOT(showHint()));

    layout->addWidget(button);
  }

  // ------------- button to display logs -------------
  if(cache.logs().size()) {
    Button *button = new Button(tr("Logs"));
    button->addIcon(&m_iconLoader, "new_window");    

    connect(button, SIGNAL(clicked()), this, SLOT(showLogs()));

    layout->addWidget(button);
  }

  // ------------- button to display attributes -------------
  if(cache.attributes().size()) {
    Button *button = new Button(tr("Attributes"));
    button->addIcon(&m_iconLoader, "new_window");    

    connect(button, SIGNAL(clicked()), this, SLOT(showAttributes()));

    layout->addWidget(button);
  }

  QWidget *stretchWidget = new QWidget;
  if(sbox) sbox->setPortraitOnly(stretchWidget);
  layout->addWidget(stretchWidget, 1);
}

CacheWindow::~CacheWindow() {
  qDebug() << __FUNCTION__;
}

void CacheWindow::showDescription() {
  DescriptionWindow *descriptionWindow = 
    new DescriptionWindow(m_cache, this);

  descriptionWindow->show();
}

void CacheWindow::showNavigation() {
  NavigationWindow *navigationWindow = 
    new NavigationWindow(m_cache, m_locationProvider, this);

  navigationWindow->show();
}

void CacheWindow::showLogs() {
  LogWindow *logWindow = 
    new LogWindow(m_cache.logs(), this);

  logWindow->show();
}

void CacheWindow::showAttributes() {
  AttributeWindow *attributeWindow = 
    new AttributeWindow(m_cache.attributes(), this);

  attributeWindow->show();
}
