// Filename: mainwindow.cpp

#include <QDebug>
#include <QMessageBox>

#include "mainwindow.h"
#include "mapwidget.h"
#include "mapdialog.h"
#include "iconloader.h"  
#include "cachewindow.h"

#include <QIcon>
#include <QSettings>

#ifndef Q_WS_MAEMO_5
#include <QStatusBar>
#include <QProgressBar>
#include <QVBoxLayout>
#else
#include <QMaemo5InformationBox>
#endif

void MainWindow::createMenu() {
  QMenu *menu;
  QAction *action;

#ifndef Q_WS_MAEMO_5
  menu = new QMenu(tr("&File"), this);
  action = new QAction(tr("E&xit"), this);
  action->setShortcuts(QKeySequence::Quit);
  connect(action, SIGNAL(triggered()), this, SLOT(close()));
  menu->addAction(action);
  menuBar()->addMenu(menu);
#endif
  
  menu = new QMenu(tr("&Map"), this);
  action = new QAction(tr("Map &Settings"), this);
  connect(action, SIGNAL(triggered()), this, SLOT(mapdialog()));
  menu->addAction(action);
  menuBar()->addMenu(menu);

  menu = new QMenu(tr("&Plugins"), this);
  m_cacheProvider->createMenu(menu);
  menuBar()->addMenu(menu);

  menu = new QMenu(tr("&Help"), this);
  action = new QAction(tr("&About"), this);
  connect(action, SIGNAL(triggered()), this, SLOT(about()));
  menu->addAction(action);
  menuBar()->addMenu(menu);
}

void MainWindow::mapdialog() {
  qDebug() << __FUNCTION__;

  MapDialog dialog(m_mapWidget, this);
  dialog.exec(); 
}

void MainWindow::about() {
  QMessageBox::about(this, tr("About CacheMe"),
	     tr("CacheMe") + " " + tr("Version") + " " + VERSION + "\n" + 
	     tr("Build date:") + " " + __DATE__ + "\n" +
	     tr("(c) 2011 by Till Harbaum <till@harbaum.org>") + "\n\n" +
	     tr("CacheMe is an online geocaching application.")
	     );
}


MainWindow::MainWindow(LocationProvider *locationProvider, QWidget *parent) : 
  CustomWindow("MainWindow", parent), m_locationProvider(locationProvider) {

  qDebug() << __FUNCTION__;
  m_cacheProvider = new CacheProvider();

  setWindowTitle(tr("CacheMe"));

  IconLoader iconLoader(48);
  QPixmap *icon = iconLoader.load(APPNAME);
  if(icon) setWindowIcon(QIcon(*icon));

  createMenu();

  // search for service providers
  QStringList list = QGeoServiceProvider::availableServiceProviders();
  for(int i=0;i<list.size();i++)
    qDebug() << __FUNCTION__ << "Service provider: " << list[i];

  // there needs to be at least one geo service provider
  Q_ASSERT(list.size() > 0);

  QSettings settings;
  settings.beginGroup("Map");
  QString serviceProviderStr = settings.value("Service", "nokia").toString();
  settings.endGroup();

  this->m_serviceProvider = new QGeoServiceProvider(serviceProviderStr);
  if(!this->m_serviceProvider->mappingManager())
    this->m_serviceProvider = new QGeoServiceProvider("nokia");

  this->m_mapWidget = new MapWidget(this->m_serviceProvider->mappingManager());

  // make sure cache updates are processed
  QObject::connect(this->m_locationProvider, SIGNAL(positionUpdated(const QGeoPositionInfo &)), 
		   this->m_mapWidget, SLOT(positionUpdated(const QGeoPositionInfo &)));

  QObject::connect(this->m_cacheProvider, SIGNAL(replyOverview(const CacheList &)), 
		   this, SLOT(updateCaches(const CacheList &)));

  QObject::connect(this->m_cacheProvider, SIGNAL(replyInfo(const Cache &)), 
		   this->m_mapWidget, SLOT(showBubble(const Cache &)));

  QObject::connect(this->m_cacheProvider, SIGNAL(replyDetail(const Cache &)), 
		   this, SLOT(showDetail(const Cache &)));

  QObject::connect(this->m_cacheProvider, SIGNAL(replyError(const QString &)), 
		   this, SLOT(displayError(const QString &)));

  QObject::connect(this->m_cacheProvider, SIGNAL(settingsChanged()),
		   this, SLOT(reloadCaches()));

  QObject::connect(this->m_cacheProvider, SIGNAL(busy(bool)),
		   this, SLOT(setBusy(bool)));

  QObject::connect(this->m_mapWidget, SIGNAL(mapChanged()),
		   this, SLOT(reloadCaches()));

  QObject::connect(this->m_mapWidget, SIGNAL(showMessage(const QString &)),
		   this, SLOT(statusMessage(const QString &)));

  QObject::connect(this, SIGNAL(zoomInPressed()),
		   this->m_mapWidget, SLOT(zoomIn()));

  QObject::connect(this, SIGNAL(zoomOutPressed()),
		   this->m_mapWidget, SLOT(zoomOut()));

  // connect to map to handle clicks on cache icons ...
  QObject::connect(this->m_mapWidget, SIGNAL(cacheClicked(const QString &)),
		   this, SLOT(selectCache(const QString &)));

  // ... and the detail button
  QObject::connect(this->m_mapWidget, SIGNAL(detailClicked(const QString &)),
		   this, SLOT(detailCache(const QString &)));

  // build a graphics scene based on the mapwidget
  QGraphicsScene *scene = new QGraphicsScene(this);
  m_view = new QGraphicsView( scene );
  m_view->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
  m_view->setVerticalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
  m_view->setVisible(true);
  m_view->setInteractive(true);
  scene->addItem(m_mapWidget);
  setCentralWidget(m_view);

#ifndef Q_WS_MAEMO_5
  // on maemo5 we use QMaemo5InformationBox
  setStatusBar(new QStatusBar(this));
#endif

  m_cacheProvider->start();
}

MainWindow::~MainWindow() {
  qDebug() << __FUNCTION__;

  delete m_serviceProvider;
}

void MainWindow::resizeEvent(QResizeEvent*) {
  m_view->setSceneRect(QRectF(QPointF(0.0, 0.0), m_view->size()));
  m_mapWidget->resize(m_view->size());
}

void MainWindow::selectCache(const QString &name) {
  // ask CacheProvider for info about this cache
  this->m_cacheProvider->requestInfo(name);
}

void MainWindow::detailCache(const QString &name) {
  // ask CacheProvider for details about this cache
  this->m_cacheProvider->requestDetail(name);
}

void MainWindow::reloadCaches() {
  // request new caches for the maps new viewport
  m_cacheProvider->requestOverview(this->m_mapWidget->viewport());
}

void MainWindow::updateCaches(const CacheList &cacheList) {
  m_message = "";
  this->m_mapWidget->updateCaches(cacheList);
}

void MainWindow::statusMessage(const QString &message) {
#ifdef Q_WS_MAEMO_5
  QMaemo5InformationBox::information(this, message);
#else
  statusBar()->setProperty("error", false);  
  statusBar()->showMessage(message, 5000);
#endif
}

void MainWindow::displayError(const QString &message) {
  if(m_message != message) {
    statusMessage(message);
    m_message = message;
  }
}

void MainWindow::showDetail(const Cache &cache) {
  qDebug() << __FUNCTION__;

  CacheWindow *cacheWindow = new CacheWindow(cache, this);
  cacheWindow->show();
}

void MainWindow::setBusy(bool on) {
  static int busyCount = 0;

#ifdef Q_WS_MAEMO_5
  if((!on && busyCount  > 0) || ( on && busyCount <= 0))
    setAttribute(Qt::WA_Maemo5ShowProgressIndicator, on);
#else
  static QProgressBar *progressBar = NULL;
  if(on && busyCount <= 0) {
    if(!progressBar) {
      progressBar = new QProgressBar();
      progressBar->setMinimum(0);
      progressBar->setMaximum(0);
      progressBar->setValue(25);
      progressBar->setMaximumWidth(128);
      progressBar->setMaximumHeight(16);
      statusBar()->addPermanentWidget(progressBar);
    } else {
      statusBar()->addPermanentWidget(progressBar);
      progressBar->reset();
      progressBar->show();
    }
  } else if(!on && busyCount > 0) {
    statusBar()->removeWidget(progressBar);
  }
#endif

  if(on)                        busyCount++;
  else if(!on && busyCount > 0) busyCount--;
}

