/*
    watchdogs.c - Keep alive watchdogs
    Copyright (C) 2010-2011  Pali Rohár <pali.rohar@gmail.com>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

*/

#include <errno.h>
#include <string.h>
#include <stdlib.h>
#include <fcntl.h>
#include <sched.h>
#include <signal.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <sys/mman.h>
#include <sys/wait.h>
#include <linux/watchdog.h>

#ifdef WITH_LIBCAL
#include <cal.h>
#endif

/* structure from DSME */
struct wd_t {
	const char * file;      /* pathname of the watchdog device */
	int period;             /* watchdog timeout (s); 0 for keeping the default */
	const char * cal_flag;  /* R&D flag in cal that disables the watchdog */
};

/* the table of HW watchdogs; notice that their order matters! */
static const struct wd_t wd[] = {
	/* path,               timeout (s), disabling R&D flag */
	{  "/dev/twl4030_wdt", 30,          "no-ext-wd"  }, /* twl (ext) wd */
	{  "/dev/watchdog",    15,          "no-omap-wd" }  /* omap wd      */
};

#define WD_COUNT (sizeof(wd) / sizeof(wd[0]))

/* watchdog file descriptors */
static int wd_fd[WD_COUNT];

static sig_atomic_t running = 1;

static void sighandler(int param) {

	running = 0;
	(void)param;

}

#ifdef WITH_LIBCAL

static char ** get_cal_flags(void) {

	unsigned long int len;
	struct cal * cal;
	void * ptr;
	int ret;

	unsigned int i;
	unsigned int count;
	char * str;
	char * begin;
	char * end;
	char ** flags;

	if ( cal_init(&cal) < 0 )
		return NULL;

	ret = cal_read_block(cal, "r&d_mode", &ptr, &len, CAL_FLAG_USER);

	if ( ret < 0 || ! ptr )
		return NULL;

	str = calloc(len+1, 1);
	memcpy(str, ptr, len);
	len = strlen(str);

	cal_finish(cal);

	count = 0;

	for ( i = 0; i < len; ++i )
		if ( str[i] == ',' )
			++count;

	flags = malloc((count + 1) * sizeof(char *));

	begin = str;

	for ( i = 0; i < count; ++i ) {

		end = strchr(begin, ',');

		if ( ! end )
			end = begin + strlen(begin);

		flags[i] = malloc(end-begin+1);
		strncpy(flags[i], begin, end-begin);

		begin = end + 1;

	}

	free(str);

	return flags;

}

#endif

static unsigned int init(void) {

	unsigned int ret = 0;
	unsigned int i;

	for ( i = 0; i < WD_COUNT; ++i ) {

		int skip = 0;

#ifdef WITH_LIBCAL

		char ** cal_flags = get_cal_flags();
		char ** cal_ptr;

		if ( wd[i].cal_flag ) {

			cal_ptr = cal_flags;

			while ( *cal_ptr ) {

				if ( strcmp(*cal_ptr, wd[i].cal_flag) == 0 ) {

					skip = 1;
					break;

				}

				++cal_ptr;

			}

		}
#endif

		if ( ! skip )
			wd_fd[i] = open(wd[i].file, O_RDWR);
		else
			wd_fd[i] = -1;

		if ( wd_fd[i] > 0 ) {

			int period = wd[i].period;

			if ( ioctl(wd_fd[i], WDIOC_SETTIMEOUT, &period) == 0 )
				++ret;

		}

	}

	return ret;

}

static void loop(unsigned int count) {

	unsigned int i;

	while ( running && count > 0 ) {

		for ( i = 0; i < WD_COUNT; ++i ) {

			if ( wd_fd[i] > 0 ) {

				int status = 0;

				if ( ioctl(wd_fd[i], WDIOC_KEEPALIVE, &status) != 0 ) {

					close(wd_fd[i]);
					wd_fd[i] = -1;
					--count;

				}

			}

		}

		sleep(5);

	}

}

static void quit(void) {

	unsigned int i;

	for ( i = 0; i < WD_COUNT; ++i ) {

		if ( wd_fd[i] > 0 ) {

			close(wd_fd[i]);
			wd_fd[i] = -1;

		}

	}

}

int main(void) {

	struct sched_param sp;
	unsigned int count;
	int lock;
	int status;
	pid_t pid;

	if ( signal(SIGTERM, sighandler) == SIG_ERR )
		return 1;

	if ( signal(SIGINT, sighandler) == SIG_ERR )
		return 1;

	count = init();

	if ( count == 0 )
		return 1;

	lock = mlockall(MCL_CURRENT | MCL_FUTURE);

	if ( lock != 0 )
		return errno;

	sp.sched_priority = 1;

	if ( sched_setscheduler(0, SCHED_RR, &sp) != 0 )
		return errno;

	pid = fork();

	if ( pid < 0 )
		return errno;

	if ( pid > 0 ) {

		pid_t w = waitpid(pid, &status, WUNTRACED | WCONTINUED);

		if ( w < 0 )
			return 1;

		if ( WIFEXITED(status) )
			return WEXITSTATUS(status);

		return 1;

	}

	pid = fork();

	if ( pid < 0 )
		_exit(errno);
	else if ( pid > 0 )
		_exit(0);

	setsid();

	close(0);
	close(1);
	close(2);

	loop(count);

	munlockall();

	quit();

	return 0;

}
