#include "readwidget.h"

#include <QVBoxLayout>
#include <QLabel>
#include <QProgressBar>
#include <QPushButton>
#include <QLineEdit>
#include <QTime>
#include <QMessageBox>
#include <QFileInfo>
#include <QDebug>
#include <QCloseEvent>

#include "audiobook.h"

VolumeDialog::VolumeDialog(Phonon::AudioOutput* aAudioOutput,QWidget* aParent ):QDialog(aParent)
{
    mVolumeSlider = new Phonon::VolumeSlider(this);    
    mVolumeSlider->setAudioOutput(aAudioOutput);   
    QVBoxLayout* mainLayout = new QVBoxLayout(this);
    mainLayout->addWidget(mVolumeSlider);
}

ReadWidget::ReadWidget( QWidget* aParent ): QWidget(aParent), mCurrentBook(0),mMediaObject(0)
{
    mAudioOutput = new Phonon::AudioOutput(Phonon::MusicCategory, this);
    //setting default volume to 50%
    mAudioOutput->setVolume(0.5);
    mAudioOutput->setObjectName("audio");
    mMediaObject = new Phonon::MediaObject(this);
    mMediaObject->setObjectName("media");
    mMediaObject->setTickInterval(1000);   

    createConnection();

    Phonon::createPath(mMediaObject, mAudioOutput);

    setupUi();
}

void ReadWidget::createConnection()
{
    connect(mMediaObject,SIGNAL(tick(qint64)),this,SLOT(readProgress(qint64)));
    connect(mMediaObject,SIGNAL(totalTimeChanged(qint64)),this,SLOT(totalTimeChanged(qint64)));
    connect(mMediaObject, SIGNAL(stateChanged(Phonon::State, Phonon::State)),
             this, SLOT(stateChanged(Phonon::State, Phonon::State)));
    connect(mMediaObject,SIGNAL(finished()),this,SLOT(finished()));
    connect(mMediaObject,SIGNAL(metaDataChanged()),this,SLOT(metaDataChanged()));
}

ReadWidget::~ReadWidget()
{    
}

void ReadWidget::setupUi()
{
    mAlbumName = new QLabel("",this);
    mBookName = new QLabel("",this);
    mArtist = new QLabel("",this);

    mReadProgress = new Phonon::SeekSlider(this);
    mReadProgress->setMediaObject(mMediaObject);

    mProgress = new QLabel("",this);
    mTotal = new QLabel("",this);

    mPlayButton = new QPushButton(">",this);
    mPlayButton->setEnabled(false);
    connect(mPlayButton,SIGNAL(clicked()),mMediaObject,SLOT(play()));

    mPauseButton = new QPushButton("||",this);
    mPauseButton->setEnabled(false);
    connect(mPauseButton,SIGNAL(clicked()),mMediaObject,SLOT(pause()));

    mStopButton = new QPushButton("X",this);
    mStopButton->setEnabled(false);
    connect(mStopButton,SIGNAL(clicked()),mMediaObject,SLOT(stop()));

    mSoundButton = new QPushButton("<))",this);
    connect(mSoundButton,SIGNAL(clicked()),this,SLOT(setVolume()));

    QHBoxLayout* progressLabelLayout = new QHBoxLayout;
    progressLabelLayout->addWidget(mProgress);
    progressLabelLayout->addStretch();
    progressLabelLayout->addWidget(mTotal);

    QVBoxLayout* progressSliderLayout = new QVBoxLayout;
    progressSliderLayout->addWidget(mReadProgress);
    progressSliderLayout->addLayout(progressLabelLayout);
    progressSliderLayout->addStretch();

    QHBoxLayout* controlLayout = new QHBoxLayout;
    controlLayout->addWidget(mPlayButton);
    controlLayout->addWidget(mPauseButton);
    controlLayout->addWidget(mStopButton);
    controlLayout->addWidget(mSoundButton);

    QVBoxLayout* mainLayout = new QVBoxLayout;
    mainLayout->addWidget(  mAlbumName );
    mainLayout->addWidget( mBookName );
    mainLayout->addWidget( mArtist );
    mainLayout->addStretch();
    mainLayout->addLayout( progressSliderLayout);    
    mainLayout->addStretch();
    mainLayout->addLayout( controlLayout );

    this->setLayout(mainLayout);
}

void ReadWidget::readBook( AudioBook* aBook )
{    
    if( aBook == 0 ) {
        mCurrentBook = 0;
        return;
    }

    qDebug() << "ReadWidget::readBook() "<< aBook->progress();        
    mCurrentBook = aBook;
    mCurrentSourceIndex = mCurrentBook->currentSource();

    mAudioOutput->setVolume( mCurrentBook->volume() );
    mMediaObject->setCurrentSource( aBook->source().at(mCurrentSourceIndex) );
    mMediaObject->seek( mCurrentBook->progress() );
    mProgress->setText(timeToString(mCurrentBook->progress()));
    mPlayButton->setEnabled(true);
    mPauseButton->setEnabled(false);
    mStopButton->setEnabled(true);
    mMediaObject->blockSignals(false);
    this->update();
}

void ReadWidget::closeEvent(QCloseEvent *event)
{
    qDebug() << "ReadWidget::closeEvent()";
    mMediaObject->pause();
    event->accept();
    mMediaObject->blockSignals(true);
    emit bookClosed( mCurrentBook );
}

void ReadWidget::totalTimeChanged(qint64 aTotalTime)
{    
    mTotal->setText(timeToString(aTotalTime));    
}

void ReadWidget::readProgress(qint64 aCurrentPosition )
{        
        qDebug() << " readProgress::" << aCurrentPosition;
        // change progress only in playing state
        if( mPlayButton->isEnabled() == false ) {
            mProgress->setText(timeToString(aCurrentPosition));
            mCurrentBook->setProgress( aCurrentPosition );
        }
}

QString ReadWidget::timeToString(qint64 aTime)
{
    QTime time;
    time = time.addMSecs(aTime);
    return time.toString(Qt::TextDate);
}

void ReadWidget::stateChanged(Phonon::State aNewState, Phonon::State /*aOldState */)
 {
    qDebug() << "ReadWidget::stateChanged()";
     switch (aNewState) {

     case Phonon::PlayingState:{
         qDebug() << "PlayingState";
         mMediaObject->seek( mCurrentBook->progress() );
         mPlayButton->setEnabled(false);
         mPauseButton->setEnabled(true);
         mStopButton->setEnabled(true);
         this->update();
         break;
     }
     case Phonon::StoppedState:{
         qDebug() << "StoppedState";
         mPlayButton->setEnabled(true);
         mPauseButton->setEnabled(false);
         mStopButton->setEnabled(false);
         this->update();
         break;
     }
     case Phonon::PausedState:{
         qDebug() << "PausedState";
         mPlayButton->setEnabled(true);
         mPauseButton->setEnabled(false);
         mStopButton->setEnabled(true);
         this->update();
         emit bookPaused( mCurrentBook );
         break;
     }
     default:
         break;
     }
 }

void ReadWidget::finished ()
{
    mCurrentSourceIndex++;
    if( mCurrentSourceIndex >= mCurrentBook->source().count() ) {
        mCurrentSourceIndex = 0;
    }

    mCurrentBook->setProgress(0);
    mCurrentBook->setCurrentSource( mCurrentSourceIndex );
    mMediaObject->setCurrentSource( mCurrentBook->source().at(mCurrentSourceIndex) );            
    mMediaObject->play();
}

void ReadWidget::setVolume()
{
    VolumeDialog dialog(mAudioOutput);
    dialog.exec();
    mCurrentBook->setVolume( mAudioOutput->volume() );
}

void ReadWidget::metaDataChanged()
{
    QMultiMap<QString, QString> metaData = mMediaObject->metaData();
    qDebug() << "metadata:" << metaData;
    QString albumName = metaData.value("ALBUM","");
    mAlbumName->setText( "<h2>" + albumName+ "</h2>");

    QString bookName = metaData.value("TITLE",mCurrentBook->bookName());
    mBookName->setText( bookName);

    QString artistName = metaData.value("ARTIST","");
    mArtist->setText(artistName);
}
