#! /usr/bin/env python

""" Copyright 2010 Edwin Marshall 

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
"""

import sys
from os import makedirs, path
from cmd import Cmd
from subprocess import Popen, PIPE
from threading import Timer

from config import Config
from utils import Catalogues

STDOUT, STDERR = range(2)
config = Config()

class Aptly(Cmd):
    def __init__(self):
        Cmd.__init__(self)
        self.prompt = '<aptly> '
        self.intro = ("Aptly 2010.5.15\n"
                      "Type 'help' for more information or 'quit' to exit.")
        self.known_commands = ['help', 'install', 'update', 'remove', 'list', 
			       'prompt', 'set', 'quit']
        self.commands = []
        self.ready = False

        def get_catalogues():
            self.catalogues = Catalogues()
            self.ready = True

        Timer(0, get_catalogues).start()

    def emptyline(self):
        pass

    def precmd(self, line):
        indexes = []

        for command in self.known_commands:
            if command in line:
                index = line.find(command, 0)
                while index >= 0:
                    indexes.append(index)
                    index = line.find(command, index+1)

        indexes.sort()

        for index in xrange(len(indexes) - 1):
            self.commands.append(line[indexes[index]:indexes[index+1]].strip())

        if len(indexes) > 0 and line[indexes[-1]:].strip() not in self.commands:
            self.commands.append(line[indexes[-1]:].strip())

        for i in xrange(len(self.commands)):
            if i > 0 and self.commands[i-1] in self.known_commands:
                self.commands[i-1] += ' ' + self.commands[i]
                self.commands.pop(i)

        for command in self.commands:
            self.onecmd(command)

        self.commands = []  

        return '' if indexes else line

    def complete_install(self, text, line, start_index, end_index):
        ls = Popen('ls', stdout=PIPE).communicate()[STDOUT].split()
        debs = [item for item in ls if item[-4:] == '.deb']

        if ' '.join(debs) not in line:
            if text:
                return [deb for deb in debs if deb.startswith(text)]
            else:
                return debs
        else:
            return []

    def do_install(self, packages):
        packages = packages.replace(',', ' ').split()
        remote = []
        local = []

        if packages:
            # separate local and remote packages
            for package in packages:
                if package[-4:] == '.deb':
                    local.append(package)
                else:
                    remote.append(package)

            # find dependencies for all local packages
            for deb in local:
                p = Popen(['dpkg', '-I', deb], stdout=PIPE).communicate()
                deps = None

                for line in p[STDOUT].split('\n'):
                    if 'Depends' in line:
                        deps = line.split('Depends: ')[1].split(', ')
                        deps = ' '.join([dep.split()[0] for dep in deps])

                self.do_install(deps)

                Popen(['dpkg', '-i', deb]).communicate()

            Popen(' '.join(['apt-get', 'install', config.main['options']] + remote),
                  shell=True).communicate()
        else:
            packages = raw_input('What package(s) would you like to install?\n'
                                 '===> ')
            if packages and packages != '':
                self.do_install(packages)

    def help_install(self):
        print ("install [package1 package2 package3]\n"
               "\tInstalls the named packages. If packages are omitted, aptly "
               "will prompt you for package names. \n"
               "\tYou may exit this prompt by using the package name 'cancel'.")

    def do_remove(self, packages):
        packages = packages.replace(',', ' ').split()

        if packages:
            Popen(['apt-get', 'remove'] + packages).communicate()
        else:
            packages = raw_input('What package(s) would you like to remove?\n'
                                 '===> ')
            if packages and packages != '':
                self.do_remove(packages)

    def help_remove(self):
        print ("remove [package1 package2 package3]\n"
               "\tRemoves the named packages. If packages are omitted, aptly "
               "will prompt you for package names. \n"
               "\tYou may exit this prompt by using the package name 'cancel'.")

    def do_update(self, line):
        Popen(['apt-get', 'update']).communicate()

    def do_upgrade(self, line):
        if config.main['ask-first']:
            config.main['update-first'] = raw_input('Would you like to update '
                                                    'first? (Y/n)\n'
                                                    '===> ')

        if config.main['update-first']:
            self.do_update(None)

        Popen(['apt-get', 'upgrade']).communicate()

    def do_list(self, packages):
	print 'line 168'
        packages = packages.replace(',', ' ').split()

        for package in packages:
            print "Files installed by '%s': " % package
            Popen(['dpkg', '-L', package]).communicate()

    def do_search(self, phrase):
        phrase = phrase.split() 
        Popen(['apt-cache', 'search'] + phrase).communicate()
        
    def help_list(self):
        print ("list [package1 package2]\n"
               "\tLists all files installed by named packages.")

    def do_set(self, options):
        def get_cache_directory():
            cache_dir = raw_input('Which directory would you like to use '
                                     'as your cache directory?\n'
                                      '===> ')
            if cache_dir == '':
                print ('Using default cache directory.')
                return False

            return path.abspath(cache_dir)

        # convert options into a dictionary of values delimted by commas from
        # the original options variable
        options = dict([(option.split()[0], 
                         ' '.join(option.split()[1:]).strip()) 
                        for option in options.split(',')])

        for key in options.keys():
            if key == 'cache':
                #TODO: see about moving this outside of this enclosure
                if '-o' not in self.options:
                    config.main['options'].insert(0, '-o')

                if options[key] == '':
                    options[key] = get_cache_directory()
                    if not options[key]:
                        break
                else:
                    options[key] = path.abspath(options[key])

                try:
                    makedirs(path.abspath(options[key]))
                except OSError, (errno, _):
                    if errno == 17:
                        pass
                    else:
                        print '%s could not be created.' % options[key]
                        options[key] = get_cache_directory()
                        if not options[key]:
                            break

                try:
                    partial_dir = path.join(options[key], 'partial')
                    makedirs(partial_dir)
                except OSError:
                    pass

                config.main['options'].append("dir::cache::archives='%s'" % options[key])
            elif key == 'ask-first':
                self.config.main[key] = options[key]
    
    def help_set(self):
        print ("set miscellaneous options. Currently available options:\n"
               "\tcache <directory> - directory to download deb files\n"
               "\task-first (Yes/No - whether or not to ask about certain "
               "actions. For example, if No, you will not be asked whether "
               "or not you want to update before upgrading.")

    def do_prompt(self, new_prompt):
        if new_prompt:
            self.prompt = new_prompt
            if self.prompt[-1] != ' ':
                self.prompt += ' '
        else:
            self.prompt = '<aptly> '

    def help_prompt(self):
        print ("prompt new_prompt\n"
               "\tChange the default prompt (<aplty>) to new_prompt\n"
               "\tIf new_prompt is not given, the default prompt is used.")

    def do_quit(self, line):
        sys.exit(1)

    def do_exit(self, line):
        self.do_quit(None)


if __name__ == '__main__':
    if len(sys.argv) > 1:
        Aptly().precmd(' '.join(sys.argv[1:]))
    else:
        Aptly().cmdloop()
