#! /usr/bin/env python

""" Copyright 2010 Edwin Marshall 

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
"""

import sys
from cmd import Cmd
from subprocess import call

class Aptly(Cmd):
    def __init__(self):
        Cmd.__init__(self)
        self.prompt = '<aptly> '
        self.intro = ("Aptly 2.0a1\n"
                      "Type 'help' for more information or 'quit' to exit.")
        self.options = []
        self.known_commands = ['install', 'remove', 'prompt', 'set', 'quit']

    def emptyline(self):
        pass

    def precmd(self, line):
        commands = []
        indexes = []

        for command in self.known_commands:
            if command in line:
                index = line.find(command, 0)
                while index >= 0:
                    indexes.append(index)
                    index = line.find(command, index+1)

        indexes.sort()

        for index in xrange(len(indexes) - 1):
            commands.append(line[indexes[index]:indexes[index+1]].strip())

        if len(indexes) > 0:
            commands.append(line[indexes[-1]:].strip())

        for command in commands:
            self.onecmd(command)

        return ''

    def do_install(self, packages):
        packages = packages.replace(',', ' ').split()

        if packages:
            self.aptget('install', packages, self.options)
        else:
            packages = raw_input('What package(s) would you like to install?\n'
                                 '===> ')
            if packages and packages != 'cancel':
                self.do_install(packages)

    def help_install(self):
        print ("install [package1 package2 package3]\n"
               "  Installs the named packages. If packages are omitted, aptly "
               "will prompt you for package names. \n"
               "  You may exit this prompt by using the package name 'cancel'.")

    def do_remove(self, packages):
        packages = packages.replace(',', ' ').split()

        if packages:
            call(['apt-get', 'remove'] + packages)
        else:
            packages = raw_input('What package(s) would you like to remove?\n'
                                 '===> ')
            if packages and packages != 'cancel':
                self.do_install(packages)

    def help_remove(self):
        print ("remove [package1 package2 package3]\n"
               "  Removes the named packages. If packages are omitted, aptly "
               "will prompt you for package names. \n"
               "  You may exit this prompt by using the package name 'cancel'.")

    def do_set(self, options):
        options = dict([option.split() for option in options.split(',')])

        for key in options.keys():
            if key == 'cache':
                if '-o' not in self.options:
                    self.options.insert(0, '-o')

                self.options.append("dir::cache::archives='%s'" % options[key])

    def do_prompt(self, new_prompt):
        if new_prompt:
            self.prompt = new_prompt
            if self.prompt[-1] != ' ':
                self.prompt += ' '
        else:
            self.prompt = '<aptly> '

    def help_prompt(self):
        print ("prompt new_prompt\n"
               "  Change the default prompt (<aplty>) to new_prompt\n"
               "  If new_prompt is not given, the default prompt is used.")

    def do_quit(self, line):
        sys.exit(1)

    def aptget(self, operation='install', packages=None, options=None):
        if operation in ['install', 'remove']:
            call(['apt-get', operation] + packages + options)
        else:
            print "The command %s is unsupported." % operation

if __name__ == '__main__':
    if len(sys.argv) > 1:
        Aptly().onecmd(' '.join(sys.argv[1:]))
    else:
        Aptly().cmdloop()
