#include <gtk/gtk.h>
#include <hildon/hildon.h>

#include <libhildondesktop/libhildondesktop.h>

#include "app-search-widget.h"
#include <libintl.h>
#include <locale.h>
#include <string.h>

HD_DEFINE_PLUGIN_MODULE (AppSearchHomePlugin, app_search_home_plugin, HD_TYPE_HOME_PLUGIN_ITEM)

#define CONFIG_PATH "/usr/share/applications/hildon/"
#define GETTEXT_PACKAGE "maemo-af-desktop"
#define APP_TITLE "App Search"
#define APP_MGR_DBUS_NAME "com.nokia.HildonDesktop.AppMgr"
#define APP_MGR_DBUS_PATH "/com/nokia/HildonDesktop/AppMgr"
#define APP_MGR_LAUNCH_METHODE "LaunchApplication"
#define ICON_PATH "/usr/share/pixmaps/app-search-widget/app-search-widget.png"
#define DESKTOP_ENTRY_CONFIG_GROUP "Desktop Entry" 
#define TEXTDOMAIN_KEY "X-Text-Domain"
#define NAME_KEY "Name"
#define ICON_KEY "Icon"
#define NODISPLAY_KEY "NoDisplay"

static
gchar* 
_retrieve_app_name(GKeyFile* key_file)
{
  if(g_key_file_has_key(key_file, DESKTOP_ENTRY_CONFIG_GROUP, TEXTDOMAIN_KEY, NULL) &&
     g_key_file_has_key(key_file, DESKTOP_ENTRY_CONFIG_GROUP, NAME_KEY, NULL))
  {
    gchar* text_domain = g_key_file_get_string(key_file, DESKTOP_ENTRY_CONFIG_GROUP, TEXTDOMAIN_KEY, NULL);
    gchar* app_name_key = g_key_file_get_string(key_file, DESKTOP_ENTRY_CONFIG_GROUP, NAME_KEY, NULL);
    const gchar* app_name = dgettext(text_domain, app_name_key);
    
    g_free(text_domain);
    g_free(app_name_key);
    
    return g_strdup(app_name);
  }
  else if(g_key_file_has_key(key_file, DESKTOP_ENTRY_CONFIG_GROUP, NAME_KEY, NULL))
  {
    gchar* app_name = g_key_file_get_string(key_file, DESKTOP_ENTRY_CONFIG_GROUP, NAME_KEY, NULL);
    gchar* name = g_strdup(dgettext(GETTEXT_PACKAGE, app_name));
    g_free(app_name);
    return name;
  }
  return NULL;
}

static GdkPixbuf*
_retrieve_icon(GKeyFile* key_file)
{
  GdkPixbuf* icon = NULL;
  if(g_key_file_has_key(key_file, DESKTOP_ENTRY_CONFIG_GROUP, ICON_KEY, NULL))
  {

    gchar* icon_name = 
      g_key_file_get_string(key_file, DESKTOP_ENTRY_CONFIG_GROUP, ICON_KEY, NULL);
    if(g_path_is_absolute(icon_name))
    {
      icon = gdk_pixbuf_new_from_file_at_size(icon_name,
					      HILDON_ICON_PIXEL_SIZE_FINGER,
					      HILDON_ICON_PIXEL_SIZE_FINGER, NULL);
    }
    else
    {
      icon = gtk_icon_theme_load_icon(gtk_icon_theme_get_default(),
				      icon_name,
				      HILDON_ICON_PIXEL_SIZE_FINGER,
				      GTK_ICON_LOOKUP_NO_SVG, NULL);
    }
    g_free(icon_name);
  }
  if(icon == NULL)
  {
    icon = gtk_icon_theme_load_icon(gtk_icon_theme_get_default(),
				    "tasklaunch_default_application",
				    HILDON_ICON_PIXEL_SIZE_FINGER,
				    GTK_ICON_LOOKUP_NO_SVG, NULL);      
  }
  return icon;
}

static gboolean
_shouldnt_be_visible(GKeyFile* key_file)
{
  if(g_key_file_has_key(key_file, DESKTOP_ENTRY_CONFIG_GROUP, NODISPLAY_KEY, NULL))
  {
    gboolean no_display = g_key_file_get_boolean(key_file, DESKTOP_ENTRY_CONFIG_GROUP, NODISPLAY_KEY, NULL);
    return no_display;
  }
  return FALSE;
}

static void
_add_app_from_desktop_file_to_liststore(const gchar* file_name, GtkListStore* list_store)
{
  GKeyFile *key_file = g_key_file_new();
  if(g_key_file_load_from_file(key_file, file_name, G_KEY_FILE_KEEP_COMMENTS, NULL))
  {
    if(_shouldnt_be_visible(key_file))
    {
    }
    else
    {
      gchar* app_name = _retrieve_app_name(key_file);
      GdkPixbuf* icon = _retrieve_icon(key_file);
      gchar* desktop_file_name = g_path_get_basename(file_name);
      GtkTreeIter iter;
      gtk_list_store_append(list_store, &iter);
      gtk_list_store_set(list_store, &iter,
			 0, icon,
			 1, app_name,
			 2, desktop_file_name,
			 -1);
      g_free(desktop_file_name);
      g_free(app_name);
      g_object_unref(icon);
    }
    g_key_file_free(key_file);
  }
}

static void
_fill_app_list_liststore_from_app_config_path(GtkListStore* list_store)
{
  GError* error = NULL;
  GDir* directory = g_dir_open(CONFIG_PATH, 0, &error);

  if(error)
  {
    g_warning("error on open directory %s\n", CONFIG_PATH);
    g_error_free(error);
    error = NULL;
  }
  else
  {
    const gchar* file_name = NULL;
    while((file_name = g_dir_read_name(directory))!=NULL)
    {
      gchar* full_path = g_build_filename(CONFIG_PATH, file_name, NULL);
      _add_app_from_desktop_file_to_liststore(full_path, list_store);
      g_free(full_path);
    }
    g_dir_close(directory);
  }
}

/*
static gboolean 
search_app_func(GtkTreeModel *model,
		gint column,
		const gchar *key,
		GtkTreeIter *iter,
		gpointer search_data)
{
  gchar* app_name = NULL;
  gtk_tree_model_get(model, iter,
		     1, &app_name,
		     -1);
  if(app_name)
  {
    gchar* key_downcase = g_utf8_strdown(key, -1);
    gchar* app_name_downcase = g_utf8_strdown(app_name, -1);
    gboolean key_app_name = strstr(app_name_downcase, key_downcase) != NULL;
    g_free(key_downcase);
    g_free(app_name_downcase);
    g_free(app_name);
    return !key_app_name;
  }
  return TRUE;
}
*/
static void
_call_application(GtkTreeView       *tree_view,
		  GtkTreePath       *path,
		  GtkTreeViewColumn *column,
		  gpointer           user_data)
{
  g_return_if_fail(GTK_IS_WIDGET(user_data));
  GtkWidget* dialog = GTK_WIDGET(user_data);
  DBusGConnection *connection;
  GError *error = NULL;
  connection = dbus_g_bus_get(DBUS_BUS_SESSION, &error);
  if(error)
  {
    g_warning ("Can not connect to session bus. %s",error->message);
    g_error_free (error);
    return ;
  }
  DBusGProxy* proxy = NULL;
  proxy = dbus_g_proxy_new_for_name(connection,
				    APP_MGR_DBUS_NAME,
				    APP_MGR_DBUS_PATH,
				    APP_MGR_DBUS_NAME);

  if(!proxy)
  {
    g_warning ("Can not connect dbus proxy.");
    return;
  }

  GtkTreeSelection* selection = gtk_tree_view_get_selection(tree_view);
  GtkTreeIter iter;
  if(gtk_tree_model_get_iter(GTK_TREE_MODEL(gtk_tree_view_get_model(tree_view)),
			     &iter,
			     path))
  {
    gchar* app_desktop_file = NULL;
    gtk_tree_model_get(gtk_tree_view_get_model(tree_view),
		       &iter,
		       2, &app_desktop_file,
		       -1);
    if(app_desktop_file && g_str_has_suffix(app_desktop_file, ".desktop"))
    {
      gtk_widget_hide(dialog);
      gchar *id = g_strndup(app_desktop_file, strlen (app_desktop_file) - strlen (".desktop"));
      dbus_g_proxy_call_no_reply (proxy,
                                  APP_MGR_LAUNCH_METHODE,
                                  G_TYPE_STRING,
                                  id,
                                  G_TYPE_INVALID,
                                  G_TYPE_INVALID);
      g_free (id);
    }
    if(app_desktop_file)
      g_free(app_desktop_file);
  }
  g_object_unref(proxy);
}

static gboolean
_filter_app_list_func(GtkTreeModel* model,
		      GtkTreeIter* iter,
		      gpointer data)
{
  g_return_if_fail(GTK_IS_ENTRY(data));
  GtkEntry* entry = GTK_ENTRY(data);
  
  if(gtk_entry_get_text_length(entry)>0)
  {
    const gchar* text = gtk_entry_get_text(entry);
    gchar* app_name = NULL;
    
    gtk_tree_model_get(model, iter,
		       1, &app_name,
		       -1);
    gboolean visible = FALSE;
    if(app_name)
    {
      gchar* app_name_downcase = g_utf8_strdown(app_name, -1);
      if(strstr(app_name_downcase, text)!=NULL)
      {
	visible = TRUE;
      }
      g_free(app_name);
      g_free(app_name_downcase);
    }
    return visible;
  }
  
  return TRUE;
}

static void
_refilter_by_text(GtkEditable *entry,
		  gpointer     user_data)
{
  g_return_if_fail(GTK_IS_TREE_MODEL_FILTER(user_data));
  GtkTreeModelFilter* filter = GTK_TREE_MODEL_FILTER(user_data);
  gtk_tree_model_filter_refilter(filter);
}

static gboolean
_button_release(GtkWidget* widget, GdkEventButton *event, AppSearchHomePlugin* desktop_plugin)
{
  GtkWidget* dialog;
  GtkWidget* search_bar;
  GtkWidget *app_selector;
  GtkListStore* app_list;
  GtkTreeModel* filtered_app_list;

  dialog = gtk_dialog_new();/*_with_buttons("Start",
				       NULL,
				       (GtkDialogFlags)0,
				       dgettext("hildon-libs","wdgt_bd_done"),
				       GTK_RESPONSE_ACCEPT,
				       NULL);*/
  gtk_window_set_title(GTK_WINDOW(dialog), APP_TITLE);
  app_selector = hildon_gtk_tree_view_new(HILDON_UI_MODE_NORMAL);
  search_bar = hildon_entry_new(HILDON_SIZE_FINGER_HEIGHT);
  g_object_set(search_bar,"hildon-input-mode",
               HILDON_GTK_INPUT_MODE_FULL, NULL);

  app_list = gtk_list_store_new(3, GDK_TYPE_PIXBUF, G_TYPE_STRING, G_TYPE_STRING);
  gtk_tree_sortable_set_sort_column_id(GTK_TREE_SORTABLE(app_list),
				       1, GTK_SORT_ASCENDING);
  _fill_app_list_liststore_from_app_config_path(app_list);
  
  filtered_app_list = gtk_tree_model_filter_new(GTK_TREE_MODEL(app_list),
						NULL);
  gtk_tree_model_filter_set_visible_func(GTK_TREE_MODEL_FILTER(filtered_app_list),
					 _filter_app_list_func,
					 search_bar,
					 NULL);

  GtkTreeViewColumn* column_icon = gtk_tree_view_column_new();
  GtkTreeViewColumn* column_app_name = gtk_tree_view_column_new();
  GtkCellRenderer* icon_renderer = gtk_cell_renderer_pixbuf_new();
  GtkCellRenderer* app_name_renderer = gtk_cell_renderer_text_new();

  gtk_tree_view_append_column(GTK_TREE_VIEW(app_selector),
			      column_icon);
  gtk_tree_view_append_column(GTK_TREE_VIEW(app_selector),
			      column_app_name);
  gtk_tree_view_column_pack_start(column_icon, icon_renderer, FALSE);
  gtk_tree_view_column_pack_start(column_app_name, app_name_renderer, TRUE);  

  gtk_tree_view_column_set_attributes(column_icon,
				      icon_renderer,
				      "pixbuf",0,
				      NULL);

  gtk_tree_view_column_set_attributes(column_app_name,
				      app_name_renderer,
				      "text",1,
				      NULL);

  gtk_tree_view_set_model(GTK_TREE_VIEW(app_selector),
			  GTK_TREE_MODEL(filtered_app_list));
  

  /*
  gtk_tree_view_set_search_equal_func(GTK_TREE_VIEW(app_selector),
				      search_app_func,
				      desktop_plugin,
				      NULL);
  */
  /*
  gtk_tree_view_set_search_entry(GTK_TREE_VIEW(app_selector),
				 GTK_ENTRY(search_bar));
  */
  GtkWidget* pan = hildon_pannable_area_new();
  gtk_container_add(GTK_CONTAINER(pan), app_selector);
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), pan, TRUE, TRUE, 0);
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), search_bar,FALSE,FALSE,0); 
  g_signal_connect(app_selector, "row-activated", G_CALLBACK(_call_application), dialog);
  g_signal_connect(search_bar, "changed", G_CALLBACK(_refilter_by_text), filtered_app_list);
  gtk_window_set_default_size(GTK_WINDOW(dialog),-1,250);
  gtk_widget_grab_focus(search_bar);
  gtk_widget_show_all(dialog);
  gtk_dialog_run(GTK_DIALOG(dialog));
  gtk_widget_destroy(dialog);
  g_object_unref(app_list);
  g_object_unref(filtered_app_list);
}


static gboolean
app_search_expose(GtkWidget* widget, GdkEventExpose *event)
{
  cairo_t *cr;
  cr = gdk_cairo_create(GDK_DRAWABLE(widget->window));
  gdk_cairo_region(cr, event->region);
  cairo_clip(cr);
  cairo_set_operator(cr, CAIRO_OPERATOR_SOURCE);
  GdkPixbuf* pixbuf =  gdk_pixbuf_new_from_file(ICON_PATH, NULL);

  gdk_cairo_set_source_pixbuf(cr, pixbuf, 0, 0);
  cairo_paint(cr);
  g_object_unref(pixbuf);

  cairo_destroy(cr);
  return GTK_WIDGET_CLASS(app_search_home_plugin_parent_class)->expose_event(widget, event);
}

static void
app_search_realize(GtkWidget* widget)
{
  GdkScreen *screen = gtk_widget_get_screen(widget);
  gtk_widget_set_colormap(widget, gdk_screen_get_rgba_colormap(screen));
  gtk_widget_set_app_paintable(widget, TRUE);
  GTK_WIDGET_CLASS(app_search_home_plugin_parent_class)->realize(widget);
}

static void
app_search_finalize(GObject* object)
{
  AppSearchHomePlugin* desktop_plugin = APP_SEARCH_HOME_PLUGIN(object);
  G_OBJECT_CLASS(app_search_home_plugin_parent_class)->finalize(object);
}

static void
app_search_home_plugin_init(AppSearchHomePlugin* desktop_plugin)
{
  GtkWidget *contents = gtk_event_box_new();
  gtk_event_box_set_visible_window(GTK_EVENT_BOX(contents), FALSE);
  gtk_container_set_border_width(GTK_CONTAINER(contents), 0);
  g_signal_connect(GTK_CONTAINER(contents), 
		   "button-release-event", 
		   G_CALLBACK(_button_release), 
		   desktop_plugin);
  gtk_widget_show_all(contents);
  gtk_widget_set_size_request(GTK_WIDGET(desktop_plugin), 
			      96, 96);
  gtk_container_add(GTK_CONTAINER(desktop_plugin), contents);
}

static void
app_search_home_plugin_class_init(AppSearchHomePluginClass* klass)
{
  GtkWidgetClass* widget_class = GTK_WIDGET_CLASS(klass);
  widget_class->realize = app_search_realize;
  widget_class->expose_event = app_search_expose;
  G_OBJECT_CLASS(klass)->finalize = app_search_finalize;
}

static void
app_search_home_plugin_class_finalize(AppSearchHomePluginClass* klass)
{
}
