#include <QtCore/QFile>
#include <QtGui/QPainter>

#include <math.h>

#include "ameterwidget.h"

qreal g_n = 9.80665;
qreal a_max = 2;
int  angle_step = 30;
int divisions = 4;

// Smoothing: 0 - none, 1 - light, 2 - medium, 3 - strong
int smoothing = 1;
qreal smoothing_k[] = {1, 0.5, 0.25, 0.125};

AMeterWidget::AMeterWidget(QWidget *parent)
    : QWidget(parent)
{
	// Reciprocal of gravity
	r_g = 1.0 / g_n;
	r_a_max = 1.0 / a_max;

	background = 0;

	ax = 0;
	ay = 0;
	az = 0;

	bx = 0;
	by = 0;
	bz = 0;
}

AMeterWidget::~AMeterWidget()
{

}

int AMeterWidget::setGravity(qreal g)
{
	// Reciprocal of gravity
	r_g = 1.0 / g;

	drawScale();

	return 0;
}

// This method is called on every update, keep division operations out

void AMeterWidget::paintEvent(QPaintEvent *e)
{
	QPainter paint(this);
	qreal cx, cy, rx, ry;
	qreal x1, y1, a;

	a = sqrt(ax * ax + ay * ay + az * az);

	if (background)
	{
		paint.drawImage(QPoint(0, 0), *background);
	}

	paint.setRenderHints(QPainter::Antialiasing);

	cx = width() / 2;
	cy = height() / 2;
	rx = cy * r_a_max;
	ry = cy * r_a_max;

	x1 = cx - ax * rx;
	y1 = cy + ay * ry;

	paint.setPen(QPen(QBrush(QColor(255, 255, 255, 255)), 3));
	paint.setBrush(QBrush(QColor(255, 255, 255, 255)));
	paint.drawEllipse(QPointF(x1, y1), 3, 3);
	paint.drawLine(QPointF(cx, cy), QPointF(x1, y1));
}

#define DEG2RAD (M_PI / 180.0)

void AMeterWidget::resizeEvent(QResizeEvent *e)
{
	drawScale();
}

void AMeterWidget::drawScale()
{
	QPainter *paint;
	QString str;
	qreal cx, cy, rx, ry, dx, dy;
	qreal r, a;
	qreal id = 1.0 / divisions;
	int i, j;

	if (background)
	{
		delete background;
		background = 0;
	}

	r_a_max = 1.0 / a_max;
	
	background = new QImage(size(), QImage::Format_ARGB32);
	paint = new QPainter(background);
	paint->setRenderHints(QPainter::Antialiasing);

	cx = width() / 2;
	cy = height() / 2;
	
	if (cx > cy)
	{
		ry = cy * r_a_max;
		rx = ry;
	} else {
		rx = cx * r_a_max;
		ry = rx;
	}
	
	r = sqrt(cx * cx + cy * cy);

	paint->setPen(QPen(QBrush(QColor(64, 64, 64, 255)), 3));

	paint->drawLine(QPointF(cx, 0), QPointF(cx, cy + cy));
	paint->drawLine(QPointF(0, cy), QPointF(cx + cx, cy));

	for (i = 0; i * rx < cx; i++)
	{
		paint->setPen(QPen(QBrush(QColor(64, 64, 64, 255)), 3));
		if (i > 0)
		{
			paint->drawEllipse(QPointF(cx, cy), rx * i, ry * i);
		}

		paint->setPen(QPen(QBrush(QColor(64, 64, 64, 255)), 1));
		for (j = 0; j < divisions; j++)
		{
			paint->drawEllipse(QPointF(cx, cy), rx * (i + id * j), ry * (i + id * j));
		}
		if (i != 0) {
			str = QString::number(i) + "g";
			paint->setPen(QPen(QBrush(QColor(192, 192, 192, 255)), 1));
			paint->drawText(QPointF(cx - rx * i, cy), str);
			paint->drawText(QPointF(cx + rx * i, cy), str);
			paint->drawText(QPointF(cx, cy - ry * i), str);
			paint->drawText(QPointF(cx, cy + ry * i), str);
		}
	}

	for (i = 0; i < 360; i += angle_step)
	{
		a = i * DEG2RAD;
		dx = sin(a) * r;
		dy = cos(a) * r;
		paint->drawLine(QPointF(cx, cy), QPointF(cx + dx, cy + dy));
	}

    delete paint;
}

bool AMeterWidget::filter(QAccelerometerReading *reading)
{
	qreal k = 1;
	bx = ax;
	by = ay;
	bz = az;

	ax = reading->x() * r_g;
	ay = reading->y() * r_g;
	az = reading->z() * r_g;

	if (smoothing > 0 && smoothing < 4)
	{
		k = smoothing_k[smoothing];
		ax = ax * k + bx * (1.0 - k);
		ay = ay * k + by * (1.0 - k);
		az = az * k + bz * (1.0 - k);
	}
	
	update();

	return true;
}

