#!/usr/bin/env python

"""Backend for both the CLI and GUI alarmed modules"""

import os
import subprocess
import pickle
import datetime
import time
import alarm
import dbus

JOB_DIR = os.environ['HOME'] + "/.config/alarmed/jobs/"
PRESET_DIR = os.environ['HOME'] + "/.config/alarmed/presets/"

def emptyEventInfo(self):
    """Doesn't do anything, exists only as a documentative reference"""
    eventInfo = {"id": None, #alarmd cookie ID
                 "schedule": None, #weekly, hourly, in, etc.
                 "type": None, #exec, alarm, profile switch etc.
                 "ttime": None, #ttime for alarmd
                 "weekdays": None, #days of week for recurring
                 "weekday": None, #relative "next Tuesday" weekday
                 "title": None, #even name shown to user
                 "content": None, #command, profile etc.
                 "number": None, #number of steps
                 "interval": None, #type of steps
                 "cron": None, #cron string
                 "date": None, #date-syntax string
                 "enabled": None} #1 if event is enabled, 0 if disabled
    return eventInfo

def loadFieldsDict():
    """Returns a dict containing dicts of all available picker fields"""
    fields = {}
    schedules = {"only_once": "Only once",
                 "every_minute": "Every minute",
                 "hourly": "Hourly",
                 "daily": "Daily",
                 "weekly": "Weekly",
                 "biweekly": "Biweekly",
                 "once_a_month": "Once a month",
                 "end_of_month": "Last day of month",
                 "once_a_year": "Once a year",
                 "in": "In...",
                 "next": "Next...",
                 "tomorrow": "Tomorrow",
                 "cron": "Cron String",
                 "date": "Date String"}
    fields["schedules"] = schedules
    intervals = {"seconds": "Seconds",
                 "minutes": "Minutes",
                 "hours": "Hours",
                 "days": "Days"}
    fields["intervals"] = intervals
    types =     {"alarm_type": "Alarm",
                 "exec_type": "Command execution",
                 "builtin_type": "Built-in function",
                 "profile_type": "Switch profile"}
    fields["types"] = types
    weekdays =  {"mon": "Monday",
                 "tue": "Tuesday",
                 "wed": "Wednesday",
                 "thu": "Thursday",
                 "fri": "Friday",
                 "sat": "Saturday",
                 "sun": "Sunday"}
    fields["weekdays"] = weekdays
    builtins =  {"gprsdataresetter": "Reset GPRS data counter",
                 "offlineswitcher": "Enable flight mode",
                 "onlineswitcher": "Disable flight mode",
                 "bluetoothon": "Enable bluetooth",
                 "bluetoothoff": "Disable bluetooth"}
    fields["builtins"] = builtins
    return fields
    
def loadEventInfo(id):
    """Load a single event from a pickled file"""
    try:
        file = open(JOB_DIR + id, "r")
        eventInfo = pickle.load(file)
        file.close()
        return eventInfo
    except:
        return False
    
def loadEventList():
    """Load existing events by unpickling event files"""
    eventInfos = []
    if not os.path.exists(JOB_DIR):
        os.makedirs(JOB_DIR)
    for file_path in os.listdir(JOB_DIR):
        eventInfo = loadEventInfo(file_path)
        if eventInfo:
            eventInfos.append(eventInfo)
    return eventInfos
    
def deleteEvent(id):
    """Deletes an event from the alarmd queue and the config dir"""
    alarm.delete_event(id)
    try: os.remove(JOB_DIR + str(id))
    except OSError:
        return "Event file for ID " + str(id) + " does not exist"
    return "Deleted event " + str(id)
    
def deleteEventAlarmd(eventInfo):
    """Deletes an event only from the alarmd queue"""
    eventInfo["enabled"] = 0
    deleteEvent(int(eventInfo["id"]))
    if not os.path.exists(JOB_DIR):
        os.makedirs(JOB_DIR)
    job_file = open(JOB_DIR + eventInfo["id"], "w")
    pickle.dump(eventInfo, job_file)
    job_file.close()
    return "Disabled event " + str(eventInfo["id"])
    
def savePreset(preset):
    """Saves a preset made by the user to the config dir"""
    if not os.path.exists(PRESET_DIR):
        os.makedirs(PRESET_DIR)
    preset_file = open(PRESET_DIR + preset[0], "w")
    pickle.dump(preset, preset_file)
    preset_file.close()

def deletePreset(preset):
    """Deletes a preset from the config dir"""
    try: os.remove(PRESET_DIR + preset[0])
    except OSError:
        return "Preset file for " + preset[0] + " does not exist"
    return "Deleted preset " + preset[0]
    
def loadPresetList():
    """Load existing presets by unpickling preset files"""
    presets = []
    if not os.path.exists(PRESET_DIR):
        os.makedirs(PRESET_DIR)
    for file_path in os.listdir(PRESET_DIR):
        file = open(PRESET_DIR + file_path, "r")
        preset = pickle.load(file)
        file.close()
        presets.append(preset)
    return presets
    
def loadPreset(preset_name):
    if not os.path.exists(PRESET_DIR):
        os.makedirs(PRESET_DIR)
    file = open(PRESET_DIR + preset_name, "r")
    preset = pickle.load(file)
    file.close()
    return preset

def getAvailableProfiles():
        return list(dbus.Interface(dbus.SessionBus().get_object('com.nokia.profiled', '/com/nokia/profiled'), 'com.nokia.profiled').get_profiles())
 
def generateEventString(eventInfo):
    """Generates a single string from event data"""
    dt = datetime.datetime.fromtimestamp(eventInfo["ttime"])
    i = loadFieldsDict()["schedules"]
    try:
        if eventInfo["schedule"] in [i["every_minute"], i["hourly"],
                                     i["daily"], i["once_a_month"], 
                                     i["end_of_month"], i["once_a_year"]]:
            eventTitle = (eventInfo["schedule"] + " " +
                           eventInfo["type"] +
                           " starting on " + 
                           dt.strftime("%d.%m.%Y") +
                           " at " + dt.strftime("%H:%M") + ":")
        elif eventInfo["schedule"] in [i["weekly"], i["biweekly"]]:
            days = ""
            for day in eventInfo["weekdays"]:
                days = days + day[0:3] + ", "
            days = days[0:len(days)-1]
            eventTitle = (eventInfo["schedule"] + " " +
                           eventInfo["type"].lower() + " every " +
                           days + " at " + dt.strftime("%H:%M") + ":")
        elif eventInfo["schedule"] in [i["in"], i["next"], 
                                       i["tomorrow"], i["only_once"]]:
            eventTitle = (eventInfo["type"] +
                           " on " + 
                           dt.strftime("%d.%m.%Y") +
                           " at " + dt.strftime("%H:%M") + ":")
        elif eventInfo["schedule"] == i["cron"]:
            c = ""
            for part in eventInfo["cron"]:
                c = c + " " + part
            c = c.lstrip()
            eventTitle = (eventInfo["type"] + 
                           " with cron mask: " + c + ":")
        elif eventInfo["schedule"] == i["date"]:
            eventTitle = (eventInfo["type"].lower() + " with date string: "
                           + eventInfo["date"] + ":")
    except KeyError:
        return "Corrupt job file", "Please delete or re-save me!"
    eventContent = eventInfo["title"]
    return eventTitle, eventContent

def interpretCronString(string):
    """Checks if the string entered is valid and puts it in a list"""
    cronstring = string.split()
    #check string validity
    try:
        if len(cronstring) != 5:
            return False, ("cronstring - " + o.cron + 
                           " doesn't have 5 parts")
        min = cronstring[0].split(",")
        if min[0][:2]=="*/":
            n=int(min[0][2:])
            if n!=0:
                 min=[str(i) for i in range(0,59) if not i%n]
            else:
                 return False, "Division by 0"
        elif min[0].count("-")==1:
            ind=min[0].index("-")
            i=int(min[0][:ind])
            j=int(min[0][ind+1:])
            if 0 <= i <= j <= 59:
                 min=[str(i) for i in range(i,j+1)]
            else:
                 return False, "Bad interval"
        elif min != ["*"]:
            for m in min:
                if not (0 <= int(m) <= 59):
                    return False, ("minute part of cronstring - " + m +
                                  " - is not between 0 and 59 or *")
        hour = cronstring[1].split(",")
        if hour[0][:2]=="*/":
            n=int(hour[0][2:])
            if n!=0:
                 hour=[str(i) for i in range(0,23) if not i%n]
            else:
                 return False, "Division by 0"
        elif hour[0].count("-")==1:
            ind=hour[0].index("-")
            i=int(hour[0][:ind])
            j=int(hour[0][ind+1:])
            if 0 <= i <= j <= 23:
                 hour=[str(i) for i in range(i,j+1)]
            else:
                 return False, "Bad interval"
        elif hour != ["*"]:
            for h in hour:
                if not (0 <= int(h) <= 23):
                    return False, ("hour part of cronstring - " + h +
                                  " - is not between 0 and 23 or *")
        dom = cronstring[2].split(",")
        if dom[0][:2]=="*/":
            n=int(dom[0][2:])
            if n!=0:
                 dom=[str(i) for i in range(1,31) if not i%n]
            else:
                 return False, "Division by 0"
        elif dom[0].count("-")==1:
            ind=dom[0].index("-")
            i=int(dom[0][:ind])
            j=int(dom[0][ind+1:])
            if 1 <= i <= j <= 31:
                 dom=[str(i) for i in range(i,j+1)]
            else:
                 return False, "Bad interval"
        elif dom != ["*"]:
            for d in dom:
                if not (1 <= int(d) <= 31):
                    return False, ("day of month part of cronstring - "
                                 + d +" - is not between 1 and 31 or *")
        month = cronstring[3].split(",")
        if month[0][:2]=="*/":
            n=int(month[0][2:])
            if n!=0:
                 month=[str(i) for i in range(1,12) if not i%n]
            else:
                 return False, "Division by 0"
        elif month[0].count("-")==1:
            ind=month[0].index("-")
            i=int(month[0][:ind])
            j=int(month[0][ind+1:])
            if 1 <= i <= j <= 12:
                 month=[str(i) for i in range(i,j+1)]
            else:
                 return False, "Bad interval"
        elif month != ["*"]:
            for m in month:
                if not (1 <= int(m) <= 12):
                    return False, ("month part of cronstring - " + m +
                                  " - is not between 0 and 12 or *")
        dow = cronstring[4].split(",")
        if dow[0][:2]=="*/":
            n=int(dow[0][2:])
            if n!=0:
                 dow=[str(i) for i in range(0,6) if not i%n]
            else:
                 return False, "Division by 0"
        elif dow[0].count("-")==1:
            ind=dow[0].index("-")
            i=int(dow[0][:ind])
            j=int(dow[0][ind+1:])
            if 0 <= i <= j <= 6:
                 dow=[str(i) for i in range(i,j+1)]
            else:
                 return False, "Bad interval"
        elif dow != ["*"]:
            for d in dow:
                if not (0 <= int(d) <= 6):
                    return False, ("day of week part of cronstring - " +
                                   d + " - is not between 0 and 6 or *")
    except ValueError:
        return False, ("Cron string " + string +
                       "contains invalid characters")
    return True,[",".join(min),",".join(hour),",".join(dom),",".join(month),",".join(dow)]
    
def interpretDateString(string):
    """Checks if date syntax string is valid and returns ttime"""
    try:
        ttime = subprocess.Popen(['gdate','-d',string,'+%s'],
                            stdout=subprocess.PIPE,
                            stderr=subprocess.PIPE).communicate()
    except OSError:
        return False,(
                   "You can't use -d unless you install coreutils-gnu")
    if ttime[1]:
        return False, ("Got an error from gdate: \n" + ttime[1])
    else:
        return True, int(ttime[0].rstrip("\n"))
    

def makeEvent(eventInfo, old_id):
    """Makes a new alarmd event"""
    #make new event with timestamp and title
    event = alarm.Event()
    eventInfo["enabled"] = 1
    try:
        event.alarm_time = eventInfo["ttime"]
        dt = datetime.datetime.fromtimestamp(eventInfo["ttime"])
    except KeyError:
        event.alarm_time = -1
    event.appid = "alarmed"
    t = loadFieldsDict()["types"]
    s = loadFieldsDict()["schedules"]
    add_textdomain = False
    #set action depending on type
    if eventInfo["type"] == t["alarm_type"]:
        if eventInfo["schedule"] in (s["daily"], s["weekly"]):
            event.appid = "worldclock_alarmd_id"
            event.icon = "qgn_list_hclk_alarm"
        action_snooze, action_stop = event.add_actions(2)
        action_stop.flags |= (alarm.ACTION_WHEN_RESPONDED |
                              alarm.ACTION_TYPE_NOP)
        action_stop.label = 'cloc_bd_stop' #'Stop'
        action_snooze.flags |= (alarm.ACTION_WHEN_RESPONDED |
                                alarm.ACTION_TYPE_SNOOZE)
        action_snooze.label = 'cloc_bd_alarm_notification_snooze' #'Snooze'
        event.message = eventInfo["title"].encode("utf-8")
        event.title = eventInfo["title"].encode("utf-8")
        add_textdomain = True
    elif eventInfo["type"] == t["exec_type"]:
        action = event.add_actions(1)[0]
        action.flags |= (alarm.ACTION_WHEN_TRIGGERED | 
                         alarm.ACTION_TYPE_EXEC)
        action_string = eventInfo["content"].replace("'","'\\''").encode("utf-8")
        action_string = "/bin/ash -c '" + action_string + "'"
        action.command = action_string
        
    elif eventInfo["type"] == t["builtin_type"]:
        action = event.add_actions(1)[0]
        action.flags |= (alarm.ACTION_WHEN_TRIGGERED | 
                         alarm.ACTION_TYPE_EXEC)
        b = loadFieldsDict()["builtins"]
        if eventInfo["content"] == b["gprsdataresetter"]:
            action.command = str("/opt/alarmed/gprsdataresetter.sh")
        elif eventInfo["content"] == b["offlineswitcher"]:
            action.command = str(
                               "/opt/alarmed/onlineswitcher.sh offline")
        elif eventInfo["content"] == b["onlineswitcher"]:
            action.command = str(
                                "/opt/alarmed/onlineswitcher.sh normal")
        elif eventInfo["content"] == b["bluetoothon"]:
            action.command = str(
                                "/opt/alarmed/bluetoothswitcher.py on")
        elif eventInfo["content"] == b["bluetoothoff"]:
            action.command = str(
                                "/opt/alarmed/bluetoothswitcher.py off")
    elif eventInfo["type"] == t["profile_type"]:
        action = event.add_actions(1)[0]
        action.flags |= (alarm.ACTION_WHEN_TRIGGERED | 
                         alarm.ACTION_TYPE_EXEC)
        action.command = ("/opt/alarmed/profileswitcher.sh %s" % eventInfo["content"]).encode("utf-8")
    if eventInfo["schedule"] == s["every_minute"]:
        recurrence = event.add_recurrences(1)[0]
        event.recurrences_left = -1
        recurrence.mask_min |= alarm.RECUR_MIN_ALL
        recurrence.mask_hour |= alarm.RECUR_HOUR_DONTCARE
        recurrence.mask_mday |= alarm.RECUR_MDAY_DONTCARE
        recurrence.special |= alarm.RECUR_SPECIAL_NONE
    elif eventInfo["schedule"] == s["hourly"]:
        recurrence = event.add_recurrences(1)[0]
        event.recurrences_left = -1            
        recurrence.mask_min |= 1 << dt.minute
        recurrence.mask_hour |= alarm.RECUR_HOUR_ALL
        recurrence.mask_mday |= alarm.RECUR_MDAY_DONTCARE
        recurrence.special |= alarm.RECUR_SPECIAL_NONE
    elif eventInfo["schedule"] == s["daily"]:
        recurrence = event.add_recurrences(1)[0]
        event.recurrences_left = -1
        recurrence.mask_min |= 1 << dt.minute
        recurrence.mask_hour |= 1 << dt.hour
        recurrence.mask_mday |= alarm.RECUR_MDAY_ALL
        recurrence.special |= alarm.RECUR_SPECIAL_NONE
    elif eventInfo["schedule"] == s["weekly"]:
        recurrence = event.add_recurrences(1)[0]
        event.recurrences_left = -1
        recurrence.mask_min |= 1 << dt.minute
        recurrence.mask_hour |= 1 << dt.hour
        w = loadFieldsDict()["weekdays"]
        if w["mon"] in eventInfo["weekdays"]:
            recurrence.mask_wday |= alarm.RECUR_WDAY_MON
        if w["tue"] in eventInfo["weekdays"]:
            recurrence.mask_wday |= alarm.RECUR_WDAY_TUE
        if w["wed"] in eventInfo["weekdays"]:
            recurrence.mask_wday |= alarm.RECUR_WDAY_WED
        if w["thu"] in eventInfo["weekdays"]:
            recurrence.mask_wday |= alarm.RECUR_WDAY_THU
        if w["fri"] in eventInfo["weekdays"]:
            recurrence.mask_wday |= alarm.RECUR_WDAY_FRI
        if w["sat"] in eventInfo["weekdays"]:
            recurrence.mask_wday |= alarm.RECUR_WDAY_SAT
        if w["sun"] in eventInfo["weekdays"]:
            recurrence.mask_wday |= alarm.RECUR_WDAY_SUN
        recurrence.special |= alarm.RECUR_SPECIAL_NONE
    elif eventInfo["schedule"] == s["biweekly"]:
        recurrence = event.add_recurrences(1)[0]
        event.recurrences_left = -1
        recurrence.mask_min |= 1 << dt.minute
        recurrence.mask_hour |= 1 << dt.hour
        w = loadFieldsDict()["weekdays"]
        if w["mon"] in eventInfo["weekdays"]:
            recurrence.mask_wday |= alarm.RECUR_WDAY_MON
        if w["tue"] in eventInfo["weekdays"]:
            recurrence.mask_wday |= alarm.RECUR_WDAY_TUE
        if w["wed"] in eventInfo["weekdays"]:
            recurrence.mask_wday |= alarm.RECUR_WDAY_WED
        if w["thu"] in eventInfo["weekdays"]:
            recurrence.mask_wday |= alarm.RECUR_WDAY_THU
        if w["fri"] in eventInfo["weekdays"]:
            recurrence.mask_wday |= alarm.RECUR_WDAY_FRI
        if w["sat"] in eventInfo["weekdays"]:
            recurrence.mask_wday |= alarm.RECUR_WDAY_SAT
        if w["sun"] in eventInfo["weekdays"]:
            recurrence.mask_wday |= alarm.RECUR_WDAY_SUN
        recurrence.special |= alarm.RECUR_SPECIAL_BIWEEKLY
    elif eventInfo["schedule"] == s["once_a_month"]:
        recurrence = event.add_recurrences(1)[0]
        event.recurrences_left = -1
        recurrence.mask_min |= 1 << dt.minute
        recurrence.mask_hour |= 1 << dt.hour
        recurrence.mask_mday |= 1 << dt.day
        recurrence.mask_mon |= alarm.RECUR_MON_ALL
        recurrence.special |= alarm.RECUR_SPECIAL_NONE
    elif eventInfo["schedule"] == s["end_of_month"]:
        recurrence = event.add_recurrences(1)[0]
        event.recurrences_left = -1
        recurrence.mask_mday |= alarm.RECUR_MDAY_EOM
        recurrence.mask_mon |= alarm.RECUR_MON_ALL
        recurrence.special |= alarm.RECUR_SPECIAL_NONE
    elif eventInfo["schedule"] == s["once_a_year"]:
        recurrence = event.add_recurrences(1)[0]
        event.recurrences_left = -1
        recurrence.mask_min |= 1 << dt.minute
        recurrence.mask_hour |= 1 << dt.hour
        recurrence.mask_mday |= 1 << dt.day
        if dt.month == 1:
            recurrence.mask_mon = alarm.RECUR_MON_JAN
        elif dt.month == 2:
            recurrence.mask_mon = alarm.RECUR_MON_FEB
        elif dt.month == 3:
            recurrence.mask_mon = alarm.RECUR_MON_MAR
        elif dt.month == 4:
            recurrence.mask_mon = alarm.RECUR_MON_APR
        elif dt.month == 5:
            recurrence.mask_mon = alarm.RECUR_MON_MAY
        elif dt.month == 6:
            recurrence.mask_mon = alarm.RECUR_MON_JUN
        elif dt.month == 7:
            recurrence.mask_mon = alarm.RECUR_MON_JUL
        elif dt.month == 8:
            recurrence.mask_mon = alarm.RECUR_MON_AUG
        elif dt.month == 9:
            recurrence.mask_mon = alarm.RECUR_MON_SEP
        elif dt.month == 10:
            recurrence.mask_mon = alarm.RECUR_MON_OCT
        elif dt.month == 11:
            recurrence.mask_mon = alarm.RECUR_MON_NOW
        elif dt.month == 12:
            recurrence.mask_mon = alarm.RECUR_MON_DEC
        recurrence.special |= alarm.RECUR_SPECIAL_YEARLY
    elif eventInfo["schedule"] == s["cron"]:
        cronstring = eventInfo["cron"]
        recurrence = event.add_recurrences(1)[0]
        event.recurrences_left = -1
        recurrence.mask_hour |= alarm.RECUR_HOUR_DONTCARE
        recurrence.mask_mday |= alarm.RECUR_MDAY_DONTCARE
        recurrence.special |= alarm.RECUR_SPECIAL_NONE
        min = cronstring[0].split(",")
        if min == ["*"]:
            recurrence.mask_min |= alarm.RECUR_MIN_ALL
        else:
            for m in min:
                recurrence.mask_min |= 1 << int(m)
        hour = cronstring[1].split(",")
        if hour == ["*"]:
            recurrence.mask_hour |= alarm.RECUR_HOUR_ALL
        else:
            for h in hour:
                recurrence.mask_hour |= 1 << int(h)
        dom = cronstring[2].split(",")
        if dom == ["*"]:
            recurrence.mask_mday |= alarm.RECUR_MDAY_DONTCARE
        else:
            for d in dom:
                recurrence.mask_mday |= 1 << int(d)
        month = cronstring[3].split(",")
        if month == ["*"]:
            recurrence.mask_mon |= alarm.RECUR_MON_ALL
        else:
            for m in month:
                if int(m) == 1:
                    recurrence.mask_mon |= alarm.RECUR_MON_JAN
                elif int(m) == 2:
                    recurrence.mask_mon |= alarm.RECUR_MON_FEB
                elif int(m) == 3:
                    recurrence.mask_mon |= alarm.RECUR_MON_MAR
                elif int(m) == 4:
                    recurrence.mask_mon |= alarm.RECUR_MON_APR
                elif int(m) == 5:
                    recurrence.mask_mon |= alarm.RECUR_MON_MAY
                elif int(m) == 6:
                    recurrence.mask_mon |= alarm.RECUR_MON_JUN
                elif int(m) == 7:
                    recurrence.mask_mon |= alarm.RECUR_MON_JUL
                elif int(m) == 8:
                    recurrence.mask_mon |= alarm.RECUR_MON_AUG
                elif int(m) == 9:
                    recurrence.mask_mon |= alarm.RECUR_MON_SEP
                elif int(m) == 10:
                    recurrence.mask_mon |= alarm.RECUR_MON_OCT
                elif int(m) == 11:
                    recurrence.mask_mon |= alarm.RECUR_MON_NOW
                elif int(m) == 12:
                    recurrence.mask_mon |= alarm.RECUR_MON_DEC
        dow = cronstring[4].split(",")
        if dow == ["*"]:
            pass
        else:
            for d in dow:
                if int(d) == 1:
                    recurrence.mask_wday |= alarm.RECUR_WDAY_MON
                if int(d) == 2:
                    recurrence.mask_wday |= alarm.RECUR_WDAY_TUE
                if int(d) == 3:
                    recurrence.mask_wday |= alarm.RECUR_WDAY_WED
                if int(d) == 4:
                    recurrence.mask_wday |= alarm.RECUR_WDAY_THU
                if int(d) == 5:
                    recurrence.mask_wday |= alarm.RECUR_WDAY_FRI
                if int(d) == 6:
                    recurrence.mask_wday |= alarm.RECUR_WDAY_SAT
                if int(d) == 0:
                    recurrence.mask_wday |= alarm.RECUR_WDAY_SUN
    #set the event in alarmd
    try:
        new_id = str(alarm.add_event(event))
    except alarm.InvalidEventException:
        return "invalid"
    if add_textdomain:
        from ctypes import cdll
        dll = cdll.LoadLibrary("/usr/lib/libalarm.so.2")
        ev = dll.alarmd_event_get(int(new_id))
        dll.alarm_event_set_attr_string(ev, "textdomain", "osso-clock")
        new_id = str(dll.alarmd_event_update(ev))
    #save the pickled event info
    if old_id:
        deleteEvent(int(old_id))
    eventInfo["id"] = new_id
    if not os.path.exists(JOB_DIR):
        os.makedirs(JOB_DIR)
    job_file = open(JOB_DIR + eventInfo["id"], "w")
    pickle.dump(eventInfo, job_file)
    job_file.close()
    return new_id
