#!/usr/bin/env python

"""GUI frontend to alarmd for scheduling event actions and alarms
Version 0.31"""

import sys
import os
import pickle
import datetime
import alarm

from PySide import QtCore, QtGui, QtMaemo5

# recurrence types
only_once = "Only once"
every_minute = "Every minute"
hourly = "Hourly"
daily = "Daily"
weekly = "Weekly"
biweekly = "Biweekly"
once_a_month = "Once a month"
end_of_month = "Last day of month"
once_a_year = "Once a year"
# event types
alarm_type = "Alarm"
dbus_type = "Dbus call"
exec_type = "Command execution"
builtin_type = "Built-in function"
# days of week
mon = "Monday"
tue = "Tuesday"
wed = "Wednesday"
thu = "Thursday"
fri = "Friday"
sat = "Saturday"
sun = "Sunday"
# builtin functions
gprsdataresetter = "Reset GPRS data counter"

class DataCounterResetter(QtGui.QWidget):
    def __init__(self, parent=None):
        QtGui.QWidget.__init__(self, parent)
        
        self.setWindowTitle("Alarmed")
        
        self.newEventButton = QtGui.QPushButton("New event")
        self.connect(self.newEventButton, QtCore.SIGNAL("clicked()"),
                     self.addEvent)
        self.deleteEventButton = QtGui.QPushButton("Remove")
        self.connect(self.deleteEventButton, QtCore.SIGNAL("clicked()"),
                     self.removeEvent)
        self.editEventButton = QtGui.QPushButton("Edit")
        self.connect(self.editEventButton, QtCore.SIGNAL("clicked()"),
                     self.editEvent)
        
        self.eventList = QtGui.QListWidget()
        self.eventList.setSelectionMode(
                    QtGui.QAbstractItemView.SingleSelection)

        self.addRemoveHBox = QtGui.QHBoxLayout()
        self.addRemoveHBox.addWidget(self.newEventButton)
        self.addRemoveHBox.addWidget(self.deleteEventButton)
        self.addRemoveHBox.addWidget(self.editEventButton)
        self.allVBox = QtGui.QVBoxLayout()
        self.allVBox.addLayout(self.addRemoveHBox)
        self.allVBox.addWidget(self.eventList)
        self.setLayout(self.allVBox)
        
        #load existing jobs (if any) or create config dir first time
        self.job_dir = os.environ['HOME'] + "/.config/alarmed/jobs/"
        if not os.path.exists(self.job_dir):
            os.makedirs(self.job_dir)
        for file_path in os.listdir(self.job_dir):
            self.loadEvent(self.job_dir + file_path)
        
    def loadEvent(self, file_path):
        file = open(file_path,"r")
        eventInfo = pickle.load(file)
        file.close()
        newEventString = self.generateEventString(eventInfo)
        newEvent = QtGui.QListWidgetItem(newEventString)
        newEvent.eventInfo = eventInfo
        self.eventList.addItem(newEvent)
    
    def addEvent(self, editing=False, loading=False):
        self.editing = False
        self.newEventDialog = QtGui.QDialog()
        self.newEventDialog.setWindowTitle("Add event")
        
        self.typePicker = QtGui.QComboBox()
        self.typePicker.addItem(alarm_type)
        self.typePicker.addItem(dbus_type)
        self.typePicker.addItem(exec_type)
        self.typePicker.addItem(builtin_type)
        self.connect(self.typePicker, QtCore.SIGNAL("activated(int)"),
                     self.updateInputField)
        
        self.intervalPicker = QtGui.QComboBox()
        self.intervalPicker.addItem(only_once)
        self.intervalPicker.addItem(every_minute)
        self.intervalPicker.addItem(hourly)
        self.intervalPicker.addItem(daily)
        self.intervalPicker.addItem(weekly)
        self.intervalPicker.addItem(biweekly)
        self.intervalPicker.addItem(once_a_month)
        self.intervalPicker.addItem(end_of_month)
        self.intervalPicker.addItem(once_a_year)
        self.connect(self.intervalPicker,
                     QtCore.SIGNAL("activated(int)"),
                     self.updateDatePicker)
        
        self.dateButton = QtMaemo5.QMaemo5ValueButton("Date")
        self.dateButton.setValueLayout(
                        QtMaemo5.QMaemo5ValueButton.ValueUnderText)
        self.datePicker = QtMaemo5.QMaemo5DatePickSelector()
        self.dateButton.setPickSelector(self.datePicker)
        
        self.weekdayButton = QtGui.QPushButton("Days of week")
        self.connect(self.weekdayButton, QtCore.SIGNAL("clicked()"),
                     self.pickWeekdays)
        self.weekdayButton.hide()
        self.weekdays = []
        
        self.timeButton = QtMaemo5.QMaemo5ValueButton("Time")
        self.timeButton.setValueLayout(
                        QtMaemo5.QMaemo5ValueButton.ValueUnderText)
        self.timePicker = QtMaemo5.QMaemo5TimePickSelector()
        self.timeButton.setPickSelector(self.timePicker)
        
        self.eventName = QtGui.QLineEdit("Enter name")
        self.eventDbusInterface = QtGui.QLineEdit("Interface")
        self.eventDbusService = QtGui.QLineEdit("Service")
        self.eventDbusPath = QtGui.QLineEdit("Path")
        self.eventDbusName = QtGui.QLineEdit("Name")
        self.eventDbusArgs = QtGui.QLineEdit("Args separated by spaces")
        self.eventExec = QtGui.QLineEdit("Enter command")
        self.eventBuiltin = QtGui.QComboBox()
        self.eventBuiltin.addItem(gprsdataresetter)
        self.eventDbusInterface.hide()
        self.eventDbusService.hide()
        self.eventDbusPath.hide()
        self.eventDbusName.hide()
        self.eventDbusArgs.hide()
        self.eventExec.hide()
        self.eventBuiltin.hide()
        
        self.addButton = QtGui.QPushButton("Add")
        self.connect(self.addButton, QtCore.SIGNAL("clicked()"),
                     self.setAlarmEvent)
        self.dateHBox = QtGui.QHBoxLayout()
        self.dateHBox.addWidget(self.typePicker)
        self.dateHBox.addWidget(self.intervalPicker)
        self.dateHBox.addWidget(self.dateButton)
        self.dateHBox.addWidget(self.weekdayButton)
        self.dateHBox.addWidget(self.timeButton)
        self.nameAddHBox = QtGui.QHBoxLayout()
        self.nameAddHBox.addWidget(self.eventName)
        self.nameAddHBox.addWidget(self.addButton)
        self.inputFieldHBox = QtGui.QHBoxLayout()
        self.inputField2HBox = QtGui.QHBoxLayout()
        self.inputFieldHBox.addWidget(self.eventDbusInterface)
        self.inputFieldHBox.addWidget(self.eventDbusService)
        self.inputFieldHBox.addWidget(self.eventDbusPath)
        self.inputFieldHBox.addWidget(self.eventDbusName)
        self.inputField2HBox.addWidget(self.eventDbusArgs)
        self.inputFieldHBox.addWidget(self.eventExec)
        self.inputFieldHBox.addWidget(self.eventBuiltin)
        self.newEventVBox = QtGui.QVBoxLayout()
        self.newEventVBox.addLayout(self.dateHBox)
        self.newEventVBox.addLayout(self.nameAddHBox)
        self.newEventVBox.addLayout(self.inputFieldHBox)
        self.newEventVBox.addLayout(self.inputField2HBox)
        
        self.newEventDialog.setLayout(self.newEventVBox)
        
        # If we're editing, adjust the GUI to the right values
        if (editing):
            self.addButton.setText("Edit")
            self.intervalPicker.setCurrentIndex(
             self.eventList.currentItem().eventInfo["intervalIndex"])
            self.updateDatePicker(
             self.eventList.currentItem().eventInfo["intervalIndex"])
            self.typePicker.setCurrentIndex(
             self.eventList.currentItem().eventInfo["typeIndex"])
            dateTime = QtCore.QDateTime.fromTime_t(
             self.eventList.currentItem().eventInfo["ttime"])
            self.datePicker.setCurrentDate(QtCore.QDate.fromString(
                            dateTime.toString("ddMMyyyy"),"ddMMyyyy"))
            self.timePicker.setCurrentTime(QtCore.QTime.fromString(
                            dateTime.toString("hhmm"),"hhmm"))
            self.weekdays = (
             self.eventList.currentItem().eventInfo["weekdays"])
            self.eventName.setText(
             self.eventList.currentItem().eventInfo["title"])
            if self.eventList.currentItem().eventInfo["interval"] in [
                                                 weekly, biweekly]:
                self.weekdayButton.show()
                self.dateButton.hide()
            if self.eventList.currentItem().eventInfo["type"] == (
                                                      dbus_type):
                self.eventDbusInterface.setText(
                 self.eventList.currentItem().eventInfo["content"][0])
                self.eventDbusService.setText(
                 self.eventList.currentItem().eventInfo["content"][1])
                self.eventDbusPath.setText(
                 self.eventList.currentItem().eventInfo["content"][2])
                self.eventDbusName.setText(
                 self.eventList.currentItem().eventInfo["content"][3])
                self.eventDbusArgs.setText(
                 self.eventList.currentItem().eventInfo["content"][4])
                self.eventDbusInterface.show()
                self.eventDbusService.show()
                self.eventDbusPath.show()
                self.eventDbusName.show()
                self.eventDbusArgs.show()
            elif self.eventList.currentItem().eventInfo["type"] == (
                                                      exec_type):
                self.eventExec.setText(
                 self.eventList.currentItem().eventInfo["content"])
                self.eventExec.show()
            elif self.eventList.currentItem().eventInfo["type"] == (
                                                      builtin_type):
                self.eventBuiltin.setCurrentIndex(
                 self.eventList.currentItem().eventInfo["builtinIndex"] )
                self.eventBuiltin.show()
            self.eventName.setText(
             self.eventList.currentItem().eventInfo["title"])
            self.editing = True
        
        self.newEventDialog.exec_()
    
    def generateEventString(self, eventInfo):
        dateTime = QtCore.QDateTime.fromTime_t(eventInfo["ttime"])
        sdate = QtCore.QDate.fromString(dateTime.toString("ddMMyyyy"),
                                                          "ddMMyyyy")
        stime = QtCore.QTime.fromString(dateTime.toString("hhmm"),
                                                          "hhmm")
        if eventInfo["interval"] in [only_once, every_minute,
                                    hourly, daily, once_a_month, 
                                    end_of_month, once_a_year]:
            newEventString = (eventInfo["interval"] + " " +
                              eventInfo["type"].lower() +
                              " starting on " + 
                              sdate.toString("dd.MM.yyyy") +
                              " at " + stime.toString("hh:mm") +
                              ": " + eventInfo["title"])
        elif eventInfo["interval"] in [weekly, biweekly]:
            days = ""
            for day in eventInfo["weekdays"]:
                days = days + day + ","
            days = days[0:len(days)-1]
            newEventString = (eventInfo["interval"] + " " +
                              eventInfo["type"].lower() + " every " +
                              days + " at " + stime.toString("hh:mm") +
                              ": " + eventInfo["title"])
        return newEventString

    def pickWeekdays(self):
        self.weekdayDialog = QtGui.QDialog()
        self.weekdayDialog.setWindowTitle("Pick weekdays")
        self.weekdayDialogHBox = QtGui.QHBoxLayout()
        self.weekdayDialogVBox = QtGui.QVBoxLayout()
        self.monBox = QtGui.QCheckBox(mon)
        self.tueBox = QtGui.QCheckBox(tue)
        self.wedBox = QtGui.QCheckBox(wed)
        self.thuBox = QtGui.QCheckBox(thu)
        self.friBox = QtGui.QCheckBox(fri)
        self.satBox = QtGui.QCheckBox(sat)
        self.sunBox = QtGui.QCheckBox(sun)
        self.weekdayDialogOK = QtGui.QPushButton("OK")
        self.connect(self.weekdayDialogOK, QtCore.SIGNAL("clicked()"),
                     self.pickWeekdaysDone)
        self.weekdayDialogVBox.addWidget(self.monBox)
        self.weekdayDialogVBox.addWidget(self.tueBox)
        self.weekdayDialogVBox.addWidget(self.wedBox)
        self.weekdayDialogVBox.addWidget(self.thuBox)
        self.weekdayDialogVBox.addWidget(self.friBox)
        self.weekdayDialogVBox.addWidget(self.satBox)
        self.weekdayDialogVBox.addWidget(self.sunBox)
        self.weekdayDialogHBox.addLayout(self.weekdayDialogVBox, 1)
        self.weekdayDialogHBox.addWidget(self.weekdayDialogOK)
        if self.weekdays:
            if mon in self.weekdays: self.monBox.setCheckState(
                                                QtCore.Qt.Checked)
            if tue in self.weekdays: self.tueBox.setCheckState(
                                                QtCore.Qt.Checked)
            if wed in self.weekdays: self.wedBox.setCheckState(
                                                QtCore.Qt.Checked)
            if thu in self.weekdays: self.thuBox.setCheckState(
                                                QtCore.Qt.Checked)
            if fri in self.weekdays: self.friBox.setCheckState(
                                                QtCore.Qt.Checked)
            if sat in self.weekdays: self.satBox.setCheckState(
                                                QtCore.Qt.Checked)
            if sun in self.weekdays: self.sunBox.setCheckState(
                                                QtCore.Qt.Checked)
        self.weekdayDialog.setLayout(self.weekdayDialogHBox)
        self.weekdayDialog.exec_()
        
    def pickWeekdaysDone(self):
        self.weekdays = []
        if self.monBox.isChecked(): self.weekdays.append(mon)
        if self.tueBox.isChecked(): self.weekdays.append(tue)
        if self.wedBox.isChecked(): self.weekdays.append(wed)
        if self.thuBox.isChecked(): self.weekdays.append(thu)
        if self.friBox.isChecked(): self.weekdays.append(fri)
        if self.satBox.isChecked(): self.weekdays.append(sat)
        if self.sunBox.isChecked(): self.weekdays.append(sun)
        self.weekdayDialog.accept()
        

    def updateDatePicker(self, index):
        interval = self.intervalPicker.currentText()
        if interval == every_minute:
            self.weekdayButton.hide()
            self.dateButton.hide()
            self.timeButton.hide()
        if interval in [hourly, daily]:
            self.weekdayButton.hide()
            self.dateButton.hide()
            self.timeButton.show()
        elif interval in [only_once, once_a_month, once_a_year]:
            self.weekdayButton.hide()
            self.dateButton.show()
            self.timeButton.show()
        elif interval in [weekly, biweekly]:
            self.weekdayButton.show()
            self.dateButton.hide()
            self.timeButton.show()
        elif interval == end_of_month:
            self.weekdayButton.hide()
            self.dateButton.hide()
            self.timeButton.show()
            
    def updateInputField(self, index):
        type = self.typePicker.currentText()
        if type == alarm_type:
            self.eventDbusInterface.hide()
            self.eventDbusService.hide()
            self.eventDbusPath.hide()
            self.eventDbusName.hide()
            self.eventDbusArgs.hide()
            self.eventExec.hide()
            self.eventBuiltin.hide()    
        elif type == dbus_type:
            self.eventBuiltin.hide()
            self.eventExec.hide()
            self.eventDbusInterface.show()
            self.eventDbusService.show()
            self.eventDbusPath.show()
            self.eventDbusName.show()
            self.eventDbusArgs.show()
        elif type == exec_type:
            self.eventDbusInterface.hide()
            self.eventDbusService.hide()
            self.eventDbusPath.hide()
            self.eventDbusName.hide()
            self.eventDbusArgs.hide()
            self.eventBuiltin.hide()
            self.eventExec.show()
        elif type == builtin_type:
            self.eventDbusInterface.hide()
            self.eventDbusService.hide()
            self.eventDbusPath.hide()
            self.eventDbusName.hide()
            self.eventDbusArgs.hide()
            self.eventExec.hide()
            self.eventBuiltin.show()

    def removeEvent(self):
        if (self.eventList.currentItem()):
            os.remove(self.job_dir +
             self.eventList.currentItem().eventInfo["id"])
            alarm.delete_event(
             int(self.eventList.currentItem().eventInfo["id"]))
            self.eventList.removeItemWidget(
             self.eventList.currentItem())
        else:
            banner = QtMaemo5.QMaemo5InformationBox()
            # setting the color as a workaround for a bug in pr1.1 Qt
            banner.information(self, '''<font color="#000000">Select an item first</font>''', 2000)
    
    def editEvent(self):
        if (self.eventList.currentItem()):
            self.addEvent(self.eventList.currentItem())
        else:
            banner = QtMaemo5.QMaemo5InformationBox()
            # setting the color as a workaround for a bug in pr1.1 Qt
            banner.information(self, '''<font color="#000000">Select an item first</font>''', 2000)
    
    def setAlarmEvent(self):
        if self.intervalPicker.currentText() in [weekly, biweekly]:
            if not self.weekdays:
                banner = QtMaemo5.QMaemo5InformationBox()
                # setting the color as a workaround for a bug in pr1.1 Qt
                banner.information(self, '''<font color="#000000">Please select at least one week day</font>''', 2000)
                return False
        
        sdate = self.datePicker.currentDate()
        stime = self.timePicker.currentTime()
        dateTime = QtCore.QDateTime()
        dateTime.setDate(sdate)
        dateTime.setTime(stime)
        
        type = self.typePicker.currentText()
        content = ""
        if type == dbus_type:
            content = []
            content.append(str(self.eventDbusInterface.text()))
            content.append(str(self.eventDbusService.text()))
            content.append(str(self.eventDbusPath.text()))
            content.append(str(self.eventDbusName.text()))
            content.append(str(self.eventDbusArgs.text()))
        elif type == exec_type:
            content = str(self.eventExec.text())
        elif type == builtin_type:
            content = str(self.eventBuiltin.currentText())
        
        eventInfo = {}
        eventInfo["interval"] = str(self.intervalPicker.currentText())
        eventInfo["intervalIndex"] = int(
         self.intervalPicker.currentIndex())
        eventInfo["type"] = str(self.typePicker.currentText())
        eventInfo["typeIndex"] = int(self.typePicker.currentIndex())
        eventInfo["ttime"] = int(dateTime.toTime_t())
        eventInfo["weekdays"] = self.weekdays
        eventInfo["title"] = str(self.eventName.text())
        eventInfo["content"] = content
        eventInfo["builtinIndex"] = int(self.eventBuiltin.currentIndex()) or 0
        
        dt = datetime.datetime.fromtimestamp(eventInfo["ttime"])
        
        event = alarm.Event()
        event.alarm_time = eventInfo["ttime"]
        event.appid = str(eventInfo["title"].replace(" ", "_"))
        if eventInfo["type"] == alarm_type:
            action_stop, action_snooze = event.add_actions(2)
            action_stop.flags |= (alarm.ACTION_WHEN_RESPONDED |
                                  alarm.ACTION_TYPE_NOP)
            action_stop.label = 'Stop'
            action_snooze.flags |= (alarm.ACTION_WHEN_RESPONDED |
                                    alarm.ACTION_TYPE_SNOOZE)
            action_snooze.label = 'Snooze'
        elif eventInfo["type"] == dbus_type:
            action = event.add_actions(1)[0]
            action.flags |= (alarm.ACTION_WHEN_TRIGGERED | 
                             alarm.ACTION_DBUS_USE_ACTIVATION | 
                             alarm.ACTION_TYPE_DBUS)
            dbus_args = eventInfo["content"]
            
            action.dbus_interface = "com.nokia.mce"
            action.dbus_service = "com.nokia.mce.request"
            action.dbus_path = "/com/nokia/mce/request"
            action.dbus_name = "req_vibrator_pattern_activate"
            action.set_dbus_args(("string:'PatternTouchscreen'",))
        elif eventInfo["type"] == exec_type:
            action = event.add_actions(1)[0]
            action.flags |= (alarm.ACTION_WHEN_TRIGGERED | 
                             alarm.ACTION_TYPE_EXEC)
            action.command = str(eventInfo["content"])
            
        elif eventInfo["type"] == builtin_type:
            action = event.add_actions(1)[0]
            action.flags |= (alarm.ACTION_WHEN_TRIGGERED | 
                             alarm.ACTION_TYPE_EXEC)
            if eventInfo["content"] == gprsdataresetter:
                action.command = str("/opt/alarmed/gprsdataresetter.py")
            
        if eventInfo["interval"] != only_once:
            recurrence = event.add_recurrences(1)[0]
            event.recurrences_left = -1
        if eventInfo["interval"] == every_minute:
            recurrence.mask_min |= alarm.RECUR_MIN_ALL
            recurrence.mask_hour |= alarm.RECUR_HOUR_DONTCARE
            recurrence.mask_mday |= alarm.RECUR_MDAY_DONTCARE
            recurrence.special |= alarm.RECUR_SPECIAL_NONE
        elif eventInfo["interval"] == hourly:            
            recurrence.mask_min |= 1 << dt.minute
            recurrence.mask_hour |= alarm.RECUR_HOUR_ALL
            recurrence.mask_mday |= alarm.RECUR_MDAY_DONTCARE
            recurrence.special |= alarm.RECUR_SPECIAL_NONE
        elif eventInfo["interval"] == daily:
            recurrence.mask_min |= 1 << dt.minute
            recurrence.mask_hour |= 1 << dt.hour
            recurrence.mask_mday |= alarm.RECUR_MDAY_ALL
            recurrence.special |= alarm.RECUR_SPECIAL_NONE
        elif eventInfo["interval"] == weekly:
            recurrence.mask_min |= 1 << dt.minute
            recurrence.mask_hour |= 1 << dt.hour
            if mon in self.weekdays:
                recurrence.mask_wday |= alarm.RECUR_WDAY_MON
            if tue in self.weekdays:
                recurrence.mask_wday |= alarm.RECUR_WDAY_TUE
            if wed in self.weekdays:
                recurrence.mask_wday |= alarm.RECUR_WDAY_WED
            if thu in self.weekdays:
                recurrence.mask_wday |= alarm.RECUR_WDAY_THU
            if fri in self.weekdays:
                recurrence.mask_wday |= alarm.RECUR_WDAY_FRI
            if sat in self.weekdays:
                recurrence.mask_wday |= alarm.RECUR_WDAY_SAT
            if sun in self.weekdays:
                recurrence.mask_wday |= alarm.RECUR_WDAY_SUN
            recurrence.special |= alarm.RECUR_SPECIAL_NONE
        elif eventInfo["interval"] == biweekly:
            recurrence.mask_min |= 1 << dt.minute
            recurrence.mask_hour |= 1 << dt.hour
            if mon in self.weekdays:
                recurrence.mask_wday |= alarm.RECUR_WDAY_MON
            if tue in self.weekdays:
                recurrence.mask_wday |= alarm.RECUR_WDAY_TUE
            if wed in self.weekdays:
                recurrence.mask_wday |= alarm.RECUR_WDAY_WED
            if thu in self.weekdays:
                recurrence.mask_wday |= alarm.RECUR_WDAY_THU
            if fri in self.weekdays:
                recurrence.mask_wday |= alarm.RECUR_WDAY_FRI
            if sat in self.weekdays:
                recurrence.mask_wday |= alarm.RECUR_WDAY_SAT
            if sun in self.weekdays:
                recurrence.mask_wday |= alarm.RECUR_WDAY_SUN
            recurrence.special |= alarm.RECUR_SPECIAL_BIWEEKLY
        elif eventInfo["interval"] == once_a_month:
            recurrence.mask_min |= 1 << dt.minute
            recurrence.mask_hour |= 1 << dt.hour
            recurrence.mask_mday |= 1 << dt.day
            recurrence.mask_mon |= alarm.RECUR_MON_ALL
            recurrence.special |= alarm.RECUR_SPECIAL_NONE
        elif eventInfo["interval"] == end_of_month:
            recurrence.mask_mday |= alarm.RECUR_MDAY_EOM
            recurrence.mask_mon |= alarm.RECUR_MON_ALL
            recurrence.special |= alarm.RECUR_SPECIAL_NONE
        elif eventInfo["interval"] == once_a_year:
            recurrence.mask_min |= 1 << dt.minute
            recurrence.mask_hour |= 1 << dt.hour
            recurrence.mask_mday |= 1 << dt.day
            if dt.month == 1:
                recurrence.mask_mon = alarm.RECUR_MON_JAN
            elif dt.month == 2:
                recurrence.mask_mon = alarm.RECUR_MON_FEB
            elif dt.month == 3:
                recurrence.mask_mon = alarm.RECUR_MON_MAR
            elif dt.month == 4:
                recurrence.mask_mon = alarm.RECUR_MON_APR
            elif dt.month == 5:
                recurrence.mask_mon = alarm.RECUR_MON_MAY
            elif dt.month == 6:
                recurrence.mask_mon = alarm.RECUR_MON_JUN
            elif dt.month == 7:
                recurrence.mask_mon = alarm.RECUR_MON_JUL
            elif dt.month == 8:
                recurrence.mask_mon = alarm.RECUR_MON_AUG
            elif dt.month == 9:
                recurrence.mask_mon = alarm.RECUR_MON_SEP
            elif dt.month == 10:
                recurrence.mask_mon = alarm.RECUR_MON_OCT
            elif dt.month == 11:
                recurrence.mask_mon = alarm.RECUR_MON_NOW
            elif dt.month == 12:
                recurrence.mask_mon = alarm.RECUR_MON_DEC
            recurrence.special |= alarm.RECUR_SPECIAL_YEARLY
            
        if self.editing:
            old_id = self.eventList.currentItem().eventInfo["id"]
            
        try:
            eventInfo["id"] = str(alarm.add_event(event))
        except alarm.InvalidEventException:
            banner = QtMaemo5.QMaemo5InformationBox()
            # setting the color as a workaround for a bug in pr1.1 Qt
            banner.information(self, '''<font color="#000000">Please pick a date & time in the future</font>''', 2000)
            return False
            

        # Add event to gui list
        newEventString = self.generateEventString(eventInfo)
        if self.editing:
            self.eventList.currentItem().setText(newEventString)
        else:
            newEvent = QtGui.QListWidgetItem(newEventString)
        
        if self.editing:
            self.eventList.currentItem().eventInfo = eventInfo
            alarm.delete_event(int(old_id))
            os.remove(self.job_dir + old_id)
        else:
            newEvent.eventInfo = eventInfo
            self.eventList.addItem(newEvent)
        job_file = open(self.job_dir + eventInfo["id"], "w")
        pickle.dump(eventInfo, job_file)
        job_file.close()
        self.newEventDialog.accept()
        

app = QtGui.QApplication(sys.argv)
widget = DataCounterResetter()
widget.show()
sys.exit(app.exec_())
