/* -*- mode:c; tab-width:4; c-basic-offset:4; -*-
 *
 * This file is part of maemo-security-certman
 *
 * Copyright (C) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * Contact: Juhani Mäkelä <ext-juhani.3.makela@nokia.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <stdarg.h>
#include <syslog.h>
#include <errno.h>
#include <signal.h>
#include <aegis_common.h>

#define DLOG_PORT 2300
#define INET4A(a,b,c,d) (in_addr_t)htonl(a << 24 | b << 16 | c << 8 | d)

/* Capture error messages for aegis_crypto_last_error_str
 */
char* last_dlog_error_message = NULL;

static int32_t msg_nbr = 0;
static char sndbuf [0x4000] = "";
static int subsequent_failures = 0;
static int dlog_socket = -1;
static uint32_t s_addr = (uint32_t)-1;
static int s_port = DLOG_PORT;
static int increase_dlog_signal = -1;
static int decrease_dlog_signal = -1;

/* By default show only errors
 */
#ifndef AEGIS_SHOW_ERRORS
static int show_log_level = 0;
#else
static int show_log_level = DLOG_TO_CONSOLE;
#endif

static void
close_dlog_socket()
{
    AEGIS_DEBUG(1, "%s: finish", __func__);
    if (-1 != dlog_socket) {
        close(dlog_socket);
        dlog_socket = -1;
    }
    if (last_dlog_error_message) {
        free(last_dlog_error_message);
        last_dlog_error_message = NULL;
    }
}


static void
adjust_debug_level(int signo)
{
	if (increase_dlog_signal == signo) {
		show_log_level++;
		AEGIS_DEBUG(show_log_level, "%s: more debug info (%d)", __func__, show_log_level);
	} else if (decrease_dlog_signal == signo) {
		if (0 < show_log_level) {
			show_log_level--;
			AEGIS_DEBUG(show_log_level, "%s: less debug info (%d)", __func__, show_log_level);
		} else {
			AEGIS_DEBUG(0, "%s: debug disabled", __func__);
			show_log_level = -1;
		}
	}
}


static int
init_debug_target(void)
{
	int i1, i2, i3, i4, p, rc;
	char* addr = GETENV("DLOG_TARGET","");
	if (addr && 0 < strlen(addr)) {
		rc = sscanf(addr, "%d.%d.%d.%d:%d", &i1, &i2, &i3, &i4, &p);
		if (rc >= 4) {
			s_addr = INET4A(i1,i2,i3,i4);
			if (rc == 5)
				s_port = p;
			/* $DLOG_TARGET is defined so show all debug output
			 */
			show_log_level = 9;
			return 1;
		}
	} else {
		/* No target, disable all debug output
		 */
		show_log_level = -1;
	}
	return 0;
}


void
init_dlogger(int enabled, int use_signal, int other_signal)
{
	if (!init_debug_target()) {
		/* $DLOG_TARGET was not defined, use default
		 * and log the given amount
		 */
		s_addr = INET4A(127,0,0,1);
		show_log_level = enabled;
	}
	if (enabled > 9)
		show_log_level = enabled;
	if (0 < use_signal) {
		static struct sigaction act;
		increase_dlog_signal = use_signal;
		decrease_dlog_signal = other_signal;
		act.sa_handler = adjust_debug_level;
		act.sa_flags = 0;
		if (0 > sigaction(increase_dlog_signal, &act, NULL))
			AEGIS_ERROR("%s: cannot install signal handler for %d (%s)", 
						__func__, increase_dlog_signal, strerror(errno));
		if (0 > sigaction(decrease_dlog_signal, &act, NULL))
			AEGIS_ERROR("%s: cannot install signal handler for %d (%s)", 
						__func__, decrease_dlog_signal, strerror(errno));
	}
}


void
dlog_message(const char* format, ...)
{
	static struct sockaddr_in i_rad;
	char *fp = NULL, *tmp;
	va_list p_arg;
	int level = 0, tries = 3, old_errno = errno;
	size_t printed = 0, sent = 0;

	if (DLOG_TO_CONSOLE == show_log_level) {
		va_start(p_arg, format);
		vprintf(format, p_arg);
		va_end(p_arg);
		printf("\n");
		return;
	} else if ((-1 == show_log_level) || 
			   ((uint32_t)-1 == s_addr && 0 == init_debug_target())) 
	{
		return;
	}

	if (dlog_socket == -1) {
		unsigned send_buffer_size = 64 * 1024;
		dlog_socket = socket(PF_INET, SOCK_DGRAM, 0);
		if (dlog_socket < 0) {
			syslog(LOG_ERR, "%s(%d)[%s]: ERROR cannot create debug socket (%s)\n",
				   __FILE__, __LINE__, __func__, strerror(errno));
			show_log_level = -1;
			errno = old_errno;
			return;
		}
		/* Set nonblocking mode
		 */
		if (0 > fcntl(dlog_socket, F_SETFL, (long) O_NONBLOCK)) {
			syslog(LOG_ERR, "%s(%d)[%s]: ERROR failed to set O_NONBLOCK (%s)\n",
				   __FILE__, __LINE__, __func__, strerror(errno));
		}
		/* Make the outbut buffer rather big
		 */
		setsockopt(dlog_socket, SOL_SOCKET, SO_SNDBUF, &send_buffer_size, 
				   sizeof(send_buffer_size));
        atexit(close_dlog_socket);
	}
	i_rad.sin_family = AF_INET;
	i_rad.sin_addr.s_addr = s_addr;
	i_rad.sin_port = htons(s_port);

	memcpy(sndbuf, &msg_nbr, sizeof(msg_nbr));
	msg_nbr++;

	va_start(p_arg, format);
	tmp = sndbuf + sizeof(msg_nbr);
	printed = vsnprintf(tmp, sizeof(sndbuf) - sizeof(msg_nbr) - 1, format, 
						p_arg) + sizeof(msg_nbr);
	va_end(p_arg);
	if ('<' == *tmp && '>' == *(tmp + 2))
		level = *(tmp + 1) - '0';
	else
		level = 2;

	/* Save error messages to last_dlog_error_message
	 */
	if (0 == level) {
		if (last_dlog_error_message)
			free(last_dlog_error_message);
		last_dlog_error_message = strdup(tmp + 3);
	}

	if (level > show_log_level)
		return;

	fp = sndbuf;
	do {
		sent = sendto(dlog_socket, fp, printed, 0, // MSG_DONTWAIT, 
					  (struct sockaddr*)&i_rad, 
					  sizeof(struct sockaddr_in));
		if (0 > sent) {
			if (EAGAIN == errno)
				usleep(10);
			tries--;
		} else {
			printed -= sent;
			fp += sent;
		}
	} while (0 < printed && 0 < tries);
    if (0 == tries) {
        subsequent_failures++;
        if (10 <= subsequent_failures) {
			syslog(LOG_ERR, "%s(%d)[%s]: disable debug logging\n",
				   __FILE__, __LINE__, __func__);
            show_log_level = -1;
        }
    } else {
        subsequent_failures = 0;
        usleep(1);
    }
	errno = old_errno;
}
