/* -*- mode:c++; tab-width:4; c-basic-offset:4; -*-
 *
 * This file is part of Aegis crypto services
 *
 * Copyright (C) 2010-2011 Nokia Corporation and/or its subsidiary(-ies).
 *
 * Contact: Juhani Mäkelä <ext-juhani.3.makela@nokia.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

/**

   \file aegis_storage.cpp
   \ingroup sec_storage
   \brief The protected storage implementation

*/

/* This would put all private date into uid specific directories
 * but that would break too many things, for instance the CUD script
 * wouldn't be able to erase data created by different users.
 * #define USE_PRIVATE_DIRS
 */

#include <aegis_storage.h>
#include <aegis_common.h>
#include <aegis_crypto.h>

using namespace std;
using namespace aegis;

#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <utime.h>
#include <pwd.h>
#include <sys/mman.h>
#include <semaphore.h>
#include <sys/inotify.h>
#include <limits.h>
#include <sys/file.h> /* for flock(2) */
#include <openssl/err.h>

#include <openssl/sha.h>

/* Use SHA1 for the time being
 */
#define DIGEST_CTX SHA_CTX
#define DIGEST_OK MDOK
#define DIGEST_INIT(c) (MDOK == SHA1_Init(&c))
#define DIGEST_UPDATE(c,d,l) SHA1_Update(&c,d,l)
#define DIGEST_FINAL(c,d) SHA1_Final(d,&c)

/* Use AES with 128 bit keys.
 */
#define AES_KEY_BITLEN 128

/* Data is encrypted and decrypted in fixed size chunks
 * of 2^BLOCKSHFT bytes, currently 16 bytes. Encrypted
 * file sizes are rounded up chunk size boundary. 
 */
#define BLOCKSIZE 0x10
#define BLOCKMASK 0x0f // (BLOCKSIZE - 1)
#define BLOCKSHFT 4    // ln2(BLOCKSIZE)
#define BLOCKROUND(x) ((x&BLOCKMASK)?((x|BLOCKMASK)+1):(x))

#include <openssl/aes.h>

#ifdef USE_CREDS
#include <sys/creds.h>
#endif

/* Increase this every time the storage format
 * is changed to enable autoconversion. If it doesn't
 * match with what is stored in the index file's
 * pstore.version attribute then storage::convert_store
 * is automatically called at the first open with 
 * read privileges.
 */
#define CURRENT_STORAGE_VERSION 2

/* Encrypted file header integrity checksum. Although
 * MD5 is not cryptographically secure any more, it's 
 * still a good checksum.
 */
#include <openssl/md5.h>
#define CSUM_CMP(data,length,to) MD5(data,length,to)
#define CSUM_LEN MD5_DIGEST_LENGTH

#define DATA_ROOT "/var/lib/aegis/ps"
#define HOME_ROOT "/home/user/.aegis/ps"
#define DEFAULT_CRYPTO_ROOT HOME_ROOT "/.e/"
#define OLD_CRYPTO_ROOT DATA_ROOT "/.e/"
#define SIGNED_ROOT DATA_ROOT "/.s/"

static char crypto_root[256] = DEFAULT_CRYPTO_ROOT;

const char*
aegis::storage_root()
{
    return DATA_ROOT;
}

bool 
aegis::set_alternative_storage_root(const char *to_this)
{
	if (NULL != to_this && sizeof crypto_root > (strlen(to_this) + 1)
		&& strlen(to_this) && '/' == *to_this) 
	{
		strncpy(crypto_root, to_this, sizeof(crypto_root));
	} else {
		errno = EINVAL;
		AEGIS_ERROR("%s: invalid argument '%s'", __func__, 
					to_this?to_this:"<null>");
		return false;
	}
	if ('/' != crypto_root[strlen(crypto_root) - 1]) {
		strcat(crypto_root, "/");
	}
	AEGIS_DEBUG(1, "%s: %s", __func__, crypto_root);
	return true;
}


/**
 * \def MDOK
 * \brief The "no error" return code of the EVP-functions
 * in openssl library
 */

#define MDOK 1

namespace aegis {

	struct keypack_t {
		AES_KEY twk_key;
		AES_KEY enc_key;
		AES_KEY dec_key;
		int refcount;
	};

	class storage_lock 
	{
		friend class storage;
	public:
		storage_lock(storage *parent);
		~storage_lock();

		/* Locking
		 */
		void wait();
		void post();
		bool locked();

		/* Is readable, writable?
		 */
		bool readable();
		bool writable();
		void readable(bool);
		void writable(bool);
		void unreadable() { readable(false); }
		void unwritable() { writable(false); }

		/* Keep count of member files open 
		 * for writing to flush them at commit.
		 */
		void opened_for_writing(p_file*);
		void flush_open_files();
		void closed(p_file*);

		/* Check for changes made outside current process
		 */
		void mark(struct aegis_digest_t*);
		void remember();
		bool changed();

		/* Cleanup
		 */
		void unlink();
		
		/* TODO: Abstract these later
		 */
		struct keypack_t keypack;
		string m_storage_path;

	private:
		storage *m_parent;
		int m_fd;
		int lock_cnt;
		bool can_read;
		bool can_write;
		bool m_read_only;
		vector<p_file*> open_files;
		int version;
		struct aegis_digest_t m_localhash, *m_sharedhash;
		string mem_name;
		string sem_name;
	};

	struct metadata_t {
		/* File metadata */
		struct stat fs;
		/* Checksum */
		string csum;
		/* File has been added since last commit */
		bool new_file;
	};

	class pe_file : public p_file 
	{
		friend class storage;
		friend class storage_lock;
	public:
		virtual bool    p_open(int flags);
		virtual ssize_t p_read(off_t at, RAWDATA_PTR data, size_t len);
		virtual ssize_t p_write(off_t at, const RAWDATA_PTR data, size_t len);
		virtual int     p_trunc(off_t at);
		virtual void    p_close();
		virtual int     p_stat(struct stat *st);
#if 0
        virtual int     p_rename(const char* new_name);
#endif
        virtual int p_chmod(mode_t flags);
        virtual int p_chown (uid_t uid, gid_t gid);
        virtual int p_utime(struct utimbuf *ntime);

	private:
		pe_file(storage* owner, const char* pathname);
		virtual ~pe_file();
		virtual const char* p_name();
		void p_flush(bool do_trunc);
        void set_pathname(const char *to_this, bool convert);
        bool change_pathname(const char *to_this);
		virtual const char* digest();
		virtual size_t datasize();
        virtual int p_cleanup();

		typedef enum {cop_encrypt, cop_decrypt} cop_t;

		void crypto_block(size_t blocknr, cop_t cop, int8_t* encrypted, int8_t *plaintext);
		void crypto_op(cop_t cop, off_t at, RAWDATA_PTR data, size_t len);

		/* Automatic conversion from the old format to the new
		 */
		void decrypt_old(AES_KEY *okey, size_t blocknr, int8_t* encrypted, int8_t *plaintext);
		bool convert_crypto(AES_KEY *okey, string& new_digest);

		struct f_hdr {
			int8_t salt [AES_BLOCK_SIZE];
			struct stat fs;
            unsigned char csum [CSUM_LEN];
		} m_hdr;
		string  m_actname;
	};
}

static int
get_storage_directory(storage::visibility_t visibility, 
					  storage::protection_t protection, 
					  string& dir_name) 
{
	int rc, access_mode = 0;

	if (storage::prot_encrypted == protection)
		dir_name.assign(HOME_ROOT);
	else
		dir_name.assign(DATA_ROOT);

    switch (visibility)
        {
        case storage::vis_global:
            dir_name.append("/G");
            break;
        case storage::vis_shared:
            dir_name.append("/S");
            break;
        case storage::vis_private:
            dir_name.append("/P");
            break;
        }
    switch (protection)
        {
        case storage::prot_encrypted:
            dir_name.append("e/");
            break;
        case storage::prot_signed:
            dir_name.append("s/");
            break;
        }

    access_mode = 0777;

	if (!directory_exists(dir_name.c_str())) {
		rc = create_directory(dir_name.c_str(), access_mode);
		if (0 != rc) {
			AEGIS_ERROR("cannot create directory '%s'", dir_name.c_str());
			return(rc);
		} else {
			AEGIS_DEBUG(1, "%s: created directory '%s'", __func__, dir_name.c_str());
		}
	}
	return(0);
}


void
storage::lock_store()
{
    AEGIS_ENTER;
	return m_lock->wait();
}


void
storage::unlock_store()
{
    AEGIS_ENTER;
	m_lock->post();
}


#define HAS_CHANGED(s)  (s->storage_changed() == true)

#define REINITIALIZE_STORE(s) do { \
		AEGIS_DEBUG(2, "store has changed, reread!");					\
		s->reinitialize();												\
	} while (0);

#define CHECK_VALID(f) do {												\
		if(f->m_semname == "")											\
			break;														\
		sem_t *sem = sem_open(f->m_semname.c_str(), 0);					\
		if (SEM_FAILED == sem) {										\
			errno = EINVAL;												\
			return;														\
		} else {														\
			sem_close(sem); 											\
		}																\
	} while (0);

#define CHECK_VALID_RET(f, d) do {										\
		if(f->m_semname == "") break;									\
		sem_t *sem = sem_open(f->m_semname.c_str(), 0);					\
		if(SEM_FAILED == sem) {											\
			errno = EINVAL;												\
			return d;													\
		} else {														\
			sem_close(sem);												\
		}																\
	} while (0);

#define IS_INITIALIZED(s) (0 < s->m_filename.length())
#define IS_READABLE(s)    (s->m_lock->readable() && IS_INITIALIZED(s))
#define IS_MODIFIABLE(s)  (s->m_lock->writable() && IS_READABLE(s))

storage::status_t
storage::status()
{
	if (IS_MODIFIABLE(this))
		return storage::writable;
	else if (IS_READABLE(this))
		return storage::readable;
	else
		return storage::no_access;
}

/* Access control checks
 */
#define CAP_DAC_OVERRIDE "CAP::dac_override"

#ifndef USE_CREDS
typedef int creds_t;
#define creds_gettask(p) 0
#define creds_free(c)
#endif

static bool
has_dac_override(creds_t pcreds, pid_t pid)
{
#ifdef USE_CREDS
    static creds_value_t cap_nbr = 0;
    static creds_type_t cap_type = creds_str2creds(CAP_DAC_OVERRIDE,
                                                   &cap_nbr);
    bool res = false;
    if (CREDS_BAD != cap_type) {
        res = (1 == creds_have_p(pcreds, cap_type, cap_nbr));
        AEGIS_DEBUG(1, "%s: %d does %shave %s", __func__, 
                    (int)pid, res?"":"not ", CAP_DAC_OVERRIDE);
    } else {
        AEGIS_DEBUG(1, "%s: %s invalid credential?", __func__, 
                    CAP_DAC_OVERRIDE);
    }
    return res;
#else
	return false;
#endif
}


static bool
has_supplementary(creds_t pcreds, pid_t pid, gid_t gid)
{
#ifdef USE_CREDS
    static creds_value_t grp_nbr;
    static creds_type_t grp_type = creds_str2creds("GRP::", &grp_nbr);
    bool res = (1 == creds_have_p(pcreds, grp_type, (creds_value_t)gid));
    AEGIS_DEBUG(1, "%s: %d does %shave GRP::%d", __func__, (int)pid,
                res?"":"not ", (int)gid);
    return res;
#else
    return false;
#endif
}

const char *access_modes[] = {
    "reading",
    "writing",
    "executing",
    "owning"
};

bool
aegis_storage_dac_check(struct stat *fs, int of_type, pid_t pid, uid_t uid, gid_t gid)
{
    creds_t pcreds = creds_gettask(pid);

    if (sizeof(access_modes)/sizeof(char*) <= (size_t)of_type) {
        AEGIS_DEBUG(1, "%s: invalid access mode %d", __func__, of_type);
        creds_free(pcreds);
        return false;
    }

    if (has_dac_override(pcreds, pid)) {
        creds_free(pcreds);
        return true;
    }

    int allow = 0;
    if (fs->st_uid == uid) {
        switch (of_type) {
        case 0:
            allow = fs->st_mode & S_IRUSR;
            break;
        case 1:
            allow = fs->st_mode & S_IWUSR;
            break;
        case 2:
            allow = fs->st_mode & S_IXUSR;
            break;
        case 3:
            allow = 1;
            break;
        default:
            allow = 0;
        }
    } else if ((fs->st_gid == gid)
               || has_supplementary(pcreds, pid, fs->st_gid)) 
    {
        switch (of_type) {
        case 0:
            allow = fs->st_mode & S_IRGRP;
            break;
        case 1:
            allow = fs->st_mode & S_IWGRP;
            break;
        case 2:
            allow = fs->st_mode & S_IXGRP;
            break;
        case 3:
            allow = 0;
            break;
        default:
            allow = 0;
        }
    } else {
        switch (of_type) {
        case 0:
            allow = fs->st_mode & S_IROTH;
            break;
        case 1:
            allow = fs->st_mode & S_IWOTH;
            break;
        case 2:
            allow = fs->st_mode & S_IXOTH;
            break;
        case 3:
            allow = 0;
            break;
        default:
            allow = 0;
        }
    }

    creds_free(pcreds);

    if (0 == allow) {
        AEGIS_DEBUG(1, 
                    "%s: access denied, %0o[%d/%d] doesn't allow %s for [%d/%d]", 
                    __func__,
                    (int)fs->st_mode, (int)fs->st_uid, (int)fs->st_gid,
                    access_modes[of_type],
                    (int)uid, (int)gid);
        errno = EACCES;
        return false;
    } else {
        AEGIS_DEBUG(1, 
                    "%s: access pass, %0o[%d/%d] allows %s for [%d/%d]", 
                    __func__,
                    (int)fs->st_mode, (int)fs->st_uid, (int)fs->st_gid,
                    access_modes[of_type],
                    (int)uid, (int)gid);
        return true;
    }
}


void 
storage::normalize_filename(const char *name, string& result)
{
    if (NULL == name)
        return;
    if (prot_encrypted == m_prot) {
        if ('/' == *name)
            result.assign(name);
        else {
            result.assign("/");
            result.append(name);
        }
    } else {
        absolute_pathname(name, result);
    }
}


bool
storage::remove_all_files()
{
    AEGIS_ENTER;
    if (!IS_MODIFIABLE(this)) {
        AEGIS_ERROR("%s: access denied", __func__);
        errno = EACCES;
        return false;
    }
    if (prot_encrypted == m_prot) {
        for (
			 contents_map_t::reverse_iterator ii = m_contents.rbegin();
			 ii != m_contents.rend();
			 ii++
        ) {
            AEGIS_DEBUG(1, "%s: remove %s", __func__, ii->first.c_str());
			p_file *item = member(ii->first.c_str());
			if (item) {
				item->p_cleanup();
				delete item;
			}
        }
    }
	m_contents.clear();
	m_links.clear();
	m_lock->unlink();
    AEGIS_DEBUG(1, "%s: remove %s", __func__, m_filename.c_str());
    if (file_exists(m_filename.c_str())) {
		if (0 > unlink(m_filename.c_str())) {
			AEGIS_DEBUG(1, "%s: cannot erase '%s' (%s)",
						__func__, m_filename.c_str(), strerror(errno));
			return false;
		}
    }
    return true;
}


bool
storage::generate_new_symkey()
{
    size_t plakey_len = 0;
    RAWDATA_PTR plakey = NULL;
	bool res = false;
    
    plakey_len = aegis_crypto_new_symkey(&plakey);
    if (0 < plakey_len && NULL != plakey) {
        aegis_crypto_result erc;
        erc = aegis_crypto_encrypt(plakey, plakey_len, m_token.c_str(), 
                                   &m_symkey, &m_symkey_len);
        if (aegis_crypto_ok == erc) {
            AEGIS_DEBUG(1, "%s: success", __func__);
			res = true;
        } else {
            AEGIS_ERROR("Could not encrypt symkey (%s)", aegis_crypto_last_error_str());
        }
        memset(plakey, '\0', plakey_len);
        aegis_crypto_free(plakey);
    } else {
        AEGIS_ERROR("Could not generate symkey (%s)", aegis_crypto_last_error_str());
    }
	return res;
}


bool
has_token(const char *token, bool *valid_token)
{
#ifdef USE_CREDS
	creds_t creds = NULL;
	creds_type_t c_type;
	creds_value_t c_value;

	if (NULL == token || 0 == strlen(token)) {
        *valid_token = false;
		return false;
	}
	if (0 == strcmp(token, UNKNOWN_APP_ID)) {
        *valid_token = false;
		return true;
    }
	c_type = creds_str2creds(token, &c_value);
	if (CREDS_BAD == c_type) {
		AEGIS_DEBUG(1, "%s: undefined credential '%s'", __func__, token);
        *valid_token = false;
		/* In build environment assume that invalid tokens are owned
		 */
		switch (aegis_current_mode()) 
			{
			case aegis_system_plain:
			case aegis_system_emulated:
				return true;
			default:
				return false;
			}
	}
    *valid_token = true;
	creds = creds_gettask(0);
	if (1 != creds_have_p(creds, c_type, c_value)) {
		AEGIS_DEBUG(1, "%s: don't have '%s'", __func__, token);
        creds_free(creds);
		return false;
	}
    creds_free(creds);
    AEGIS_DEBUG(1, "%s: has '%s'", __func__, token);
    return true;
#else
    AEGIS_DEBUG(1, "%s: libcreds not available", __func__);
    *valid_token = true;
    return true;
#endif
}

bool
storage::move_file(const char *old_name, const char *new_name)
{
	int fd_o = -1, fd_n = -1;
	char *data = NULL;
	struct stat fs;
	ssize_t written;
	bool res = false;

	fd_o = open(old_name, O_RDONLY);
	if (0 > fd_o) {
		AEGIS_ERROR("%s: cannot open '%s' (%s)", __func__, old_name, 
					strerror(errno));
		goto end;
	}
	if (0 > fstat(fd_o, &fs)) {
		AEGIS_ERROR("%s: cannot stat '%s' (%s)", __func__, old_name, 
					strerror(errno));
		goto end;
	}
	data = (char*)mmap(NULL, fs.st_size, PROT_READ, MAP_SHARED, fd_o, 0);
	if (MAP_FAILED == data) {
		AEGIS_ERROR("%s: cannot stat '%s' (%s)", __func__, old_name, 
					strerror(errno));
		goto end;
	}
	fd_n = open(new_name, O_CREAT | O_WRONLY, fs.st_mode);
	if (0 > fd_n) {
		AEGIS_ERROR("%s: cannot create '%s' (%s)", __func__, new_name, 
					strerror(errno));
		goto end;
	}
	written = write(fd_n, data, fs.st_size);
	if (written < fs.st_size) {
		AEGIS_ERROR("%s: cannot write to '%s', written %ld, size %ld (%s)", 
					__func__, new_name, (long)written, 
					(long)fs.st_size, strerror(errno));
		unlink(new_name);
		goto end;
	}
	if (0 > unlink(old_name))
		AEGIS_ERROR("Couldn't unlink '%s' (%s)", old_name, strerror(errno));
	AEGIS_DEBUG(1, "%s: moved '%s' to '%s'", __func__, old_name, new_name);
	res = true;
end:
	if (-1 != fd_n)
		close(fd_n);
	if (NULL != data)
		munmap(data, fs.st_size);
	if (-1 != fd_o)
		close(fd_o);
	return res;
}


/* Global signatures are stored in a special sha1sum formatted
 * file. It's integrity is verified at boot and kernel guards
 * it against all modifications.
 */

#define GLOBAL_SIGNATURE "external"

/* From aegis_crypto.cpp */
aegis_crypto_result 
verify_global_signature(const char *filename, 
						struct aegis_digest_t *digest);
bool 
save_global_signature(const char *filename, 
					  struct aegis_digest_t *digest);


void
storage::init_storage(const char* name, 
                      const char* token) 
{
	c_xmldoc xdoc;
    bool valid_token = false;
    bool taken_over = false;
	string save_filename;

    if (NULL == name) {
        AEGIS_ERROR("Must give name for storage, using default");
        m_name = ".anonymous";
    } else {
        m_name = name;
    }

	/* Create m_lock and set it nonswappable to protect
	 * the keypack in it
	 */
	if (NULL == m_lock) {
		m_lock = new storage_lock(this);
		if (0 > mlock(m_lock, sizeof(*m_lock))) {
			AEGIS_DEBUG(1, "%s: cannot mlock m_lock (%s)", __func__, strerror(errno));
		}
	}

	m_contents.clear();
	m_links.clear();

    /* Make sure crypto is initialized
     */
	if (!aegis_crypto_init()) {
		AEGIS_ERROR("%s: error from aegis_crypto_init", __func__);
		m_lock->unreadable();
        errno = EACCES;
        return;
    }

	if (NULL != token) {
		if (has_token(token, &valid_token)) {
			m_token.assign(token);
			AEGIS_DEBUG(1, "%s: using token '%s'", __func__, m_token.c_str());
		} else if (prot_encrypted == m_prot) {
			/* Do not allow using a bogus or not owned token 
			 * with encrypted stores
			 */
			AEGIS_DEBUG(1, "%s: no access to '%s'", __func__, token);
			m_lock->unreadable();
			errno = EACCES;
			return;
		} else {
			token = NULL;
		}
	}
	if (NULL == token) {
        /* Use the application id as default
         */
		char* appid = NULL;
		aegis_application_id(0, &appid);
		m_token.assign(appid);
		aegis_crypto_free(appid);
        if (!has_token(m_token.c_str(), &valid_token) || !valid_token) {
			AEGIS_DEBUG(1, "%s: no identity (%s)", __func__, m_token.c_str());
			m_lock->unwritable();
			m_token = "";
		}
	}

	AEGIS_DEBUG(1, "%s: enter %s (%s)", __func__, m_name.c_str(), m_token.c_str());

	if (0 != get_storage_directory(m_vis, m_prot, m_filename)) {
		AEGIS_ERROR("%s: cannot bind storage '%s' to file", __func__, 
					m_filename.c_str());
		m_lock->unreadable();
        errno = EACCES;
		return;
	}
	m_filename.append(m_name);

	AEGIS_DEBUG(1, "%s: map to %s", __func__, m_filename.c_str());

	/* Check with encrypted stores if the index file needs to be
	 * moved into a new location.
	 */
	if (prot_encrypted == m_prot && !file_exists(m_filename.c_str())) {
		AEGIS_DEBUG(1, "%s: '%s' does not exist", __func__, m_filename.c_str());
		string old_file_name = m_filename;
		old_file_name.erase(0, strlen(HOME_ROOT));
		old_file_name.insert(0, DATA_ROOT);
		AEGIS_DEBUG(1, "%s: check for '%s'", __func__, old_file_name.c_str());
		if (file_exists(old_file_name.c_str())) {
			AEGIS_DEBUG(1, "%s: using old index file '%s'", __func__, old_file_name.c_str());
			save_filename = m_filename;
			m_filename = old_file_name;
		}
	}

	lock_store();
	xdoc.parse_file(m_filename.c_str());
	xdoc.release_parser();
	m_lock->remember();
	unlock_store();

	if (NULL == xdoc.root()) {
		struct stat fs;
		if (0 == stat(m_filename.c_str(), &fs)) {
			AEGIS_ERROR("invalid storage index '%s' size %lu, recreate", 
						m_filename.c_str(), (unsigned long)fs.st_size);
		} else if (ENOENT != errno) {
			AEGIS_ERROR("'%s' is not readable by %d, fail (%s)", 
						m_filename.c_str(), (int)geteuid(), strerror(errno));
			m_lock->unreadable();
			errno = EACCES;
			return;
		}
		if (!m_lock->writable()) {
			AEGIS_ERROR("'%s' is not writable by %d, fail", 
						m_filename.c_str(), (int)geteuid());
			errno = EACCES;
			return;
		} else {
			AEGIS_DEBUG(1, "%s: new store '%s'", __func__, m_filename.c_str());
			if (prot_encrypted == m_prot) {
				if (!generate_new_symkey()) {
					m_lock->unreadable();
					errno = EACCES;
				}
			}
			m_lock->m_storage_path = crypto_root;
			AEGIS_DEBUG(1, "%s: storage path is '%s'", __func__, 
						m_lock->m_storage_path.c_str());
		}
		return;
	}
	
	if (prot_encrypted == m_prot) {
		m_lock->m_storage_path = xdoc.root()->attribute("storage-path", false, crypto_root);
		AEGIS_DEBUG(1, "%s: storage path is '%s'", __func__, m_lock->m_storage_path.c_str());
	}

	c_xmlnode *nodes = xdoc.root()->child("envelope", true);

	for (int i = 0; i < nodes->nbrof_children(); i++) {

		c_xmlnode *node = nodes->child(i);

		if (0 == strcmp("file", node->name())) {
			const char *attr, *name;
			name = node->attribute("name", true, "");
			m_contents[name].csum =	node->attribute("hash", true, "");
			m_contents[name].new_file = false;

			/* For signed stores read the metadata from the store file 
			 */
			if (prot_signed == m_prot){
				attr = node->attribute("size", false, NULL);
				struct stat fs = {0};
				/* The attributes are in the storage file */
				if (NULL != attr) {
					fs.st_size = atol(attr);
					fs.st_uid  = atol(node->attribute("uid", false, "0"));
					fs.st_gid  = atol(node->attribute("gid", false, "0"));
					fs.st_mode = atol(node->attribute("mode", false, "0"));

					/* M,A and C times */
					fs.st_mtime = fs.st_atime = fs.st_ctime = time(NULL);
					attr = node->attribute("mtime", false, NULL);
					if (NULL != attr)
						fs.st_mtime = atol(attr);
					attr = node->attribute("atime", false, NULL);
					if (NULL != attr)
						fs.st_atime = atol(attr);
					attr = node->attribute("ctime", false, NULL);
					if (NULL != attr)
						fs.st_ctime = atol(attr);
				} else {
					/* Compatibility mode: no data in the store file, try to 
					 * stat the original file
					 */
					if (0 > stat(name, &fs)) {
						AEGIS_ERROR("%s: failed to stat '%s' (%s)", __func__, 
									name, strerror(errno));
					}
				}
				/* Everything is ok */
				m_contents[name].fs = fs;
			}

		} else if (0 == strcmp("link", node->name())) {
			m_links[node->attribute("name",true,"")] = 
				node->attribute("to",true,"");
		}
	}

    /* Check owner
     */
	string owner = xdoc.root()->attribute("owner", false, m_token.c_str());
	if (owner != m_token) {
        bool is_owner_valid;
        bool has_owner = has_token(owner.c_str(), &is_owner_valid);

        if (is_owner_valid) {
            /* A healthy environment. If the caller has both tokens,
             * it can take over the pstore and change the owner. 
			 * Do not change a into an appid, or try to take over
			 * a non-empty encrypted store.
             */
            if (has_owner) {
                if (valid_token && 0 != m_token.compare(0, 5, "AID::")
					&& (0 == m_contents.size() || prot_signed == m_prot))
				{
                    AEGIS_DEBUG(1, "%s: change owner: '%s' -> '%s'", 
                                __func__, owner.c_str(), m_token.c_str());
                    owner = m_token;
                    taken_over = true;
                } else {
                    AEGIS_DEBUG(1, "%s: use owner token '%s'", 
                                __func__, owner.c_str());
                    m_token = owner;
                }
				m_lock->writable(true);
            } else {
				if (prot_encrypted == m_prot) {
					AEGIS_ERROR("%s: token mismatch (owner=%s, given=%s),"
								" access_denied", __func__, owner.c_str(), 
								m_token.c_str());
					m_lock->unreadable();
					errno = EACCES;
					return;
				} else {
					AEGIS_DEBUG(1, "%s: token mismatch (owner=%s, given=%s),"
								" only reading allowed", __func__, owner.c_str(), 
								m_token.c_str());
					m_token = owner;
					m_lock->unwritable();
				}
            }
        } else {
            /* Unhealthy environment. Allow all access but 
             * don't change the token.
             */
            AEGIS_DEBUG(1, "%s: token mismatch (owner=%s, given=%s),"
                        " allow all access", 
                        __func__, owner.c_str(), m_token.c_str());
            m_token = owner;
			m_lock->writable(true);
        }
    } else if ("" == m_token) {
		AEGIS_DEBUG(1, "%s: empty token, only reading allowed", __func__);
		m_lock->unwritable();
	}

    /* Check integrity
     */
	string s_hash;
	bool hash_ok = internal_hash(NULL, s_hash);
	c_xmlnode *section = xdoc.root()->child("signature", false);

	if (!hash_ok || NULL == section) {
		/* Allow access but do not trust the current content
		 */
		AEGIS_ERROR("%s: cannot check integrity of '%s', reset", __func__, 
					m_filename.c_str());
		m_contents.clear();
		m_links.clear();
        errno = EACCES;
		if (prot_encrypted == m_prot)
			goto handle_key;
		else
			return;
	}

    if (0 != strcmp(GLOBAL_SIGNATURE, section->content())) {
        /* A genuine signature
         */
		aegis_crypto_result erc;
		struct aegis_signature_t signature;
		aegis_system_mode_t signing_mode;
		char* made_by = NULL;
        string sgn_made_by = owner;

		erc = aegis_crypto_string_to_signature(section->content(), &signature, &made_by);

        if (aegis_crypto_ok != erc) {
            AEGIS_ERROR("Invalid signature ('%s')", section->content());
			hash_ok = false;
            m_contents.clear();
            m_links.clear();

		} else {
			if (made_by) {
				if (0 != strcmp(made_by, owner.c_str()) && !taken_over) {
					/* Allow access but do not trust the current content
					 */
					AEGIS_ERROR("Signature made by wrong token '%s' owner is '%s'",
								made_by, owner.c_str());
					m_contents.clear();
					m_links.clear();
					hash_ok = false;
				}
				sgn_made_by.assign(made_by);
				free(made_by);
			} else {
				sgn_made_by = owner;
			}

			erc = aegis_crypto_verify(&signature, sgn_made_by.c_str(), 
									  s_hash.c_str(), strlen(s_hash.c_str()), 
									  &signing_mode);

			if (aegis_crypto_ok != erc) {
				/* Do not trust the existing content but allow adding new.
				 */
				AEGIS_ERROR("Signature does not verify ('%s' signed by '%s')",
							m_name.c_str(), owner.c_str());
				hash_ok = false;
				m_contents.clear();
				m_links.clear();
			} else {
				AEGIS_DEBUG(1, 
							"Signature verified ('%s' signed by '%s' in %s mode)", 
							m_name.c_str(), owner.c_str(), 
							signing_mode==aegis_system_protected?"protected":"unsafe");
				m_lock->readable(true);
			}
		}

    } else {
        if (aegis_crypto_ok != verify_global_signature(m_filename.c_str(), NULL)) {
            AEGIS_ERROR("Global signature does not verify");
            /* Do not trust the existing content but allow adding new.
             */
            m_contents.clear();
            m_links.clear();
			hash_ok = false;
        } else {
            AEGIS_DEBUG(1, "Global signature verified");
        }
    }

  handle_key:
	if (prot_encrypted == m_prot) {
        if (!m_lock->readable()) {
            AEGIS_ERROR("caller does not have '%s', access denied", 
						owner.c_str());
            return;
        }
		section = xdoc.root()->child("key", false);
		if (section) {
			m_symkey_len = base64_decode(section->content(), &m_symkey);
		} else {
			AEGIS_ERROR("%s: missing encryption key, clear", m_name.c_str());
			m_contents.clear();
			m_links.clear();
            if (!generate_new_symkey()) {
				m_lock->unreadable();
				errno = EACCES;
			}
		}
	}

	/* Check for version and convert the store if necessary
	 * and possible.
	 */
	m_lock->version = atoi(xdoc.root()->attribute("version", false, "0"));
	if (prot_encrypted == m_prot && 
		CURRENT_STORAGE_VERSION != m_lock->version 
		&& m_lock->writable()) 
	{
        if (!convert_store(save_filename.c_str())) {
			errno = EACCES;
		}
	}

	AEGIS_DEBUG(1, "%s: exit, %s read %s write", __func__, 
				m_lock->readable()?"can":"cannot",
				m_lock->writable()?"can":"cannot");

}


void
storage::reinitialize() 
{
	contents_map_t tmp_contents;
	std::map<std::string, std::string> tmp_links;

	/* Save the current contents which will be cleared
	 * in init_storage. TODO: Add a boolean variable
	 * to mark a storage dirty (contains uncommitted changes)
	 * because otherwise it's not necessary to do this.
	 */
	if (m_lock->writable()) {
		m_contents.swap(tmp_contents);
		m_links.swap(tmp_links);
	}

	/* TODO: There is a lot of initialization in init_storage
	 * which wouldn't be necessary the second time.
	 */
	if (m_token.length())
		init_storage(m_name.c_str(), m_token.c_str());
	else
		init_storage(m_name.c_str(), NULL);

	/* Merge contents and links. TODO: The logic here is faulty 
	 * and potentially dangerous. What really should be done is 
	 * to mark new additions made by the current process since
	 * the last commit and then merging only those. It would mean
	 * that the contents array should have a struct as an element
	 * with some metadata in it. It would also be a good idea to
	 * eliminate the m_links array.
	 */
	/* This is the new code that uses the new_file flag in the
	 * file metadata structure, so the comments above are obsolete
	 * and kept just for reference.
	 */
	if (m_lock->writable()) {
		contents_map_t::const_iterator ii;
		map<string, string>::const_iterator ij;
		for (ii = tmp_contents.begin();
			 ii != tmp_contents.end();
			 ii++) 
		{
			/* Check if the file to be merged was added after the commit and
			 * only then merge it.
			 */
			if (ii->second.new_file) {
				/* Merge if the file still exists.
				 */
				p_file *tmp = member(ii->first.c_str());
				if (0 == access(tmp->p_name(), R_OK)) {
					AEGIS_DEBUG(1, "%s: merge file '%s'", __func__, ii->first.c_str());
					m_contents[ii->first.c_str()].csum = ii->second.csum.c_str();
					m_contents[ii->first.c_str()].fs = ii->second.fs;
					/* Keep the new file flag since it wasn't commited anyway */
					m_contents[ii->first.c_str()].new_file = true;
				}
				delete tmp;
			}
		}
		for (ij = tmp_links.begin();
			 ij != tmp_links.end();
			 ij++)
		{
			if (m_links.end() == m_links.find(ij->first.c_str())) {
				/* Merge if the target of the link can be found from this same
				 * store. This logic fails if just the link was removed on 
				 * purpose or it points outside of the store.
				 */
				if (m_contents.end() != m_contents.find(ij->second.c_str())) {
					AEGIS_DEBUG(1, "%s: merge link '%s'", __func__, ij->first.c_str());
					m_links[ij->first.c_str()] = ij->second.c_str();
				}
			}
		}
	}
}


bool
storage::storage_changed()
{
	return m_lock->changed();
}


const char*
storage::store_sem_name(void)
{
	if (m_lock)
		return m_lock->sem_name.c_str();
	else
		return "";
}


storage::storage(const char* name, 
				 const char* owner, 
				 visibility_t visibility, 
				 protection_t protection) 
	: m_prot(protection), m_vis(visibility), m_symkey(NULL), m_symkey_len(0), m_lock(NULL)
{
	init_storage(name, owner);
}


storage::storage(const char* name, 
				 visibility_t visibility, 
				 protection_t protection) 
	: m_prot(protection), m_vis(visibility), m_symkey(NULL), m_symkey_len(0), m_lock(NULL)
{
	init_storage(name, NULL);
}


storage::~storage()
{
	AEGIS_ENTER;
	if (m_symkey)
		free(m_symkey);
	if (m_lock) 
		delete m_lock;
}


size_t
storage::get_files(stringlist& names)
{
    AEGIS_ENTER;
	size_t pos = 0;

    if (!IS_READABLE(this)) {
        AEGIS_ERROR("%s: access denied", __func__);
        errno = EACCES;
        return -1;
    }
	if (HAS_CHANGED(this)) {
		REINITIALIZE_STORE(this);
	}
	for (
		contents_map_t::const_iterator ii = m_contents.begin();
		ii != m_contents.end();
		ii++
	) {
		names.push_back(strdup(ii->first.c_str()));
		pos++;
	}
	for (
		map<string, string>::const_iterator ii = m_links.begin();
		ii != m_links.end();
		ii++
	) {
		names.push_back(strdup(ii->first.c_str()));
		pos++;
	}
	return(pos);
}


ssize_t
storage::get_ufiles(stringlist& names)
{
    AEGIS_ENTER;
	/* Allow reading the files list even if the caller doesn't
	 * have the encryption token if he has dac_override.
	 */
	bool old_rdlock = m_lock->readable();
	if (!old_rdlock) {
		creds_t pcreds = creds_gettask(0);
		if (has_dac_override(pcreds, 0)) {
			m_lock->readable(true);
		}
		creds_free(pcreds);
		if (!m_lock->readable()) {
			AEGIS_ERROR("%s: access denied", __func__);
			errno = EACCES;
			return -1;
		}

	}
	AEGIS_DEBUG(1, "%s: iterating files", __func__);
    for (
		 contents_map_t::const_iterator ii = m_contents.begin();
         ii != m_contents.end();
         ii++
    ) {
        if (prot_encrypted == m_prot) {
            p_file *item = member(ii->first.c_str());
			AEGIS_DEBUG(1, "%s: item %p %s", __func__, item, item?item->p_name():"N/A");
			if (item) {
				names.push_back(strdup(item->p_name()));
				delete item;
			}
        } else {
            names.push_back(strdup(ii->first.c_str()));
        }
    }
	if (file_exists(m_filename.c_str()))
		names.push_back(strdup(m_filename.c_str()));
	m_lock->readable(old_rdlock);
	AEGIS_DEBUG(1, "%s: total of %u files", __func__, (unsigned)names.size());
	return(names.size());
}


void
storage::release(stringlist &list)
{
    for (size_t i = 0; i < list.size(); i++) {
        free((void*)list[i]);
    }
    list.clear();
}


bool
storage::contains_file(const char* pathname)
{
    if (!IS_READABLE(this)) {
        AEGIS_ERROR("%s: access denied", __func__);
        errno = EACCES;
        return false;
    }
	if (HAS_CHANGED(this)) {
		REINITIALIZE_STORE(this);
	}
	string truename;
	contents_map_t::iterator ii;
	normalize_filename(pathname, truename);
	ii = m_contents.find(truename);
	return ii != m_contents.end();
}


bool
storage::hash_of_file(const char* pathname, string &hash)
{
    if (!IS_READABLE(this)) {
        AEGIS_ERROR("%s: access denied", __func__);
        errno = EACCES;
        return false;
    }
	if (HAS_CHANGED(this)) {
		REINITIALIZE_STORE(this);
	}
	string truename;
	contents_map_t::iterator ii;
	normalize_filename(pathname, truename);
	ii = m_contents.find(truename);
	if (m_contents.end() != ii) {
		hash.assign(ii->second.csum.c_str());
		return true;
	} else {
		return false;
	}
}


bool
storage::contains_link(const char* pathname)
{
    if (!IS_READABLE(this)) {
        AEGIS_ERROR("%s: access denied", __func__);
        errno = EACCES;
        return false;
    }
	if (HAS_CHANGED(this)) {
		REINITIALIZE_STORE(this);
	}
	map<string,string>::iterator ii;
	ii = m_links.find(pathname);
	return ii != m_links.end();
}


bool
storage::compute_digest(unsigned char* data, 
                        size_t bytes, 
                        char format, 
                        string& digest)
{
	unsigned char md[DIGESTLEN];
    DIGEST_CTX mdctx;
	int rc;

    if (!DIGEST_INIT(mdctx)) {
		AEGIS_ERROR("%s: DIGEST_INIT (%s)", __func__, strerror(errno));
		return(false);
    }
    if (DIGEST_OK != (rc = DIGEST_UPDATE(mdctx, data, bytes))) {
		AEGIS_DEBUG(1, "%s: DIGEST_UPDATE failed (%d)", __func__, rc);
		return(false);
    }
    if (DIGEST_OK != (rc = DIGEST_FINAL(mdctx, md))) {
		AEGIS_DEBUG(1, "%s: DIGEST_FINAL failed (%d)", __func__, rc);
		return(false);
    }
    if ('x' == format) {
        char tmp[3];
        digest.assign("");
        for (int i = 0; i < DIGESTLEN; i++) {
            snprintf(tmp, sizeof(tmp), "%02hx", md[i]);
            digest.append(tmp);
        }
    } else {
        char *tmp = base64_encode(md, DIGESTLEN);
        digest.assign(tmp);
        free(tmp);
    }
    return(true);
}


void
storage::add_file(const char* pathname)
{
    AEGIS_ENTER;
    if (!IS_MODIFIABLE(this)) {
        AEGIS_ERROR("%s: access denied", __func__);
        errno = EACCES;
        return;
    }
	if (HAS_CHANGED(this)) {
		REINITIALIZE_STORE(this);
	}
	p_file* pf;
	int flags = O_RDONLY;
	string truename;

	if (!pathname || 0 == strlen(pathname)) {
        AEGIS_ERROR("%s: invalid parameter (NULL)", __func__);
        errno = EINVAL;
		return;
	}
    AEGIS_DEBUG(1, "%s: %s", __func__, pathname);
	normalize_filename(pathname, truename);
	if (prot_encrypted == m_prot) {
		if (!contains_file(truename.c_str())) {
			flags = O_CREAT | O_RDWR | O_TRUNC;
		} else {
			AEGIS_ERROR("%s: store already contains file '%s'", 
						__func__, truename.c_str());
			return;
		}
	}
	pf = member(pathname);
	AEGIS_DEBUG(2, "Adding %s to storage", pf->name());
	if (pf) {
		if (pf->p_open(flags)) {
			AEGIS_DEBUG(2, "Files %s added to storage", pf->name());
			if (prot_signed == m_prot) {
				struct stat fs = {0};
				if (0 > pf->p_stat(&fs)){
					//TODO: What should be done about this error?
					AEGIS_DEBUG(2, "Failed to p_stat file!");
				}
				m_contents[pf->name()].fs = fs;
			}
			m_contents[pf->name()].csum = pf->digest();
			pf->p_close();
		}
		delete pf;
	}
}


void
storage::remove_file(const char* pathname)
{
	if (NULL == pathname) {
        AEGIS_ERROR("%s: invalid parameter (NULL)", __func__);
        errno = EINVAL;
		return;
	}
    if (!IS_MODIFIABLE(this)) {
        AEGIS_ERROR("%s: access denied", __func__);
        errno = EACCES;
        return;
    }
	if (HAS_CHANGED(this)) {
		REINITIALIZE_STORE(this);
	}
    AEGIS_DEBUG(1, "%s: %s", __func__, pathname);
	string truename;
	normalize_filename(pathname, truename);
	if (!contains_file(truename.c_str())) {
		AEGIS_DEBUG(0, "%s: '%s' not found", __func__, truename.c_str());
		return;
	}
    if (prot_encrypted == m_prot) {
        p_file *item = member(truename.c_str());
		if (item) {
			item->p_cleanup();
			delete item;
		}
	}
	m_contents.erase(m_contents.find(truename));

	/* Remove link automatically. TODO: this violates
     * the POSIX semantics, check if it really makes 
     * sense.
     */
	map<string,string>::iterator del_i, ii;
	for (
		ii = m_links.begin();
		ii != m_links.end();
	) {
		if (0 == strcmp(truename.c_str(), ii->second.c_str())) {
			del_i = ii;
			ii++;
			m_links.erase(del_i);
		} else
			ii++;
	}
}


void
storage::rename(const char *pathname, const char *to_this, p_file *use_member)
{
    AEGIS_ENTER;
	if (NULL == pathname || NULL == to_this) {
        AEGIS_ERROR("%s: invalid parameter (NULL)", __func__);
        errno = EINVAL;
		return;
	}
    if (!IS_MODIFIABLE(this)) {
        AEGIS_ERROR("%s: access denied", __func__);
        errno = EACCES;
        return;
    }
	if (HAS_CHANGED(this)) {
		REINITIALIZE_STORE(this);
	}
    AEGIS_DEBUG(1, "%s: %s -> %s", __func__, pathname, to_this);
    string digest;
	struct stat fs;
    string truename;
    string oldname;
    normalize_filename(pathname, truename);

	contents_map_t::iterator ii = m_contents.find(truename.c_str());
    if (m_contents.end() != ii) {
        oldname = truename;
		digest = ii->second.csum;
		fs = ii->second.fs;
        normalize_filename(to_this, truename);
        if (prot_encrypted == m_prot) {
            /* In an encrypted store also the target filename
             * needs to be changed.
             */
            pe_file *pf = NULL;
            bool ok;
            if (NULL != use_member) {
                pf = (pe_file*)use_member;
                ok = pf->change_pathname(truename.c_str());
            } else {
                pf = (pe_file*)member(oldname.c_str());
                if (pf) {
                    ok = pf->change_pathname(truename.c_str());
                    delete(pf);
                } else {
                    AEGIS_ERROR("Cannot find '%s'", oldname.c_str());
                    errno = ENOENT;
                    return;
                }
            }
            if (!ok) {
                AEGIS_ERROR("Rename '%s' to '%s' failed (%s)", 
                            oldname.c_str(), truename.c_str(), strerror(errno));
                if (0 == errno)
                    errno = ENOENT;
                return;
            }
        }
        m_contents.erase(ii);
        normalize_filename(to_this, truename);
		m_contents[truename.c_str()].csum = digest.c_str();
		if (prot_signed == m_prot) {
			m_contents[truename.c_str()].fs = fs;
		}
        AEGIS_DEBUG(1, "%s: file %s -> %s", __func__, oldname.c_str(), 
                    truename.c_str());
        return;
    }

	map<string, string>::iterator ij = m_links.find(pathname);
	if (m_links.end() != ij) {
		digest = ij->second;
		m_links.erase(ij);
        m_links[to_this] = digest;
        AEGIS_DEBUG(1, "%s: link %s -> %s", __func__, pathname, to_this);
        return;
    }
    AEGIS_ERROR("%s: '%s' not found", __func__, pathname);
}


void
storage::rename(const char *pathname, const char *to_this)
{
    if (!IS_MODIFIABLE(this)) {
        AEGIS_ERROR("%s: access denied", __func__);
        errno = EACCES;
        return;
    }
	if (HAS_CHANGED(this)) {
		REINITIALIZE_STORE(this);
	}
    rename(pathname, to_this, NULL);
}


void
storage::add_link(const char *pathname, const char *to_file)
{
    AEGIS_ENTER;
	if (!pathname || !to_file || 0 == strlen(pathname)) {
        AEGIS_ERROR("%s: invalid parameter (NULL)", __func__);
        errno = EINVAL;
		return;
	}
    if (!IS_MODIFIABLE(this)) {
        AEGIS_ERROR("%s: access denied", __func__);
        errno = EACCES;
        return;
    }
	if (HAS_CHANGED(this)) {
		REINITIALIZE_STORE(this);
	}
#ifdef INTELLIGENT_LINKS
    string target_name;
    normalize_filename(to_file, target_name);
    AEGIS_DEBUG(1, "%s: %s -> %s", __func__, pathname, target_name.c_str());
	map<string,string>::iterator ii;
	ii = m_contents.find(target_name.c_str());
	if (m_contents.end() == ii) {
		AEGIS_ERROR("%s: file not exists (%s)", __func__, target_name.c_str());
		return;
	}
	m_links[pathname] = target_name.c_str();
#else
    AEGIS_DEBUG(1, "%s: %s -> %s", __func__, pathname, to_file);
	m_links[pathname] = to_file;
#endif
}


void
storage::remove_link(const char* pathname)
{
	if (!pathname) {
        AEGIS_ERROR("%s: invalid parameter (NULL)", __func__);
        errno = EINVAL;
		return;
	}
    if (!IS_MODIFIABLE(this)) {
        AEGIS_ERROR("%s: access denied", __func__);
        errno = EACCES;
        return;
    }
	if (HAS_CHANGED(this)) {
		REINITIALIZE_STORE(this);
	}
    AEGIS_DEBUG(1, "%s: %s", __func__, pathname);
    map<string, string>::iterator ii = m_links.find(pathname);
    if (m_links.end() != ii)
        m_links.erase(ii);
    else
        AEGIS_DEBUG(1, "%s: %s not found!", __func__, pathname);
}


void
storage::read_link(const char* pathname, string& points_to)
{
    AEGIS_ENTER;
	if (!pathname) {
        AEGIS_ERROR("%s: invalid parameter (NULL)", __func__);
        errno = EINVAL;
		return;
	}
	if (HAS_CHANGED(this)) {
		REINITIALIZE_STORE(this);
	}
    if (!IS_READABLE(this)) {
        AEGIS_ERROR("%s: access denied", __func__);
        errno = EACCES;
        return;
    }
    AEGIS_DEBUG(1, "%s: %s", __func__, pathname);
	map<string,string>::iterator ii;
	ii = m_links.find(pathname);
	if (m_links.end() != ii) {
		points_to.assign(ii->second.c_str());
	}
}


bool 
storage::verify_file(const char* pathname)
{
    AEGIS_ENTER;
	if (!pathname) {
        AEGIS_ERROR("%s: invalid parameter (NULL)", __func__);
        errno = EINVAL;
		return false;
	}
	if (HAS_CHANGED(this)) {
		REINITIALIZE_STORE(this);
	}
    if (!IS_READABLE(this)) {
        AEGIS_ERROR("%s: access denied", __func__);
        errno = EACCES;
        return false;
    }

    AEGIS_DEBUG(1, "%s: %s", __func__, pathname);
	p_file* pf;
	contents_map_t::iterator ii;
	map<string, string>::iterator ij;
	bool res = false;
	const char* realname = pathname;

	ij = m_links.find(pathname);
	if (m_links.end() != ij)
		realname = ij->second.c_str();

	pf = member(realname);
	if (NULL == pf) {
        AEGIS_DEBUG(1, "%s: file not found", __func__);
        errno = ENOENT;
        return false;
	}
	ii = m_contents.find(pf->name());

	if (m_contents.end() == ii) {
		AEGIS_DEBUG(0, "%s: '%s' not found", __func__, pf->name());
		delete pf;
		return false;
	}

	if (pf->p_open(O_RDONLY)) {
		if (0 == (strcmp(pf->digest(), ii->second.csum.c_str()))) {
			res = true;
		} else {
			AEGIS_DEBUG(1, "%s: '%s' fail, computed '%s' != stored '%s'", __func__, 
						pathname, pf->digest(), ii->second.csum.c_str());
		}
		pf->p_close();

	} else {
		AEGIS_DEBUG(1, "%s: '%s' fail (%s)", __func__, pathname, strerror(errno));
	}

	delete pf;
	return res;
}


bool 
storage::verify_content(const char* pathname, 
						unsigned char* data, 
						size_t of_len)
{
    AEGIS_DEBUG(1, "%s: '%s' of %lu", __func__, pathname, 
				(unsigned long)of_len);

	if (!pathname || !data) {
        AEGIS_ERROR("%s: invalid parameter (NULL)", __func__);
        errno = EINVAL;
		return -1;
	}
	if (HAS_CHANGED(this)) {
		REINITIALIZE_STORE(this);
	}
    if (!IS_READABLE(this)) {
        AEGIS_ERROR("%s: access denied", __func__);
        errno = EACCES;
        return false;
    }

	string digest1;
    compute_digest(data, of_len, 'b', digest1);

	if (digest1 == m_contents[pathname].csum.c_str()) {
		AEGIS_DEBUG(1, "%s: %s matches with index, return OK", 
					__func__, digest1.c_str());
		return true;
	}

	/* Allow verifying just n bytes from the start
	 */
	string digest2;
    RAWDATA_PTR idata = NULL;
    size_t ilen = 0;
    if (0 == get_file(pathname, &idata, &ilen)) {
        if (of_len <= ilen)
            compute_digest((unsigned char*)idata, of_len, 'b', digest2);
		else {
			AEGIS_DEBUG(1, "%s: real len is %lu, fail", __func__, 
						(unsigned long)ilen);
			return false;
		}
		/* Do not leave traces in memory
		 */
        memset(idata, '\0', ilen);
        aegis_crypto_free(idata);
    }
    return digest1 == digest2;
}


bool
storage::internal_hash(struct aegis_digest_t* as_bin, string& to_this)
{
	DIGEST_CTX mdctx;
	char *tmp = NULL;
	unsigned char signmd[DIGESTLEN];
	contents_map_t::const_iterator ii;
	map<string, string>::const_iterator ij;
	int rc;
	
	AEGIS_DEBUG(3, "%s: enter %s, %s", __func__, m_filename.c_str(), 
                m_token.c_str());

	if (!DIGEST_INIT(mdctx)) {
		AEGIS_ERROR("%s: DIGEST_INIT failed (%s)", __func__, strerror(errno));
		return false;
	}

	/* Include absolute pathname and storage path if defined
	 * in the signature to prevent renaming.
	 */
    if (DIGEST_OK != DIGEST_UPDATE(mdctx, m_filename.c_str(), 
                                   strlen(m_filename.c_str()))) 
	{
		AEGIS_ERROR("%s: DIGEST_UPDATE failed (%s)", __func__, strerror(errno));
		return false;
    }

#if 0
	/* Storage path should be included too but at the moment
	 * it would break too many things.
	 */
	if ("" != m_lock->m_storage_path) {
		if (DIGEST_OK != DIGEST_UPDATE(mdctx, m_lock->m_storage_path.c_str(), 
                                   strlen(m_lock->m_storage_path.c_str()))) 
		{
			AEGIS_ERROR("%s: DIGEST_UPDATE failed (%s)", __func__, strerror(errno));
			return false;
		}
	}
#endif

	/* TODO: Should include owner token too. 
	 */

	/* Be careful to do this in the same order every time. This code 
	 * relies to that the elements go in the map in the same order 
	 * every time, regardless of the order they are being added 
	 * to it. Luckily STL maps always seem to iterate the same way,
	 * in alphabetical order.
	 */
	for (
		ii = m_contents.begin();
		ii != m_contents.end();
		ii++
	) {
		tmp = (char*)ii->first.c_str();
		if (NULL != tmp)
			DIGEST_UPDATE(mdctx, tmp, strlen(tmp));
		tmp = (char*)ii->second.csum.c_str();
		if (NULL != tmp)
			DIGEST_UPDATE(mdctx, tmp, strlen(tmp));
	}

	for (
		ij = m_links.begin();
		ij != m_links.end();
		ij++
	) {
		tmp = (char*)ij->first.c_str();
		if (NULL != tmp)
			DIGEST_UPDATE(mdctx, tmp, strlen(tmp));
		tmp = (char*)ij->second.c_str();
		if (NULL != tmp)
			DIGEST_UPDATE(mdctx, tmp, strlen(tmp));
	}

	rc = DIGEST_FINAL(mdctx, signmd);

	if (DIGEST_OK == rc) {

		if (NULL != as_bin)
			memcpy(as_bin, signmd, sizeof(struct aegis_digest_t));

		char* tmp = base64_encode(signmd, DIGESTLEN);
		to_this.assign(tmp);
		free(tmp);
		AEGIS_DEBUG(3, "%s: hash=%s", __func__, to_this.c_str());
		return true;

	} else {
		AEGIS_ERROR("%s: error %d from DIGEST_FINAL", __func__, rc);
		return false;
	}
}


bool
storage::commit(void)
{
    AEGIS_ENTER;
    bool valid_token = false;

    if (!IS_MODIFIABLE(this) || !has_token(m_token.c_str(), &valid_token)) {
		AEGIS_ERROR("%s: access denied, cannot commit '%s'", __func__, filename());
        errno = EACCES;
        return false;
    }

	c_xmldoc xdoc;
    string s_sign;
	c_xmlnode *xnode = xdoc.create("pstore");
	c_xmlnode *section = xnode->append_child("envelope");
	contents_map_t::iterator ii;
	map<string, string>::const_iterator ij;
	struct aegis_digest_t bhash;

	lock_store();

	if (HAS_CHANGED(this)) {
		REINITIALIZE_STORE(this);
	}

	if (!internal_hash(&bhash, s_sign)) {
		AEGIS_ERROR("%s: cannot compute digest", __func__);
		unlock_store();
		errno = EACCES;
		return false;
	}

    if (valid_token) {
        aegis_crypto_result erc;
        struct aegis_signature_t signature;

        /* Sign the digest. If this returns an error it's because
         * we don't have access to the token.
         */
        erc = aegis_crypto_sign(s_sign.c_str(), strlen(s_sign.c_str()),
                                m_token.c_str(), &signature);
        
        if (aegis_crypto_ok == erc) {
            char* tmp = NULL;

            aegis_crypto_signature_to_string(&signature, 
                                             aegis_as_base64, 
                                             m_token.c_str(), 
                                             &tmp);
            if (NULL != tmp) {
                s_sign.assign(tmp);
                free(tmp);
            } else {
				AEGIS_ERROR("%s: memory error? (%s)", __func__, strerror(errno));
				errno = ENOMEM;
				unlock_store();
                return false;
            }
        } else {
			AEGIS_ERROR("%s: Cannot make signature", __func__);
			errno = EACCES;
			unlock_store();
            return false;
        }
    } else {
        s_sign.assign(GLOBAL_SIGNATURE);
    }

    xnode->append_attribute("owner", m_token.c_str());
	char vers_str[20];
	snprintf(vers_str, sizeof(vers_str), "%d", CURRENT_STORAGE_VERSION);
    xnode->append_attribute("version", vers_str);
	if (prot_encrypted == m_prot 
		&& 0 != m_lock->m_storage_path.compare(DEFAULT_CRYPTO_ROOT))
		xnode->append_attribute("storage-path", m_lock->m_storage_path.c_str());
	m_lock->flush_open_files();

	for (
		ii = m_contents.begin();
		ii != m_contents.end();
		ii++
	) {
		c_xmlnode* ifile = section->append_child("file");
		ifile->append_attribute("name", (char*)ii->first.c_str());
		ifile->append_attribute("hash", (char*)ii->second.csum.c_str());
		/* Extra metadata for signed stores */
		if (prot_signed == m_prot) {
			ifile->append_attribute("size", (long)ii->second.fs.st_size);
			ifile->append_attribute("uid", (long)ii->second.fs.st_uid);
			ifile->append_attribute("gid", (long)ii->second.fs.st_gid);
			ifile->append_attribute("mode", (long)ii->second.fs.st_mode);
			ifile->append_attribute("mtime", (long)ii->second.fs.st_mtime);
			ifile->append_attribute("atime", (long)ii->second.fs.st_atime);
			ifile->append_attribute("ctime", (long)ii->second.fs.st_ctime);
		}
		/* Reset the new file flag */
		ii->second.new_file = false;

	}

	for (
		ij = m_links.begin();
		ij != m_links.end();
		ij++
	) {
		c_xmlnode* ifile = section->append_child("link");
		ifile->append_attribute("name", (char*)ij->first.c_str());
		ifile->append_attribute("to", (char*)ij->second.c_str());
	}

	section = xnode->append_child("signature");
	section->append_content(s_sign.c_str());
    
	if (prot_encrypted == m_prot) {
		char* key;
		section = xnode->append_child("key");
		key = base64_encode(m_symkey, m_symkey_len);
		section->append_content(key);
		free(key);
	}

	AEGIS_DEBUG(1, "%s: saving to '%s' (%d files, %d links)", __func__, 
				m_filename.c_str(), (int)m_contents.size(), (int)m_links.size());

	if (!xdoc.save(m_filename.c_str())) {
		AEGIS_ERROR("%s: failed to commit '%s'", __func__, m_filename.c_str());
		unlock_store();
		return false;
	}

    if (aegis_system_plain == aegis_current_mode()) {
        save_global_signature(m_filename.c_str(), NULL);
    }

	errno = 0;
	m_lock->mark(&bhash);
	unlock_store();
	return true;
}


bool
storage::refresh(void)
{
    if (!IS_READABLE(this)) {
		AEGIS_ERROR("%s: access denied", __func__);
        errno = EACCES;
        return false;
    }
	if (HAS_CHANGED(this)) {
		REINITIALIZE_STORE(this);
		return true;
	}
	return false;
}


bool
storage::set_aes_keys(bool is_protected, AES_KEY *to_this)
{
	RAWDATA_PTR plakey = NULL;
	size_t plainsize = 0;
	bool res = true;
    aegis_crypto_result erc;

    if (NULL == m_symkey || NULL == m_lock)
        return false;

    AEGIS_ENTER;

	if (0 < m_lock->keypack.refcount) {
		m_lock->keypack.refcount++;
		AEGIS_DEBUG(1, "%s: keys already set, refcount %d", __func__, 
					m_lock->keypack.refcount);
		return true;
	}

	if (is_protected) {
		erc = aegis_crypto_decrypt(m_symkey, m_symkey_len, m_token.c_str(), 
								   &plakey, &plainsize);
		if (aegis_crypto_ok != erc) {
			AEGIS_ERROR("%s: cannot decrypt (%s)", __func__, 
						aegis_crypto_last_error_str());
			res = false;
		}
	} else {
		/* The key is plaintext, for testing purposes only
		 */
		plainsize = m_symkey_len;
		plakey = malloc(plainsize);
		memcpy(plakey, m_symkey, plainsize); 
	}
	if ((AES_KEY_BITLEN >> 2) > plainsize) {
		AEGIS_ERROR("%s: not enough keymaterial (%d)", __func__, (int)plainsize);
		res = false;
	}
    if (res) {
		int rc;
		rc = AES_set_encrypt_key((unsigned char*)plakey, AES_KEY_BITLEN, 
								 &m_lock->keypack.enc_key);
		if (0 != rc) {
			AEGIS_ERROR("%s: AES_set_encrypt_key failed (%d)", __func__, rc);
			res = false;
		}
		rc = AES_set_decrypt_key((unsigned char*)plakey, AES_KEY_BITLEN, 
								 &m_lock->keypack.dec_key);
		if (0 != rc) {
			AEGIS_ERROR("%s: AES_set_decrypt_key failed (%d)", __func__, rc);
			res = false;
		}
		rc = AES_set_encrypt_key((unsigned char*)plakey + (AES_KEY_BITLEN >> 3),
								 AES_KEY_BITLEN, &m_lock->keypack.twk_key);
		if (0 != rc) {
			AEGIS_ERROR("%s: AES_set_encrypt_key failed (%d)", __func__, rc);
			res = false;
		}
		/* If given, set also the AES256 key. This is needed for conversion
		 * from the old format which used that algorithm.
		 */
		if (NULL != to_this) {
			rc = AES_set_encrypt_key((unsigned char*)plakey, 256, to_this);
			if (0 != rc) {
				AEGIS_ERROR("%s: AES_set_encrypt_key failed (%d)", __func__, rc);
				res = false;
			}
		}
	}
	if (plakey) {
		memset(plakey, '\0', plainsize);
		free(plakey);
    }
	if (res) {
		m_lock->keypack.refcount = 1;
	} else {
		memset(&m_lock->keypack, '\0', sizeof(m_lock->keypack));
	}
	return(res);
}


void
storage::clear_aes_keys(void)
{
	if (1 < m_lock->keypack.refcount) {
		m_lock->keypack.refcount--;
		AEGIS_DEBUG(1, "%s: keys still in use, refcount %d", __func__, 
					m_lock->keypack.refcount);
		return;
	}
	memset(&m_lock->keypack, '\0', sizeof(m_lock->keypack));
	AEGIS_DEBUG(1, "%s: cleared keys", __func__);
}


int
storage::get_file(const char* pathname, RAWDATA_RPTR to_buf, size_t* bytes)
{
    AEGIS_ENTER;

	if (!pathname || !to_buf || !bytes) {
        AEGIS_ERROR("%s: invalid parameter (NULL)", __func__);
        errno = EINVAL;
		return -1;
	}
	if (HAS_CHANGED(this)) {
		REINITIALIZE_STORE(this);
	}
    if (!IS_READABLE(this)) {
        AEGIS_ERROR("%s: access denied", __func__);
        errno = EACCES;
        return -1;
    }
	p_file* pf = NULL;
	ssize_t rlen;
	map<string,string>::iterator ii;
	int rc = 0;
    string truename;

	*to_buf = NULL;
    *bytes = 0;
    normalize_filename(pathname, truename);
    
	AEGIS_DEBUG(1, "%s: %s -> %s", __func__, pathname, truename.c_str());

	ii = m_links.find(pathname);
	if (m_links.end() != ii) {
        pf = member(ii->second.c_str());
    } else if (m_contents.end() != m_contents.find(truename.c_str())) {
        pf = member(truename.c_str());
    } 
	if (NULL == pf) {
        AEGIS_DEBUG(1, "%s: file not found", __func__);
        errno = ENOENT;
        return -1;
    }
	if (pf->p_open(O_RDONLY)) {
        /* TODO: mlock the buffer in memory in case of
         * an encrypted file.
         */
		*to_buf = (RAWDATA_PTR)malloc(pf->datasize());
		rlen = pf->p_read(0, *to_buf, pf->datasize());
		if (rlen < (ssize_t)pf->datasize()) {
			AEGIS_ERROR("Could not read from '%s'", pathname);
		}
		AEGIS_DEBUG(1, "%s: size %d", __func__, pf->datasize());
		*bytes = rlen;
		pf->p_close();
	} else {
		AEGIS_ERROR("Could not open '%s' (%s)", pathname, strerror(errno));
		rc = -1;
	}
	delete pf;
	return(rc);

}


int
storage::put_file(const char* pathname, RAWDATA_PTR data, size_t bytes)
{
	if (!pathname || 0 == strlen(pathname)) {
        AEGIS_ERROR("%s: invalid parameter (NULL)", __func__);
        errno = EINVAL;
		return -1;
	}
    if (!IS_MODIFIABLE(this)) {
        AEGIS_ERROR("%s: access denied", __func__);
        errno = EACCES;
        return -1;
    }
	if (HAS_CHANGED(this)) {
		REINITIALIZE_STORE(this);
	}
    AEGIS_DEBUG(1, "%s: %s", __func__, pathname);

	p_file* pf;
	ssize_t rlen;
	int rc = 0;

	/* TODO: Write to a temporary name and rename
	 * if successful
	 */
	pf = member(pathname);
	if (NULL == pf) {
        AEGIS_DEBUG(1, "%s: file not found", __func__);
        errno = ENOENT;
        return -1;
	}
	if (pf->p_open(O_RDWR | O_CREAT | O_TRUNC)) {
        if (data && 0 < bytes) {
            rlen = pf->p_write(0, data, bytes);
            if (rlen < (ssize_t)bytes) {
                AEGIS_ERROR("Could not write to '%s'", pathname);
            }
        }
		pf->p_close();
	} else {
		AEGIS_ERROR("Could not open '%s' (%s)", pathname, strerror(errno));
		rc = -1;
	}
	delete pf;
	return(rc);
}


void
storage::release_buffer(RAWDATA_PTR buf)
{
	if (buf) 
		free(buf);
}


int
storage::nbrof_files(void)
{
	if (HAS_CHANGED(this)) {
		REINITIALIZE_STORE(this);
	}
	if (IS_READABLE(this))
		return(m_contents.size() + m_links.size());
	else
		return -1;
}


int
storage::nbrof_links(void)
{
	if (HAS_CHANGED(this)) {
		REINITIALIZE_STORE(this);
	}
	return(m_links.size());
}


const char* 
storage::name(void)
{
	return m_name.c_str();
}


const char*
storage::token(void)
{
	return m_token.c_str();
}


const char* 
storage::filename(void)
{
	return m_filename.c_str();
}


int
storage::iterate_storage_names(storage::visibility_t of_visibility, 
							   storage::protection_t of_protection, 
							   const char* matching_names,
							   aegis_callback* cb_func,
							   void* ctx)
{
    AEGIS_ENTER;

	string directory_name;
	
	if (0 != get_storage_directory(of_visibility, of_protection, directory_name)) {
		AEGIS_ERROR("Cannot get storage name");
		return(-ENOENT);
	}
	AEGIS_DEBUG(1, "%s: iterating '%s' in '%s'", __func__, matching_names, 
				   directory_name.c_str());
	return (iterate_files(directory_name.c_str(), matching_names, cb_func, ctx));
}


int
storage::stat_file(const char* pathname, struct stat *stbuf)
{
	if (NULL == pathname) {
        AEGIS_ERROR("%s: invalid parameter (NULL)", __func__);
        errno = EINVAL;
		return -1;
	}
    if (!IS_READABLE(this)) {
        AEGIS_ERROR("%s: access denied", __func__);
        errno = EACCES;
        return -1;
    }
	if (HAS_CHANGED(this)) {
		REINITIALIZE_STORE(this);
	}
    AEGIS_DEBUG(1, "%s: %s", __func__, pathname);
    int rc = 0;
    bool is_link = false;
	const char* realname = pathname;
	map<string,string>::iterator ii;

	ii = m_links.find(pathname);
	if (m_links.end() != ii) {
		realname = ii->second.c_str();
        is_link = true;
    }
	p_file *pf = member(realname);
    if (pf) {
        rc = pf->p_stat(stbuf);
        if (0 == rc)
            AEGIS_DEBUG(1, "%s: size %d", __func__, (int)stbuf->st_size);
        delete pf;
    } else {
        errno = ENOENT;
        AEGIS_DEBUG(1, "%s: file not found", __func__);
        return -1;
    }
    if (is_link) {
        stbuf->st_mode |= (S_IFLNK | S_IRWXU | S_IRWXG | S_IRWXO);
    }
    return rc;
}


p_file* 
storage::member(const char* pathname)
{
    if (NULL == pathname)
        return NULL;
    if (!IS_READABLE(this)) {
        AEGIS_ERROR("%s: access denied", __func__);
        errno = EACCES;
        return NULL;
    }
	if (HAS_CHANGED(this)) {
		REINITIALIZE_STORE(this);
	}
    string truename;
    normalize_filename(pathname, truename);
    if (prot_encrypted == m_prot)
		return new pe_file(this, truename.c_str());
	else
		return new p_file(this, truename.c_str());
}


storage*
p_file::owner()
{
    return m_owner;
}

void p_file::update_metadata(struct stat *fs)
{
	AEGIS_ENTER;
	CHECK_VALID(this);
	/* Update only existing files */
	if (m_owner->contains_file(name())) {
		m_owner->m_contents[name()].fs = *fs;
	}
}

p_file::p_file(storage* owner, const char* pathname)
	: m_owner(owner), m_fd(-1), m_size(0), m_data(MAP_FAILED), m_mapsize(0), 
	  m_new_file(false), m_omode(om_closed) // , m_lock(NULL)
{
	if (!pathname || !owner) {
        AEGIS_ERROR("%s: invalid parameter (NULL)", __func__);
        errno = EINVAL;
		return;
	}
	m_semname = owner->store_sem_name();
	owner->normalize_filename(pathname, m_name);
	AEGIS_DEBUG(1, "%s: '%s' by '%s'", __func__, m_name.c_str(), owner->name());
}


int
p_file::p_rename(const char* new_name)
{
    AEGIS_ENTER;
	CHECK_VALID_RET(this, -1);
	if (!new_name) {
        AEGIS_ERROR("%s: invalid parameter (NULL)", __func__);
        errno = EINVAL;
		return -1;
	}
    if (!IS_MODIFIABLE(m_owner)) {
        AEGIS_ERROR("%s: access denied", __func__);
        errno = EACCES;
        return -1;
    }
    string m_oldname = m_name;
	m_owner->rename(m_name.c_str(), new_name, this);
    m_owner->normalize_filename(new_name, m_name);
    return 0;
}


void
pe_file::set_pathname(const char* to_this, bool convert)
{
	if (!to_this) {
        AEGIS_ERROR("%s: invalid parameter (NULL)", __func__);
        errno = EINVAL;
		return;
	}
	string seedname;
	string hashname;

	/* Generate a unique name for the real data file
	 * TODO: Storage attributes are not included in the seedname
	 * so the same filename in two stores with the same name but
	 * different storage attributes will collide!
	 */
	seedname.assign(m_owner->name());
	seedname.append(":");
	seedname.append(to_this);
	m_owner->compute_digest((unsigned char*)seedname.c_str(), 
							seedname.size(), 'b', hashname);

	/* Remove non-filename characters from the filename.
	 * TODO: easier way to do this?
	 */
	char *tmp = (char*)malloc(hashname.size() + 1);
	strcpy(tmp, hashname.c_str());
	for (char *c = tmp; *c; c++) {
		if ('/' == *c)
			*c = '-';
		else if ('=' == *c) {
			*c = '\0';
			break;
		}
	}
	AEGIS_DEBUG(1, "%s: storage path is '%s'", __func__, 
				m_owner->m_lock->m_storage_path.c_str());
	m_actname.assign(m_owner->m_lock->m_storage_path.c_str());
    if (!directory_exists(m_actname.c_str())) {
        if (0 != create_directory(m_actname.c_str(), 0777))
            AEGIS_ERROR("%s: %s does not exist and cannot be created", 
                        __func__, crypto_root);
    }

#ifdef USE_PRIVATE_DIRS
	/* Add uid if private store
	 */
	if (storage::vis_private == m_owner->m_vis) {
		char uids[20];
		snprintf(uids, sizeof(uids), "%06d", geteuid());
		m_actname.append(uids);
		if (!directory_exists(m_actname.c_str())) {
			int rc = create_directory(m_actname.c_str(), 0700);
			if (0 > rc) {
				AEGIS_ERROR("%s: cannot create directory '%s' (%s)", 
							__func__, m_actname.c_str(), strerror(errno));
				m_actname.assign(crypto_root);
			}
		}
		m_actname.append("/");
	}
#endif

	if (convert)
		hashname = m_actname;
	m_actname.append(tmp);
	m_actname.append(".dat");
    AEGIS_DEBUG(1, "%s: m_file(%p) %s -> %s", __func__, this, 
                seedname.c_str(), m_actname.c_str());
	free(tmp);

	/* Check with encrypted stores if the index file needs to be
	 * moved into a new location.
	 */
	if (convert) {
		string old_file_name = m_actname;
		old_file_name.erase(0, strlen(hashname.c_str()));
		old_file_name.insert(0, OLD_CRYPTO_ROOT);
		AEGIS_DEBUG(1, "%s: check for '%s'", __func__, old_file_name.c_str());
		if (file_exists(old_file_name.c_str())) {
			if (!m_owner->move_file(old_file_name.c_str(), m_actname.c_str()))
				AEGIS_ERROR("%s: cannot move '%s' to '%s'", __func__, 
							old_file_name.c_str(), m_actname.c_str());
		}
		hashname = m_actname;
	}
#ifndef USE_PRIVATE_DIRS
	/* Make distinction between private and shared
	 * stores to avoid name collisions.
	 */
	if (storage::vis_private == m_owner->m_vis) {
		m_actname.append("p");
	} else {
		m_actname.append("s");
	}
	if (convert)
		if (0 > rename(hashname.c_str(), m_actname.c_str()))
			AEGIS_ERROR("%s: cannot move '%s' to '%s' (%s)", __func__, 
						hashname.c_str(), m_actname.c_str(), strerror(errno));
#endif
}


bool
pe_file::change_pathname(const char* to_this)
{
    string old_actname = m_actname;
    set_pathname(to_this, false);
    int rc = rename(old_actname.c_str(), m_actname.c_str());
    if (0 == rc) {
        AEGIS_DEBUG(1, "%s: data %s -> %s", __func__, 
                    old_actname.c_str(), m_actname.c_str());
        return true;
    } else {
        AEGIS_ERROR("%s: cannot rename %s -> %s (%s)", __func__, 
                    old_actname.c_str(), m_actname.c_str(),
                    strerror(errno));
        return false;
    }
}


#if 0
int
pe_file::p_rename(const char* new_name)
{
	if (!new_name) {
        AEGIS_ERROR("%s: invalid parameter (NULL)", __func__);
        errno = EINVAL;
		return -1;
	}
	AEGIS_DEBUG(1, "%s: '%s -> %s'", __func__, m_name.c_str(), new_name);

    string oldname = m_name;
    string oldactname = m_actname;
    string truename;

    m_owner->normalize_filename(new_name, truename);
    set_pathname(truename.c_str(), false);
    AEGIS_DEBUG(1, "%s: %s -> %s (%s -> %s)", __func__,
                m_name.c_str(), truename.c_str(),
                oldactname.c_str(), m_actname.c_str());

    int rc = rename(oldactname.c_str(), m_actname.c_str());

    if (0 == rc) {
        m_owner->rename(oldname.c_str(), truename.c_str(), this);
        m_name = truename;
        return 0;
    } else {
        AEGIS_DEBUG(1, "%s: rename failed(%s)", __func__, strerror(errno));
        m_actname = oldactname;
        return rc;
    }
}
#endif


pe_file::pe_file(storage* store, const char* pathname)
	: p_file(store, pathname)
{
    AEGIS_ENTER;
	memset(&m_hdr, '\0', sizeof(m_hdr));
	m_hdr.fs.st_size = (off_t)-1;
	if (!pathname || !store) {
        AEGIS_ERROR("%s: invalid parameter (NULL)", __func__);
        errno = EINVAL;
		return;
	}
    set_pathname(pathname, false);
}


bool
p_file::int_open(int flags, bool lock)
{
	struct stat fs;
	int map_prot = 0, accflags = flags;
    mode_t mode = 0;

	AEGIS_DEBUG(1, "%s: open %s flags %x", __func__, name(), flags);

	accflags &= O_ACCMODE;
	if (O_RDONLY == accflags) {
		m_omode = om_readonly;
		map_prot = PROT_READ;

	} else if (O_RDWR == accflags) {
		m_omode = om_readwrite;
		map_prot = PROT_READ | PROT_WRITE;

	} else if (O_WRONLY == accflags) {
		/* Change to read&write in order to be able
		 * to compute hash by reading.
		 */
		m_omode = om_writeonly;
		flags |= O_RDWR;
		flags ^= O_WRONLY;
		map_prot = PROT_READ | PROT_WRITE;
	} else {
		AEGIS_ERROR("%s: odd flags %03o", __func__, flags);
        errno = EINVAL;
		return false;
	} 
	/* Discard the non-standard O_RECOVER flag
	 */
	if (flags & O_RECOVER) {
		flags ^= O_RECOVER;
	}
    if ((m_omode != om_readonly && !IS_MODIFIABLE(m_owner))
		|| ((flags & O_CREAT) && !IS_MODIFIABLE(m_owner))
        || (!IS_READABLE(m_owner))) 
    {
        AEGIS_ERROR("%s: access denied", __func__);
        errno = EACCES;
        return false;
    }

	mode_t omask = umask(0);
	/* Check if the file exists already, and if so,
	 * get its size. A bit of a toctou.
	 */
#if 0
    uid_t uid = geteuid();
    if (0 == uid) {
        /* TODO: Use world-writable flags for a while
         * until the usage pattern becomes clear. A better
		 * approach would be to create private files with
		 * 0600 and shared files with 0666. A
         */
        mode = 0666;
    } else {
        mode = 0600;
	}
#else
	if (storage::vis_private == m_owner->m_vis) {
		mode = 0600;
	} else {
		mode = 0666;
	}
#endif

	memset(&fs, '\0', sizeof(fs));
	if (0 > stat(p_name(), &fs)) {
        if (ENOENT == errno) {
			AEGIS_DEBUG(1, "%s: new file '%s'", __func__, p_name());
            m_new_file = true;
            m_size = 0;
			if (om_readonly == m_omode 
				&& storage::prot_encrypted == m_owner->m_prot) 
			{
				/* Make the mapping writable to store the attributes
				 * at the start.
				 */
				AEGIS_DEBUG(1, "%s: make writable", __func__);
				flags |= O_RDWR;
				map_prot = PROT_READ | PROT_WRITE;
			}
        } else {
            AEGIS_ERROR("%s: cannot stat '%s' (%s)", __func__, 
                        p_name(), strerror(errno));
            return false;
        }
	} else {
		/* Do not truncate encrypted files
		 */
		m_size = fs.st_size;
		/* If the file is in the store the actual file size should be used
		 * from the metadata
		 */
		if (storage::prot_signed == m_owner->m_prot && 
			m_owner->contains_file(name())){
				m_size = owner()->m_contents[name()].fs.st_size;
		}
		if (O_TRUNC & flags) {
			if (storage::prot_encrypted == m_owner->m_prot)
				flags ^= O_TRUNC;
			else
				m_size = 0;
			m_new_file = true;
		} else {
			m_new_file = false;
		}
	}

    m_fd = open(p_name(), flags, mode);
	umask(omask);

	if (0 > m_fd) {
		AEGIS_ERROR("Cannot open '%s' (%s)", p_name(), strerror(errno));
		m_omode = om_closed;
		return false;
	}

	/* a lock has been requested for the file, only get it
	 * if the file is to be opened for writing 
	 */
	if (true == lock && m_omode != om_readonly) {
		/* Request a exclusive non-blocking lock for the file.
		 * Note: The lock is specific to the file descriptor so
		 * there is no need to explicitly remove it 
		 */
		if (0 > flock(m_fd, LOCK_EX | LOCK_NB)) {
			AEGIS_ERROR("Failed to get lock on '%s' (%s)", 
						name(), strerror(errno));
			m_omode = om_closed;
			close(m_fd);
			return false;
		}
		AEGIS_DEBUG(2, "%s locked", p_name());
	}

	AEGIS_DEBUG(1, "%s: '%s' size %d, blocksize is %d", __func__, p_name(),
                (int)m_size, (int)fs.st_blksize);

	if ((flags & O_ACCMODE) != O_RDONLY) {
        /*
         * Read and write, map to the nearest 4 kB boundary
         */
        m_mapsize = m_size;
		if(false == roundup(m_size)) {
			close(m_fd);
			return false;
		}
	} else {
        /* Readonly, map the exact size for reading only
         */
		m_mapsize = m_size;
        AEGIS_DEBUG(1, "%s: '%s' mapsize is %d", __func__, p_name(), 
					(int)m_mapsize);
        if (0 < m_mapsize) {
            /* Use shared map as the memory reserved for private mappings
			 * is very limited.
             */
            m_data = (int8_t*)mmap(NULL, m_mapsize, PROT_READ, MAP_SHARED, m_fd, 0);
            if (MAP_FAILED == m_data) {
                AEGIS_ERROR("Cannot mmap '%s' (%s)", p_name(), strerror(errno));
                m_omode = om_closed;
                goto error;
            }
            AEGIS_DEBUG(1, "%s: mapped read-only %d bytes at %p", __func__, 
                        m_mapsize, m_data);
        }
    }
	return true;

error:
	return false;
}


bool
p_file::check_integrity()
{
    if (!m_new_file 
		&& (0 < datasize()) 
        && m_owner->contains_file(name())) 
    {
        m_digest = ""; 
        digest();
		if (m_digest != m_owner->m_contents[name()].csum) {
            AEGIS_ERROR("Corrupted data '%s'\n(saved:%s != computed:%s)", 
						name(), m_owner->m_contents[name()].csum.c_str(),
                        m_digest.c_str());
            m_omode = om_readonly;
            errno = EACCES;
            return false;
        } else {
            AEGIS_DEBUG(1, "%s: '%s' verified '%s'", __func__, name(), 
						digest());
            return true;
        }
    } else {
		if (m_owner->contains_file(name()))
			m_digest = m_owner->m_contents[name()].csum;
        return true;
	}
}


bool
p_file::p_open(int flags)
{
    int rc = 0;

    AEGIS_DEBUG(1, "%s: flags %d", __func__, flags);
	CHECK_VALID_RET(this, false);
	if (HAS_CHANGED(m_owner)) {
		REINITIALIZE_STORE(m_owner);
	}

	/* Before opening try to get a lock on the file */
	if (!int_open(flags, true)) {
        rc = errno;
		goto error;
    }
    if (!check_integrity()) {
        rc = errno;
        goto error;
    }
    if (!m_new_file) {
        struct stat fs;
        if (0 != p_stat(&fs)) {
            rc = errno;
            goto error;
        }
        if (!aegis_storage_dac_check(&fs, m_omode == om_readonly?0:1,
                                     getpid(), geteuid(), getegid())) 
        {
            rc = EACCES;
            goto error;
        }
		AEGIS_DEBUG(2,"File in storage has %u mode", fs.st_mode);
    } else {
		bool contains_file = m_owner->contains_file(name());
		struct stat fs;
		/* If the file is already in store keep the existing flag
		 * otherwise set it to true
		 */
		m_owner->m_contents[name()].new_file = contains_file ? 
			m_owner->m_contents[name()].new_file : true;
		/* Add the file in the store
		 */
		digest();
		m_owner->m_contents[name()].csum = m_digest.c_str();
		if (0 > fstat(m_fd, &fs))
			AEGIS_ERROR("Error from fstat (%s)", strerror(errno));
		/* Initial attributes for the file */
		m_owner->m_contents[name()].fs = fs;
		AEGIS_DEBUG(1, "%s: added %s with %s, new file = %s", __func__, name(), m_digest.c_str(),
			m_owner->m_contents[name()].new_file ? "true" : "false");
	}

	if (om_readonly != m_omode) {
		m_owner->m_lock->opened_for_writing(this);
	}

    return true;

error:
	p_close();
    errno = rc;
	return false;
}


bool
pe_file::p_open(int flags)
{
    int rc = 0;
	bool recover = false;
	openmode_t omode = om_closed;

	CHECK_VALID_RET(this, false);
	if (HAS_CHANGED(m_owner)) {
		REINITIALIZE_STORE(m_owner);
	}

	if (flags & O_RECOVER) {
		recover = true;
	}

    AEGIS_DEBUG(1, "pe_file::%s: (%p) %s(%s) flags %08o", 
                __func__, this, name(), p_name(), flags);

	if (!int_open(flags, true)) {
        rc = errno;
		goto error;
    }
	omode = m_omode;

    if (m_mapsize < sizeof(m_hdr)) {
        AEGIS_ERROR("%s: file too small to be encrypted, %lu bytes", 
                    __func__, (unsigned long)m_size);
        rc = EIO;
        goto error;
    }

	if (!m_owner->set_aes_keys(true, NULL)) {
        AEGIS_ERROR("Failed to set the AES key");
        goto error;
    }
        
	if (m_new_file || !m_owner->contains_file(name())) {
        if (0 > fstat(m_fd, &m_hdr.fs))
			AEGIS_ERROR("Error from fstat (%s)", strerror(errno));
        m_hdr.fs.st_size = 0;
		aegis_crypto_random(m_hdr.salt, sizeof(m_hdr.salt));
        memcpy(m_data, m_hdr.salt, sizeof(m_hdr.salt));
        CSUM_CMP((unsigned char*)&m_hdr, sizeof(m_hdr) - sizeof(m_hdr.csum), m_hdr.csum);
        crypto_op(cop_encrypt, sizeof(m_hdr.salt), &m_hdr.fs, 
                  sizeof(m_hdr) - sizeof(m_hdr.salt));
		bool contains_file = m_owner->contains_file(name());
		/* If the file is already in store keep the existing flag
		 * otherwise set it to true
		 */
		m_owner->m_contents[name()].new_file = contains_file ? 
			m_owner->m_contents[name()].new_file : true;
		/* Add the file in the store
		 */
		digest();
		m_owner->m_contents[name()].csum = m_digest.c_str();
		AEGIS_DEBUG(1, "%s: added %s with %s, new file = %s", __func__, name(), m_digest.c_str(),
			m_owner->m_contents[name()].new_file ? "true":"false");
	} else {
        unsigned char csum [sizeof(m_hdr.csum)];
        memcpy(m_hdr.salt, m_data, sizeof(m_hdr.salt));
        crypto_op(cop_decrypt, sizeof(m_hdr.salt), &m_hdr.fs, 
                  sizeof(m_hdr) - sizeof(m_hdr.salt));
        CSUM_CMP((unsigned char*)&m_hdr, sizeof(m_hdr) - sizeof(m_hdr.csum), csum);
        if (0 != memcmp(m_hdr.csum, csum, sizeof(m_hdr.csum))) {
            AEGIS_ERROR("%s: corrupted header stored %s != computed %s", 
                        __func__, 
                        dynhex(m_hdr.csum, sizeof(m_hdr.csum)),
                        dynhex(csum, sizeof(csum)));
            rc = EIO;
            goto error;

        } else if ((size_t)m_hdr.fs.st_size + sizeof(m_hdr) > m_size ) {
            AEGIS_ERROR("%s: header checks out but file sizes (data size %lu, m_size %lu)"
                        " do not match, probably some data missing", 
                        __func__, 
                        (long unsigned)m_hdr.fs.st_size, 
                        (long unsigned)m_size);
            m_hdr.fs.st_size = m_size - sizeof(m_hdr);
            if (!recover) {
                rc = EIO;
                goto error;
            }
        } else {
            AEGIS_DEBUG(1, "%s: header integrity OK, data size is %lu,"
                        " m_size is %lu", 
                        __func__, 
                        (long unsigned)m_hdr.fs.st_size, 
                        (long unsigned)m_size);
        }
        if (!aegis_storage_dac_check(&m_hdr.fs, m_omode == om_readonly?0:1, 
                                     getpid(), geteuid(), getegid())) 
        {
            rc = EACCES;
            goto error;
        }
	}
	
    if (!check_integrity()) {
		if (recover) {
			AEGIS_DEBUG(1, "%s: integrity check failed, recover", __func__);
			m_owner->m_contents[name()].csum = m_digest;
			m_omode = omode; 
			errno = 0;
		} else {
			goto error;
		}
    }

	if (om_readonly != m_omode) {
		m_owner->m_lock->closed(this);
	}

	return true;

error:
    if (-1 != m_fd)
        close(m_fd);
    if (MAP_FAILED != m_data) {
		if (0 > munmap(m_data, m_mapsize))
            AEGIS_ERROR("%s: munmap failed (%s)", __func__, strerror(errno));
        m_data = MAP_FAILED;
    }
    m_omode = om_closed;
    m_mapsize = 0;
    m_size = 0;
    errno = rc;
	return false;
}


bool
p_file::is_open(void)
{
	return m_omode != om_closed;
}

void
p_file::p_flush(bool do_trunc)
{
	if (om_readonly == m_omode)
		return;

	AEGIS_ENTER;
	
	contents_map_t::const_iterator ii = m_owner->m_contents.find(name());
	bool file_exists = (ii != m_owner->m_contents.end());
	size_t trunc_to = datasize();
	digest();

	if (0 > fsync(m_fd)) {
		AEGIS_ERROR("%s: fsync failed (%s)", __func__,
					strerror(errno));
	}
	if (do_trunc) {
		if (0 > ftruncate(m_fd, trunc_to)) {
			AEGIS_ERROR("ftruncate to %lu failed (%s)", 
						(unsigned long)trunc_to, 
						strerror(errno));
		} else {
			AEGIS_DEBUG(1, "%s: ftruncated to %lu", __func__, 
						(unsigned long)trunc_to);
			RAWDATA_PTR m_newdata = MAP_FAILED;
			if (0 < trunc_to) {
				m_newdata = mremap(m_data, m_mapsize, trunc_to, MREMAP_MAYMOVE);
				if (MAP_FAILED != m_newdata) {
					m_data = m_newdata;
					m_mapsize = trunc_to;
				} else {
					AEGIS_DEBUG(1, "%s: mremap to %lu failed (%s)", __func__, 
								(unsigned long)trunc_to, strerror(errno));
					if (false == roundup(trunc_to)) {
						AEGIS_DEBUG(1, "%s: roundup to %lu failed (%s)", __func__, 
									(unsigned long)trunc_to, strerror(errno));
					}
				}
			} else {
				if (MAP_FAILED != m_data) {
					if (0 > munmap(m_data, m_mapsize))
						AEGIS_ERROR("%s: munmap failed (%s)", __func__, strerror(errno));
					m_data = MAP_FAILED;
					m_mapsize = 0;
				}
			}
		}
	}
	AEGIS_DEBUG(1, "%s: digest %s", __func__, m_digest.c_str());

	/* Update the digest only if the file still belongs in the
	 * store.
	 */
	if(true == file_exists) {
		struct stat fs;
		m_owner->m_contents[name()].csum = m_digest.c_str();
		/* Save the latest metadata in the owner */
		if (0 < p_stat(&fs)){
			fs.st_size = m_size;
			update_metadata(&fs);
		}
		AEGIS_DEBUG(2, "%s: update '%s' in store %s", __func__, name(), 
					m_owner->name());
	} else {
		AEGIS_DEBUG(2, "%s: '%s' does not belong in %s", __func__, name(), 
					m_owner->name());
	}
}


void
p_file::p_close(void)
{
	AEGIS_DEBUG(1, "%s: p_file", __func__);
	if( "" != m_semname) {
		/* Check if the owner is still valid, if not make sure to close the file */
		sem_t *sem = sem_open(m_semname.c_str(), 0);
		if(SEM_FAILED == sem) {
			errno = EINVAL;
			goto cleanup;
		} else {
			sem_close(sem);
		}
	}
	m_owner->lock_store();
	if (HAS_CHANGED(m_owner)) {
		REINITIALIZE_STORE(m_owner);
	}
	if (!is_open()){
		m_owner->unlock_store();
        return;
	}
	if (om_readonly != m_omode) {
		m_owner->m_lock->closed(this);
		p_flush(true);
        m_owner->commit();
	}
	m_owner->unlock_store();

cleanup:
    if (MAP_FAILED != m_data) {
		if (0 > munmap(m_data, m_mapsize))
            AEGIS_ERROR("%s: munmap failed (%s)", __func__, strerror(errno));
        m_data = MAP_FAILED;
    }
    if (-1 != m_fd) {
        if (0 > close(m_fd))
            AEGIS_ERROR("%s: close returned (%s)", __func__, strerror(errno));
        m_fd = -1;
    }
    m_omode = om_closed;
    m_mapsize = 0;
    m_size = 0;
}


void
pe_file::p_flush(bool do_trunc)
{
    if (om_readonly == m_omode)
		return;

	AEGIS_ENTER;
	if (MAP_FAILED != m_data) {
		if ((time_t)-1 != m_hdr.fs.st_ctime)
			m_hdr.fs.st_atime = m_hdr.fs.st_mtime = time(NULL);
		else
			m_hdr.fs.st_ctime = time(NULL);
		m_digest = "";
		memcpy(m_data, m_hdr.salt, sizeof(m_hdr.salt));
		CSUM_CMP((unsigned char*)&m_hdr, 
				 sizeof(m_hdr) - sizeof(m_hdr.csum), 
				 m_hdr.csum);
		crypto_op(cop_encrypt, sizeof(m_hdr.salt), &m_hdr.fs, 
				  sizeof(m_hdr) - sizeof(m_hdr.salt));
	}
	p_file::p_flush(false);
}


void
pe_file::p_close()
{
	AEGIS_DEBUG(1, "%s: pe_file, size %d", __func__, (int)m_hdr.fs.st_size);

	/* Only check if we were able to open the semaphore in the parent */
	if( "" != m_semname){
		/* Check if the owner is still valid, if not make sure to close the file */
		sem_t *sem = sem_open(m_semname.c_str(), 0);
		if(sem_open(m_semname.c_str(), 0) == SEM_FAILED) {
			errno = EINVAL;
			goto cleanup;
		} else {
			sem_close(sem);
		}
	}

	m_owner->lock_store();
	if (HAS_CHANGED(m_owner)) {
		REINITIALIZE_STORE(m_owner);
	}

	if (!is_open()){
		m_owner->unlock_store();
        return;
	}

    if (om_readonly != m_omode) {
		m_owner->m_lock->closed(this);
		p_flush(false);
        size_t trunc_to = sizeof(m_hdr) + m_hdr.fs.st_size;
		trunc_to = BLOCKROUND(trunc_to);
		if (0 > ftruncate(m_fd, trunc_to)) {
			AEGIS_ERROR("ftruncate to %lu failed (%s)", 
						(unsigned long)trunc_to, 
						strerror(errno));
		} else {
			AEGIS_DEBUG(1, "%s: ftruncated to %lu", __func__, 
						(unsigned long)trunc_to);
        }
		m_owner->commit();
	}
	m_owner->clear_aes_keys();
	m_owner->unlock_store();

cleanup:
	if (MAP_FAILED != m_data) {
		if (0 > munmap(m_data, m_mapsize))
            AEGIS_ERROR("%s: munmap failed (%s)", __func__, strerror(errno));
		m_data = MAP_FAILED;
	}
    if (-1 != m_fd) {
        if (0 > close(m_fd))
            AEGIS_ERROR("%s: close returned (%s)", __func__, strerror(errno));
        m_fd = -1;
    }
	m_omode = om_closed;
	m_mapsize = 0;
	m_size = 0;
}


void
p_file::p_rollback()
{
	if (om_closed != m_omode) {
		enum openmode_t omode = m_omode;
		m_omode = om_readonly;
		CHECK_VALID(this);
		if (om_readonly != omode) {
			m_owner->m_lock->closed(this);
		}
	}
}


p_file::~p_file(void)
{
	AEGIS_ENTER;
    if (is_open())
        p_close();
}


pe_file::~pe_file(void)
{
	AEGIS_ENTER;
    if (is_open())
        p_close();
}


ssize_t
p_file::p_write(off_t at, const RAWDATA_PTR data, size_t len)
{
	CHECK_VALID_RET(this, -1);
	if (m_omode != om_readwrite && m_omode != om_writeonly) {
		errno = EBADF;
		return -1;
	}
    if (NULL == data) {
		errno = EINVAL;
		return -1;
    }
	AEGIS_DEBUG(1, "%s: write %d from %p (this %p) at %d", __func__,
				len, data, this, at);
	if (0 < len) {
		if(false == roundup(at + len)) {
			errno = ENOMEM;
			p_close();
			return -1;
		}
		memcpy((int8_t*)m_data + at, data, len);
		m_digest = "";
	}
	return len;
}


ssize_t
pe_file::p_write(off_t at, const RAWDATA_PTR data, size_t len)
{
	CHECK_VALID_RET(this, -1);
    if (NULL == data) {
		errno = EINVAL;
		return -1;
    }
	AEGIS_DEBUG(1, "pe_file::%s: write %d from %p (this %p) at %d", __func__,
				len, data, this, at);

	if (m_omode != om_readwrite && m_omode != om_writeonly) {
		errno = EBADF;
		return -1;
	}
    size_t oldsize = m_hdr.fs.st_size;
    size_t newsize = at + len;
	if (m_hdr.fs.st_size < newsize)
		m_hdr.fs.st_size = newsize;
    newsize = BLOCKROUND(newsize);
	if(false == roundup(sizeof(m_hdr) + newsize)) {
		m_hdr.fs.st_size = oldsize;
		p_close();
		errno = ENOMEM;
		return -1;
	}
    if (at > (off_t)oldsize) {
        AEGIS_DEBUG(1, "%s: fill with zeros range %lu..%lu (%lu)",
                    __func__, 
                    (unsigned long)oldsize,
                    (unsigned long)at, 
                    (unsigned long)at - oldsize);
        crypto_op(cop_encrypt, sizeof(m_hdr) + oldsize, NULL, at - oldsize);
    }
	crypto_op(cop_encrypt, sizeof(m_hdr) + at, (RAWDATA_PTR)data, len);
	m_digest = "";
	return len;
}


int
p_file::p_trunc(off_t at)
{
	CHECK_VALID_RET(this, -1);
	AEGIS_DEBUG(1, "%s: truncate at %lu", __func__, (unsigned long)at);
	if (m_omode != om_readwrite && m_omode != om_writeonly) {
		errno = EBADF;
		return -1;
	}
    if (m_size != at) {
        m_size = at;
        m_digest = "";
    }
    if (0 == m_size) {
        AEGIS_DEBUG(1, "%s: size zero, unmap", __func__);
        if (MAP_FAILED != m_data) {
            if (0 > munmap(m_data, m_mapsize))
                AEGIS_ERROR("%s: munmap failed (%s)", __func__, strerror(errno));
            m_data = MAP_FAILED;
        }
        m_mapsize = 0;
	} else {
		if(false == roundup(m_size)){
			p_close();
			errno = ENOMEM;
			return -1;
		}
    }
	if (0 > ftruncate(m_fd, m_mapsize)) {
		AEGIS_ERROR("%s: ftruncate to %lu failed (%s)", __func__,
					(unsigned long)m_mapsize, strerror(errno));
		return -1;
	}
	return 0;
}


int
pe_file::p_trunc(off_t at)
{
	CHECK_VALID_RET(this, -1);
	AEGIS_DEBUG(1, "pe_file::%s: truncate at %d", __func__, at);
	if (m_omode != om_readwrite && m_omode != om_writeonly) {
		errno = EBADF;
		return -1;
	}
	/* Save the old size in case we need to set contents to 0 
	 */
	int old_size = m_hdr.fs.st_size;
	/* Save the actual datasize to the header
	 */
    if (m_hdr.fs.st_size != at) {
        m_hdr.fs.st_size = at;
        m_digest = "";
    }
    at = BLOCKROUND(at);
    m_size = sizeof(m_hdr) + at;
	if(false == roundup(m_size)) {
		p_close();
		errno = ENOMEM;
		return -1;
	}
	if (0 > ftruncate(m_fd, m_mapsize)) {
		AEGIS_ERROR("%s: ftruncate to %lu failed (%s)", __func__,
					(unsigned long)m_mapsize, strerror(errno));
		return errno;
	}
	/* We need to zero the file contents 
	 */
	if (old_size < m_hdr.fs.st_size) {
		size_t len = m_hdr.fs.st_size - old_size;
		RAWDATA_PTR data = malloc(len);
		if (NULL == data) {
			AEGIS_ERROR("%s: failed to allocate memory for 0 buffer",
						__func__);
			errno = ENOMEM;
			return -1;
		}
		memset(data, '\0', len);
		/* Writing failed, revert the file to the old size so 
		 * there we don't have any inconsistent data 
		 */
		if (-1 == p_write(old_size, data, len)) {
			m_hdr.fs.st_size = old_size;
			free(data);
			return -1;
		}
		free(data);
	}
	return 0;
}


ssize_t
p_file::p_read(off_t at, RAWDATA_PTR data, size_t len)
{
	CHECK_VALID_RET(this, -1);
    if (NULL == data) {
		errno = EINVAL;
		return -1;
    }
	if (om_readonly != m_omode && om_readwrite != m_omode) {
		errno = EBADF;
		return -1;
	}
	AEGIS_DEBUG(1, "%s: read %lu from %p (this %p) at %lu", __func__,
				(unsigned long)len, data, this, (unsigned long)at);
	if (at + len > m_size) {
		if (at >= m_size)
            /* TODO: Is this the right error code?
			 */
			return 0;
		else {
			memcpy(data, (int8_t*)m_data + at, m_size - at);
			return m_size - at;
		}
	} else {
		memcpy(data, (int8_t*)m_data + at, len);
		return len;
	}
}


ssize_t
pe_file::p_read(off_t at, RAWDATA_PTR data, size_t len)
{
	CHECK_VALID_RET(this, -1);
    if (NULL == data) {
		errno = EINVAL;
		return -1;
    }
	AEGIS_DEBUG(1, "pe_file::%s: read %u from %p (this %p) at %u", __func__,
				(unsigned)len, data, this, (unsigned)at);
	if (om_readonly != m_omode && om_readwrite != m_omode) {
		errno = EBADF;
		return -1;
	}
	if (at + len > m_hdr.fs.st_size) {
		if (at >= m_hdr.fs.st_size)
			return 0;
        else
            len = m_hdr.fs.st_size - at;
    }
	crypto_op(cop_decrypt, sizeof(m_hdr) + at, data, len);
	return len;
}


int
p_file::p_cleanup(void)
{
    AEGIS_ENTER;
    return 0;
}


int
pe_file::p_cleanup(void)
{
    AEGIS_DEBUG(1, "%s: rm '%s'", __func__, m_actname.c_str());
    return unlink(m_actname.c_str());
}


const char*
p_file::name(void)
{
	return m_name.c_str();
}


const char*
p_file::p_name(void)
{
	return name();
}


const char*
pe_file::p_name(void)
{
	return m_actname.c_str();
}


const char*
p_file::digest(void)
{
	CHECK_VALID_RET(this, NULL);
	if ("" == m_digest && MAP_FAILED != m_data) {
        AEGIS_DEBUG(1, "%s: m_data=%p m_mapsize=%lu m_size=%lu", __func__,
                    m_data, (unsigned long)m_mapsize, (unsigned long)m_size);
		m_owner->compute_digest((unsigned char*)m_data, m_size, 'b', m_digest);
    }
	return m_digest.c_str();
}


const char*
pe_file::digest(void)
{
	CHECK_VALID_RET(this, NULL);
	size_t siz = datasize();
	if ("" == m_digest && MAP_FAILED != m_data) {
        AEGIS_DEBUG(1, "%s: m_data=%p m_mapsize=%lu m_size=%lu data=%s", __func__,
                    m_data, (unsigned long)m_mapsize, (unsigned long)siz,
					dynhex((unsigned char*)m_data + sizeof(m_hdr), siz>16?16:siz));
		/* Including header data in the digest means that it has to be 
		 * recomputed every time header changes, which is a bit more work 
		 * than otherwise. A bit extra efficiency could be squeezed out
		 * by not including the header in the digest.
		 */
#ifdef NOT_INCLUDE_HEADER
		m_owner->compute_digest((unsigned char*)m_data + sizeof(m_hdr), 
								siz, 'b', m_digest);
#else
		m_owner->compute_digest((unsigned char*)m_data, 
								BLOCKROUND((sizeof(m_hdr) + siz)), 
								'b', m_digest);
#endif
    }
	return m_digest.c_str();
}


/* Grow in 4 kB blocks
 */
#define INC_SIZE 0x1000
#define INC_MASK 0x0fff // INC_SIZE - 1

bool
p_file::roundup(size_t len)
{
	RAWDATA_PTR old_data = MAP_FAILED;
	size_t new_mapsize;

	if (len > m_size)
		m_size = len;

    /* Round mapsize to the nearest page boundary
     */
	new_mapsize = (m_size | INC_MASK) + 1;
	if (new_mapsize > m_mapsize || 0 == m_mapsize) {
		/* Increase the file size to match the mapping
		 */
		size_t written = 0;
#ifdef _XOPEN_UNIX
		written = new_mapsize - m_mapsize;
		if (0 > ftruncate(m_fd, new_mapsize)) {
			AEGIS_ERROR("%s: ftruncate to %ld failed (%s)", __func__,
						(long)new_mapsize, strerror(errno));
			return false;
		}
		if ((new_mapsize - 1) > lseek(m_fd, new_mapsize - 1, SEEK_SET)) {
			AEGIS_ERROR("%s: cannot grow file '%s' to %lu, disk full", __func__,
						p_name(), (unsigned long)new_mapsize);
			return false;
		}
		/* Do a small write to make sure there is enough space
		 * on the disk left.
		 */
		char eofmark = '\0';
		if (1 > write(m_fd, &eofmark, 1)) {
			AEGIS_ERROR("%s: cannot grow file '%s' to %lu (%s)", __func__,
						p_name(), (unsigned long)new_mapsize, strerror(errno));
			return false;
		}
#else
		char buf[INC_SIZE];
		off_t lpos;

		memset(buf, '\0', sizeof(buf));
		lpos = lseek(m_fd, m_mapsize, SEEK_SET);
		if (lpos != m_mapsize) {
			AEGIS_DEBUG(1, "%s: seek error (%s)", __func__, strerror(errno));
		}
		while (m_mapsize + written < new_mapsize) {
			size_t blsize = new_mapsize - (m_mapsize + written);
			ssize_t blwrt = 0;
			if (blsize > sizeof(buf))
				blsize = sizeof(buf);
			blwrt = write(m_fd, buf, blsize);
			if (0 >= blwrt) {
				AEGIS_ERROR("Disk full!");
				return false;
			}
			written += blwrt;
			AEGIS_DEBUG(1, "%s: written %u", __func__, (unsigned)written);
		}
#endif
        AEGIS_DEBUG(1, "%s: grow %u -> %u to hold %u by adding %u, fs is %u", 
                    __func__, (unsigned)m_mapsize, (unsigned)new_mapsize, 
                    (unsigned)len, (unsigned)written, (unsigned)m_size);

	} else if (new_mapsize == m_mapsize) {
        AEGIS_DEBUG(1, "%s: no need to grow, %u holds %u, fs is %u", 
                    __func__, (unsigned)m_mapsize, (unsigned)len, 
					(unsigned)m_size);
		return true;

    } else {
        AEGIS_DEBUG(1, "%s: file has shrunk to %u, remap to %u to hold %u", 
                    __func__, (unsigned)m_size, (unsigned)new_mapsize, 
                    (unsigned)len);
    }

	if (MAP_FAILED != m_data) {
		old_data = m_data;
		m_data = mremap(m_data, m_mapsize, new_mapsize, MREMAP_MAYMOVE);
    } else {
        m_data = mmap(NULL, new_mapsize, PROT_READ | PROT_WRITE, MAP_SHARED, m_fd, 0);
    }

    if (MAP_FAILED == m_data) {
		/* Restore the old data pointer if available to make sure we unmap
		 * the data when closing 
		 */
		if (MAP_FAILED != old_data) {
			m_data = old_data;
		}
        AEGIS_ERROR("%s: cannot map/remap (%s)", __func__, strerror(errno));
		return false;
    } else {
        AEGIS_DEBUG(1, "%s: mapped read-write %d bytes at %p", __func__, 
                    new_mapsize, m_data);
    }
    /* Initialize with zero if necessary. Also if the file is
	 * shrinking the extra space needs to be zeroed.
     */
    if (new_mapsize > m_mapsize) {
        AEGIS_DEBUG(1, "%s: zero %lu bytes at %p (top at %p)", __func__,
                    new_mapsize - m_mapsize, (char*)m_data + m_mapsize,
                    (char*)m_data + new_mapsize);
        memset((char*)m_data + m_mapsize, '\0', new_mapsize - m_mapsize);

    } else if (new_mapsize < m_mapsize) {
        AEGIS_DEBUG(1, "%s: zero %lu bytes at %p (top at %p)", __func__,
                    new_mapsize - m_size, (char*)m_data + m_size,
                    (char*)m_data + new_mapsize);
		memset((char*)m_data + m_size, '\0', new_mapsize - m_size);
	}
	m_mapsize = new_mapsize;
	return true;
}


int
p_file::p_stat(struct stat *st)
{
	struct stat fs;
    AEGIS_ENTER;
	CHECK_VALID_RET(this, -1);
    if (NULL == st) {
		errno = EINVAL;
		return -1;
    }
    
	if (0 > stat(name(), st)){
		return -1;
	}
	/* The file is in storage, merge with the storage values.
	 */
	if (m_owner->contains_file(name())) {
		fs = m_owner->m_contents[name()].fs;
		st->st_size = fs.st_size;
		st->st_mode = fs.st_mode;
		st->st_uid = fs.st_uid;
		st->st_gid = fs.st_gid;
		st->st_mtime = fs.st_mtime;
		st->st_atime = fs.st_atime;
		st->st_ctime = fs.st_ctime;
	}
	/* Copy all the st data from one struct to the other */
	if (om_readwrite == m_omode || om_writeonly == m_omode) {
        /* If the file is open for writing, return
         * the real size in stead of the mapping's size,
         * which is rounded up to nearest 4kB.
         */
        AEGIS_DEBUG(1, "%s: real file size is %lu", __func__, m_size);
        st->st_size = m_size;
    }
    update_metadata(st);
	return 0;
}


int
pe_file::p_stat(struct stat *st)
{
    AEGIS_ENTER;
	CHECK_VALID_RET(this, -1);
    if (NULL == st) {
		errno = EINVAL;
		return -1;
    }
    /* Optimization:  if the file has been opened
     * at least once, use the cached attributes.
     */
    if ((off_t)-1 != m_hdr.fs.st_size) {
		memcpy(st, &m_hdr.fs, sizeof(struct stat));
        AEGIS_DEBUG(1, "%s: pe_file, size %d", __func__, (int)m_hdr.fs.st_size);
		return 0;
	} else {
		if (p_open(O_RDONLY | O_RECOVER)) {
            memcpy(st, &m_hdr.fs, sizeof(struct stat));
            AEGIS_DEBUG(1, "%s: pe_file, size %d", __func__, (int)m_hdr.fs.st_size);
            p_close();
            return 0;
        } else {
            return -1;
        }
	}
}


size_t 
p_file::datasize()
{
	return m_size;
}


size_t 
pe_file::datasize()
{
	if ((off_t)-1 != m_hdr.fs.st_size)
		return m_hdr.fs.st_size;
	else
		/* The header has not yet been read in so we assume
		 * that the actual file size has not yet been rounded
		 * up.
		 */
		return m_size - sizeof(m_hdr);
}


/* The encryption/decryption primitive XTS-AES as defined
 * in Std. IEEE 1619-2007. Use a combination of the 16 byte 
 * random salt generated  for each file and the block number to
 * produce an unique nonce for each block in each file that
 * uses the same keys.
 */
void
pe_file::crypto_block(size_t blocknr, cop_t cop, int8_t* encrypted, int8_t *plaintext)
{
	int8_t nonce[AES_BLOCK_SIZE];
	int8_t tweak[AES_BLOCK_SIZE];
	int8_t input[AES_BLOCK_SIZE];
	int8_t output[AES_BLOCK_SIZE];
	keypack_t *keys = &m_owner->m_lock->keypack;
	int i;

	if (0 == keys->refcount) {
		AEGIS_ERROR("%s: keys not set", __func__);
		return;
	}
	/* Make nonce
	 */
	for (i = 0; i < AES_BLOCK_SIZE; i++) {
		nonce[i] = m_hdr.salt[i] ^ (int8_t)(blocknr & 0xff);
		blocknr >>= 8;
	}
	/* Encrypt nonce with key #2
	 */
	AES_encrypt((unsigned char*)nonce, (unsigned char*)tweak, &keys->twk_key);
	/* Xor input with tweak
	 */
	int8_t* source = (cop_encrypt == cop) ? plaintext : encrypted;
	for (i = 0; i < AES_BLOCK_SIZE; i++)
		input[i] = source[i] ^ tweak[i];
	/* Encrypt or decrypt with key #1
	 */
	if (cop_encrypt == cop) {
		AES_encrypt((unsigned char*)input, (unsigned char*)output, &keys->enc_key);
	} else {
		AES_decrypt((unsigned char*)input, (unsigned char*)output, &keys->dec_key);
	}
	/* Xor the result once more with tweak
	 */
	int8_t* destination = (cop_encrypt == cop) ? encrypted : plaintext;
	for (i = 0; i < AES_BLOCK_SIZE; i++)
		destination[i] = output[i] ^ tweak[i];

	/* Zero intermediate results
	 */
	memset(tweak, '\0', sizeof(tweak));
	memset(input, '\0', sizeof(input));
	memset(output, '\0', sizeof(output));
}


void
pe_file::crypto_op(cop_t cop, off_t at, RAWDATA_PTR data, size_t len)
{
    size_t bi = at >> BLOCKSHFT;
    int8_t first_block[BLOCKSIZE], 
        *d_ptr = (int8_t*)data, 
        *m_ptr = (int8_t*)m_data + (bi << BLOCKSHFT);

    AEGIS_DEBUG(1, "%s: %s len %d at %lld", __func__,
                cop_decrypt==cop?"decrypt":"encrypt",
                len, at);

	/* Encrypting with a NULL pointer is the same as
	 * encrypting all zero bytes.
	 */
    if (NULL == d_ptr && cop_decrypt == cop) {
        AEGIS_ERROR("%s: crypto_of decrypt to NULL", __func__);
        return;
    }

    /* If the operation does not start at the
     * block boundary, handle the first block
     * separately.
     */
    off_t l_off = at & BLOCKMASK;
	if (l_off) {
        size_t l_size = BLOCKSIZE - l_off;
        if (l_size > len)
            l_size = len;
        crypto_block(bi, cop_decrypt, m_ptr, first_block);
        if (cop_encrypt == cop) {
            if (d_ptr)
                memcpy(&first_block[l_off], d_ptr, l_size);
            else
                memset(&first_block[l_off], '\0', l_size);
            crypto_block(bi, cop_encrypt, m_ptr, first_block);
        } else
            memcpy(d_ptr, &first_block[l_off], l_size);
        len -= l_size;
        if (d_ptr)
            d_ptr += l_size;
        m_ptr += BLOCKSIZE;
        bi++;
    }
    memset(first_block, '\0', BLOCKSIZE);

    if (0 == len)
        return;

    /* Handle whole blocks
     */
    size_t bcount = len >> BLOCKSHFT;
    for (size_t b_off = 0; b_off < bcount; b_off++) {
        if (d_ptr) {
            crypto_block(bi, cop, m_ptr, d_ptr);
            d_ptr += BLOCKSIZE;
        } else {
            crypto_block(bi, cop, m_ptr, first_block);
        }
        len   -= BLOCKSIZE;
        m_ptr += BLOCKSIZE;
		bi++;
    }

    /* Handle the remainder block, if any. 
     */
    if (0 < len) {
        crypto_block(bi, cop_decrypt, m_ptr, first_block);
        if (cop_encrypt == cop) {
            if (d_ptr)
                memcpy(first_block, d_ptr, len);
            else
                memset(first_block, '\0', len);
            crypto_block(bi, cop_encrypt, m_ptr, first_block);
        } else
            memcpy(d_ptr, first_block, len);
        memset(first_block, '\0', BLOCKSIZE);
    }
}


int 
p_file::p_chmod(mode_t flags)
{
	struct stat fs;
	CHECK_VALID_RET(this, -1);
    AEGIS_DEBUG(1, "%s: %s %lo", __func__, p_name(), (long unsigned)flags);
	p_stat(&fs);
	fs.st_mode = flags;
	fs.st_ctime = time(NULL);
	update_metadata(&fs);
	return 0;
}


int 
pe_file::p_chmod(mode_t flags)
{
	CHECK_VALID_RET(this, -1);
    AEGIS_DEBUG(1, "pe_file::%s: %s %lo", __func__, p_name(), (long unsigned)flags);
    if (is_open()) {
        m_hdr.fs.st_mode = flags;
        m_hdr.fs.st_ctime = time(NULL);
		m_digest = "";
    } else {
        if (p_open(O_RDWR)) {
            m_hdr.fs.st_mode = flags;
            m_hdr.fs.st_ctime = (time_t)-1;
            p_close();
        } else {
            return -1;
        }
    }
    return 0;
}


int
p_file::p_chown (uid_t uid, gid_t gid)
{
	struct stat fs;
	CHECK_VALID_RET(this, -1);
    AEGIS_DEBUG(1, "%s: %s %d.%d", __func__, p_name(), (int)uid, (int)gid);
	p_stat(&fs);
	fs.st_uid = uid;
	fs.st_gid = gid;
	fs.st_ctime = time(NULL);
	update_metadata(&fs);
	return 0;
}


int
pe_file::p_chown (uid_t uid, gid_t gid)
{
	CHECK_VALID_RET(this, -1);
    AEGIS_DEBUG(1, "pe_file::%s: %s %d.%d", __func__, p_name(), (int)uid, (int)gid);
    if (is_open()) {
        m_hdr.fs.st_uid = uid;
        m_hdr.fs.st_gid = gid;
        m_hdr.fs.st_ctime = time(NULL);
		m_digest = "";
    } else {
        if (p_open(O_RDWR)) {
            m_hdr.fs.st_uid = uid;
            m_hdr.fs.st_gid = gid;
            m_hdr.fs.st_ctime = (time_t)-1;
            p_close();
        } else {
            return -1;
        }
    }
    return 0;
}


int
p_file::p_utime(struct utimbuf *ntime)
{
	struct stat fs;
	CHECK_VALID_RET(this, -1);
    if (NULL != ntime) {
        AEGIS_DEBUG(1, "%s: %s %lu.%lu", __func__, 
                    p_name(), 
                    (long unsigned)ntime->actime, 
                    (long unsigned)ntime->modtime);
		p_stat(&fs);
		fs.st_atime = (time_t)ntime->actime;
		fs.st_mtime = (time_t)ntime->modtime;
		fs.st_ctime = time(NULL);
		update_metadata(&fs);
		return 0;
	} else {
		errno = EINVAL;
		return -1;
	}
}


int
pe_file::p_utime(struct utimbuf *ntime)
{
	CHECK_VALID_RET(this, -1);
    AEGIS_DEBUG(1, "pe_file::%s: %s %lu.%lu", __func__, p_name(), 
                (long unsigned)ntime->actime, 
                (long unsigned)ntime->modtime);
    if (is_open()) {
        m_hdr.fs.st_atime = ntime->actime;
        m_hdr.fs.st_mtime = ntime->modtime;
        m_hdr.fs.st_ctime = time(NULL);
		m_digest = "";
    } else {
        if (p_open(O_RDWR)) {
            m_hdr.fs.st_atime = ntime->actime;
            m_hdr.fs.st_mtime = ntime->modtime;
            m_hdr.fs.st_ctime = (time_t)-1;
            p_close();
        } else {
            return -1;
        }
    }
    return 0;
}


/* Access to the crypto algroithm at the low level to enable
 * verification with csrc.nist.gov test vectors.
 */
bool 
storage::test_xts_aes(bool do_encrypt, 
					  int8_t key[32],
					  int8_t ivec[16],
					  size_t block_nr,
					  int8_t idata[16],
					  int8_t odata[16])
{
	if (0 < m_contents.size()) {
		AEGIS_ERROR("%s: can be called only with an empty store", __func__);
		return false;
	}
	if (32 > m_symkey_len) {
		AEGIS_ERROR("%s: no room for external key", __func__);
		return false;
	}
	memcpy(m_symkey, key, 32);
	if (!set_aes_keys(false, NULL)) {
		AEGIS_ERROR("%s: can not set encryption keys", __func__);
		return false;
	}
	pe_file *pt = new pe_file(this, "");
	memcpy(pt->m_hdr.salt, ivec, 16);
	if (do_encrypt)
		pt->crypto_block(block_nr, pe_file::cop_encrypt, 
						 (int8_t*)odata, (int8_t*)idata);
	else
		pt->crypto_block(block_nr, pe_file::cop_decrypt, 
						 (int8_t*)idata, (int8_t*)odata);
	delete pt;
	clear_aes_keys();
	return true;
}


/* Autoconversion from the old format to the new one
 */
void
pe_file::decrypt_old(AES_KEY *okey, size_t blocknr, int8_t* encrypted, int8_t *plaintext)
{
	int8_t nonce[BLOCKSIZE];
	int i;

	memset(nonce, '\0', sizeof(nonce));
	memcpy(&nonce[sizeof(nonce) - sizeof(size_t)], &blocknr, sizeof(size_t));
	for (i = 0; i < BLOCKSIZE; i++) {
		nonce[i] ^= m_hdr.salt[i];
	}
	int8_t mask[BLOCKSIZE];
	AES_encrypt((unsigned char*)nonce, (unsigned char*)mask, okey);
	for (i = 0; i < BLOCKSIZE; i++) {
		*plaintext++ = *encrypted++ ^ mask[i];
	}
}


bool
pe_file::convert_crypto(AES_KEY *okey, string& new_digest)
{
	int of = -1, nf = -1;
	bool success = true;
	size_t blnr = 0;
	ssize_t len = 0;
	int8_t inp[AES_BLOCK_SIZE];
	int8_t otp[AES_BLOCK_SIZE];
	unsigned char md[DIGESTLEN];
    DIGEST_CTX mdctx;
    DIGEST_CTX mdctx_n;
	string tmpname(m_actname);

	AEGIS_DEBUG(1, "%s: %s", __func__, m_actname.c_str());

    if (!DIGEST_INIT(mdctx)) {
		AEGIS_ERROR("%s: DIGEST_INIT (%s)", __func__, strerror(errno));
		return false;
    }
    if (!DIGEST_INIT(mdctx_n)) {
		AEGIS_ERROR("%s: DIGEST_INIT (%s)", __func__, strerror(errno));
		return false;
    }
	of = open(m_actname.c_str(), O_RDONLY);
	if (0 > of) {
		AEGIS_ERROR("%s: cannot open '%s' (%s)", __func__, m_actname.c_str(), strerror(errno));
		return false;
	}
	tmpname.append(".tmp");
	nf = creat(tmpname.c_str(), 0600);
	if (0 > nf) {
		AEGIS_ERROR("%s: cannot open '%s' (%s)", __func__, tmpname.c_str(), strerror(errno));
		close(of);
		return false;
	}

	while (AES_BLOCK_SIZE == (len = read(of, inp, AES_BLOCK_SIZE))) {
		if (0 == blnr) {
			memcpy(m_hdr.salt, inp, AES_BLOCK_SIZE);
		} else {
			/* Old digest is computed only on payload data. Notice
			 * that it doesn't necessarily start on a block boundary.
			 */
			if ((blnr * AES_BLOCK_SIZE) > sizeof(m_hdr)) {
				AEGIS_DEBUG(3, "%s: %d: offset %d len %d", __func__, (int)blnr, 
							(int)blnr * AES_BLOCK_SIZE, AES_BLOCK_SIZE);
				DIGEST_UPDATE(mdctx, inp, AES_BLOCK_SIZE);
			} else if (((blnr + 1) * AES_BLOCK_SIZE) > sizeof(m_hdr)) {
				size_t offset = sizeof(m_hdr) % AES_BLOCK_SIZE;
				AEGIS_DEBUG(3, "%s: %d: offset %d len %d", __func__, (int)blnr, 
							(int)blnr * AES_BLOCK_SIZE + offset, 
							AES_BLOCK_SIZE - offset);
				DIGEST_UPDATE(mdctx, &inp[offset], AES_BLOCK_SIZE - offset);
			}
			decrypt_old(okey, blnr, inp, otp);
			crypto_block(blnr, cop_encrypt, inp, otp);
		}
		if (AES_BLOCK_SIZE > write(nf, inp, AES_BLOCK_SIZE)) {
			AEGIS_ERROR("%s: write failed (%s)", __func__, strerror(errno));
			success = false;
			break;
		} else {
			/* New digest is computed out of all data
			 */
			DIGEST_UPDATE(mdctx_n, inp, AES_BLOCK_SIZE);
		}
		blnr++;
	}

	if (0 < len) {
		DIGEST_UPDATE(mdctx, inp, len);
		AEGIS_DEBUG(3, "%s: %d: offset %d len %d", __func__, (int)blnr, 
					(int)blnr * AES_BLOCK_SIZE, len);
		memset(&inp[len], '\0', AES_BLOCK_SIZE - len);
		decrypt_old(okey, blnr, inp, otp);
		crypto_block(blnr, cop_encrypt, inp, otp);
		if (AES_BLOCK_SIZE > write(nf, inp, AES_BLOCK_SIZE)) {
			AEGIS_ERROR("%s: write failed (%s)", __func__, strerror(errno));
			success = false;
		} else {
			DIGEST_UPDATE(mdctx_n, inp, AES_BLOCK_SIZE);
		}
	}
	DIGEST_FINAL(mdctx, md);

	/* Check the digest 
	 */
	if (success) {
		char *tmp = base64_encode(md, DIGESTLEN);
		if (tmp) {
			if (0 == strcmp(tmp, m_owner->m_contents[name()].csum.c_str())) {
				AEGIS_DEBUG(1, "%s: digest '%s' matches", __func__, tmp);
			} else {
				AEGIS_DEBUG(1, "%s: computed digest '%s' != stored '%s'", __func__, 
							tmp, m_owner->m_contents[name()].csum.c_str());
				success = false;
			}
			free(tmp);
		} else {
			AEGIS_DEBUG(1, "%s: failed to compute digest", __func__, strerror(errno));
			success = false;
		}
	}

	/* Save the new digest
	 */
	if (success) {
		DIGEST_FINAL(mdctx_n, md);
		char *tmp = base64_encode(md, DIGESTLEN);
		if (tmp) {
			new_digest.assign(tmp);
			free(tmp);
		}
	}
	
	/* Copy metadata.
	 */
	if (success) {
		struct stat fs;
		if (0 > fstat(of, &fs)) {
			AEGIS_ERROR("%s: stat failed (%s)", __func__, strerror(errno));
			success = false;
		} else {
			if (0 > fchown(nf, fs.st_uid, fs.st_gid)) {
				AEGIS_ERROR("%s: fchown failed (%s)", __func__, strerror(errno));
				success = false;
			}
			if (0 > fchmod(nf, fs.st_mode)) {
				AEGIS_ERROR("%s: fchmod failed (%s)", __func__, strerror(errno));
				success = false;
			}
		}
	}
	/* Close files
	 */
	if (0 > close(nf)) {
		AEGIS_ERROR("%s: close failed (%s)", __func__, strerror(errno));
		success = false;
	}
	if (0 > close(of)) {
		AEGIS_ERROR("%s: close failed (%s)", __func__, strerror(errno));
		success = false;
	}
	AEGIS_DEBUG(1, "%s: finished %s", __func__, 
				success?"succesfully":"with some errors");
	if (success) {
		if (0 > rename(tmpname.c_str(), m_actname.c_str())) {
			AEGIS_ERROR("%s: rename '%s' -> '%s' failed (%s)", __func__,
						tmpname.c_str(), m_actname.c_str(), strerror(errno));
			success = false;
		}
	}
	if (!success) {
		if (0 > unlink(tmpname.c_str())) {
			AEGIS_ERROR("%s: unlink '%s' failed (%s)", __func__,
						tmpname.c_str(), strerror(errno));
		}
	}
	AEGIS_DEBUG(1, "%s: conversion of '%s' %s", __func__, m_actname.c_str(), 
				success?"succeeded":"failed");
	return success;
}


/* Move encrypted files to their new location
 */
bool
storage::convert_store(const char *new_filename)
{
	AEGIS_DEBUG(1, "%s: converting '%s'", __func__, name());
	string old_filename = m_filename;
	m_filename = new_filename;
	lock_store();
	for (
		contents_map_t::iterator ii = m_contents.begin();
		m_contents.end() != ii;
		ii++
	) {
		AEGIS_DEBUG(1, "%s: convert '%s'", __func__, ii->first.c_str());
		pe_file *tmp = (pe_file*)member(ii->first.c_str());
		tmp->set_pathname(ii->first.c_str(), true);
		delete tmp;
	}
	unlock_store();
	/* Need to do commit even with an empty store to set 
	 * the signature right.
	 */
	if (commit()) {
		if (0 > unlink(old_filename.c_str()))
			AEGIS_ERROR("Couldn't unlink '%s' (%s)", old_filename.c_str(), 
						strerror(errno));
		AEGIS_DEBUG(1, "%s: converted '%s'", __func__, name());
	}
	return true;
}


storage_lock::storage_lock(storage* parent)
	: m_parent(parent),
	  m_fd(-1), 
	  lock_cnt(0), 
	  can_read(true),
	  can_write(true), 
	  m_read_only(false),
	  version(CURRENT_STORAGE_VERSION),
	  m_sharedhash((struct aegis_digest_t*)MAP_FAILED)
{
	AEGIS_ENTER;
	char tmp[NAME_MAX];

	memset(&keypack, '\0', sizeof(keypack));

	/* The shared memory name is same for all processes
	 * and instances.
	 */
	snprintf(tmp, sizeof(tmp), "/aegis-ps:%s:%d:%d",
			 m_parent->m_name.c_str(), (int)parent->m_vis, (int)parent->m_prot);
	mem_name.assign(tmp);
	sem_name.assign(tmp);

	/* The semaphore name is process and instance specific
	 */
	snprintf(tmp, sizeof(tmp), ":%d:%p", (int)getpid(), parent);
	sem_name.append(tmp);
	sem_t *sem = sem_open(sem_name.c_str(), O_CREAT, 0644, 0);
	if (SEM_FAILED == sem) {
		/* In case we are running very early in boot and there are no mounted
		 * tmpfs filesystems or all of them are mounted RO sem_open fails
		 * but we should still allow opening the store and normal operations
		 */
		sem_name = "";
		return;
	} else {
		/* Don't need the handle.
		 */
		if (0 > sem_close(sem))
			AEGIS_ERROR("%s: failed to close '%s' (%s)", __func__, sem_name.c_str(), 
						strerror(errno));
	}

	/* Zero umask temporarily to create world-writable
	 * shared memory. 
	 * TODO: This is a potential DoS entry-point. Anyone can
	 * lock the memory forever and prevent services. 
	 * Does flock(LOCK_EX) even require write access?
	 */
	mode_t cur_umask = umask(0);
	m_fd = shm_open(mem_name.c_str(), O_RDWR | O_CREAT, 0666);
	umask(cur_umask);

	if (0 > m_fd) {
		if (EACCES == errno) {
			AEGIS_DEBUG(1, "%s: opening '%s' for read-write failed (%s)", __func__, 
						mem_name.c_str(), strerror(errno));
			m_fd = shm_open(mem_name.c_str(), O_RDONLY, 0);
			if (0 <= m_fd) {
				AEGIS_DEBUG(1, "%s: opened '%s' read-only", __func__, mem_name.c_str());
				m_read_only = true;
			}
		}
		if (0 > m_fd) {
			AEGIS_ERROR("%s: failed to open/create '%s' (%s)", __func__, 
						mem_name.c_str(), strerror(errno));
			return;
		}
	} else {
		AEGIS_DEBUG(1, "%s: opened '%s' read-write", __func__, mem_name.c_str());
		storage_lock::wait();
	}

	struct stat fs;
	if (0 > fstat(m_fd, &fs)) {
		AEGIS_ERROR("%s: failed to stat '%s' (%s)", __func__, mem_name.c_str(), 
					strerror(errno));
		close(m_fd);
		m_fd = -1;
		goto end;
	}
	if (0 == fs.st_size) {
		if (0 > ftruncate(m_fd, sizeof(struct aegis_digest_t))) {
			AEGIS_ERROR("%s: failed to allocate '%s' (%s)", __func__, 
						mem_name.c_str(), strerror(errno));
			close(m_fd);
			m_fd = -1;
			goto end;
		}
	}

	{
		int prot = PROT_READ | PROT_WRITE;
		if (m_read_only)
			prot = PROT_READ;

		m_sharedhash = (struct aegis_digest_t*)mmap(NULL, 
													sizeof(struct aegis_digest_t),
													prot,
													MAP_SHARED,
													m_fd,
													0);
	}

	if (MAP_FAILED == m_sharedhash) {
		AEGIS_ERROR("%s: failed to mmap '%s' (%s)", __func__, mem_name.c_str(), 
					strerror(errno));
		close(m_fd);
		m_fd = -1;
		goto end;
	}
	memcpy(&m_localhash, m_sharedhash, sizeof(struct aegis_digest_t));
	AEGIS_DEBUG(1, "%s: %s", __func__, 
				dynhex(m_sharedhash, sizeof(struct aegis_digest_t)));
end:
	storage_lock::post();
}


storage_lock::~storage_lock()
{
	/* Remove the marker semaphore to signal potential members
	 * that this instance is no longer valid and should not be
	 * referenced.
	 */
	AEGIS_ENTER;
	if (sem_name.size()) {
		if (0 > sem_unlink(sem_name.c_str())) {
			AEGIS_ERROR("%s: failed to unlink '%s' (%s)", __func__,
						sem_name.c_str(), strerror(errno));
		} else {
			AEGIS_DEBUG(2, "%s: unlink '%s'", __func__, sem_name.c_str());
		}
	}
	while (locked())
		storage_lock::post();
	if (MAP_FAILED != m_sharedhash) {
		munmap(m_sharedhash, sizeof(struct aegis_digest_t));
		AEGIS_DEBUG(2, "%s: released '%s'", __func__, mem_name.c_str());
	}
	if (-1 != m_fd)
		close(m_fd);
}


/* Synchronization of index file handling.
 * Support nested locks.
 */
void
storage_lock::wait()
{
	AEGIS_ENTER;
	if (-1 == m_fd)
		return;
	if (0 == lock_cnt) {
		if (0 > flock(m_fd, LOCK_EX)) {
			AEGIS_ERROR("%s: failed to lock '%s' (%s)", __func__,
						mem_name.c_str(), strerror(errno));
			return;
		}
	}
	lock_cnt++;
	AEGIS_DEBUG(1, "%s: lock %d on '%s'", __func__, lock_cnt, mem_name.c_str());
}


void
storage_lock::post()
{
	if (-1 == m_fd)
		return;
	AEGIS_DEBUG(1, "%s: release %d on '%s'", __func__, lock_cnt, mem_name.c_str());
	if (1 == lock_cnt) {
		if (0 > flock(m_fd, LOCK_UN))
			AEGIS_ERROR("%s: failed to unlock '%s' (%s)", __func__,
						mem_name.c_str(), strerror(errno));
	}
	if (0 < lock_cnt)
		lock_cnt--;
}


bool
storage_lock::locked()
{
	return (0 < lock_cnt);
}


void 
storage_lock::mark(struct aegis_digest_t *hash)
{
	memcpy(&m_localhash, hash, sizeof(struct aegis_digest_t));
	if (locked() && MAP_FAILED != m_sharedhash && !m_read_only) {
		AEGIS_DEBUG(1, "%s: '%s'", __func__, dynhex(hash, sizeof(struct aegis_digest_t)));
		memcpy(m_sharedhash, &m_localhash, sizeof(struct aegis_digest_t));
	} else {
		AEGIS_DEBUG(1, "%s: without lock!?!", __func__);
	}
}


void 
storage_lock::remember()
{
	if (locked() && MAP_FAILED != m_sharedhash) {
		AEGIS_DEBUG(1, "%s: '%s'", __func__, dynhex(m_sharedhash, sizeof(struct aegis_digest_t)));
		memcpy(&m_localhash, m_sharedhash, sizeof(struct aegis_digest_t));
	}
}


bool
storage_lock::changed()
{
	if (MAP_FAILED != m_sharedhash) {
		return (0 != memcmp(&m_localhash, m_sharedhash, 
							sizeof(struct aegis_digest_t)));
	} else {
		return false;
	}
}


void
storage_lock::flush_open_files(void)
{
	for (vector<p_file*>::iterator pos = open_files.begin();
		 pos != open_files.end(); 
		 pos++
	) {
		/* p_flush should really be virtual but that would break 
		 * ABI compatibility so do the same thing manually here
		 */
		if (storage::prot_encrypted == m_parent->m_prot) {
			pe_file *f = dynamic_cast<pe_file*>(*pos);
			if (NULL != f)
				f->p_flush(true);
		} else {
			p_file *f = (*pos);
			if (NULL != f)
				f->p_flush(true);
		}
	}
}


void
storage_lock::opened_for_writing(p_file *pfile)
{
	open_files.push_back(pfile);
	AEGIS_DEBUG(1, "%s: added %p to open files", __func__, pfile);
}


void
storage_lock::closed(p_file *pfile)
{
	for (vector<p_file*>::iterator pos = open_files.begin();
		 pos != open_files.end(); 
		 pos++
	) {
		if (pfile == *pos) {
			AEGIS_DEBUG(1, "%s: removed %p from open files", __func__, pfile);
			open_files.erase(pos);
			break;
		}
	}
}


bool
storage_lock::readable(void)
{
	return can_read;
}

bool
storage_lock::writable(void)
{
	return can_write;
}


void
storage_lock::readable(bool set)
{
	can_read = set;
	/* If cannot read surely cannot write either
	 */
	if (!can_read)
		can_write = false;
}


void
storage_lock::writable(bool set)
{
	can_write = set;
	if (can_write)
		can_read = true;
}


void
storage_lock::unlink()
{
	if (0 > shm_unlink(mem_name.c_str())) {
		AEGIS_ERROR("%s: cannot unlink '%s' (%s)", __func__, 
					mem_name.c_str(), strerror(errno));
		return;
	}
	AEGIS_DEBUG(1, "%s: unlinked '%s'", __func__, mem_name.c_str());
}
