/* -*- mode:c; tab-width:4; c-basic-offset:4; -*-
 * This file is part of maemo-security-certman
 *
 * Copyright (C) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * Contact: Juhani Mäkelä <ext-juhani.3.makela@nokia.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

/*
 * The off_t type needs to be told how big it is
 */
#define _FILE_OFFSET_BITS 64

#include <stdio.h>
#include <unistd.h>
#include <malloc.h>
#include <string.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <aegis_crypto.h>
#include <aegis_storage.h>

static void
usage()
{
	fprintf(stderr, "Usage: rndwr [-e] [-c blocks] [-s blocksize] -f <filename>\n"
			" -e to use encrypted storage\n"
			" -f for the filename\n"
			" -c for the number of blocks written\n"
			" -s for the size of each block\n"
		);
}


int main (int argc, char* argv[])
{
	int rfd = -1, fd = -1, retcode = 0;
	size_t blsize = 16, blcount = 16, hits = 0, pos;
	int8_t *bmask = NULL;
	char *cbuf = NULL, *rbuf = NULL, c, *filename = NULL;
	off_t lpos = 0;
	int use_pstore = 0;
	aegis::storage *pstore = NULL;
	aegis::p_file *pfile = NULL;

    while (1) {
		int a = getopt(argc, argv, "ef:c:s:");
		if (0 > a)
			break;
		switch (a)
		{
		case 'e':
			use_pstore = 1;
			break;
		case 'c':
			blcount = atol(optarg);
			break;
		case 's':
			blsize = atol(optarg);
			break;
		case 'f':
			filename = optarg;
			break;
		default:
			usage();
			return -1;
		}
	}
	if (NULL == filename) {
		usage();
		return -1;
	}

    fprintf(stderr, "Testing with %u blocks of %u bytes each...",
            blcount, blsize);

	rfd = open("/dev/urandom", O_RDONLY);
	if (0 > rfd) {
		fprintf(stderr, "Cannot open random (%s)\n", strerror(errno));
		return -1;
	}

	if (!use_pstore) {
		fd = open(filename, O_RDWR | O_CREAT | O_TRUNC, 0600);
		if (0 > fd) {
			fprintf(stderr, "Cannot open '%s' (%s)\n", filename, strerror(errno));
			return -1;
		}
	} else {
		try {
			pstore = new aegis::storage("tests.rndwr", NULL, 
										aegis::storage::vis_private,
										aegis::storage::prot_encrypted);
			pfile = pstore->member(filename);
			if (!pfile->p_open(O_RDWR | O_CREAT)) {
				fprintf(stderr, "Cannot open '%s' (%s)\n", filename, aegis_crypto_last_error_str());
				return -1;
			}
		} catch (char* err) {
			fprintf(stderr, "Aegis error (%s)\n", err);
			return -1;
		}
	}

	cbuf = (char*)malloc(blsize);
	rbuf = (char*)malloc(blsize);
	bmask = (int8_t*)malloc(blcount * sizeof(int8_t));
    if (NULL == cbuf || NULL == rbuf || NULL == bmask) {
        fprintf(stderr, "Malloc failure (%s)\n", strerror(errno));
        return -1;
    }
	memset(bmask, '\0', blcount * sizeof(int8_t));

	do {
		int8_t tst;
		if (0 > read(rfd, &tst, sizeof(tst))) {
			fprintf(stderr, "Cannot read random (%s)\n", strerror(errno));
			break;
		}
		if (0 > read(rfd, &pos, sizeof(pos))) {
			fprintf(stderr, "Cannot read random (%s)\n", strerror(errno));
			break;
		}

		pos %= blcount;
		c = '0' + pos % 10;
		memset(cbuf, c, blsize);
		*(cbuf + blsize - 1) = '\n';
		lpos = (off_t)(pos * blsize);

		// AEGIS_DEBUG(1, "%s: block %d at %d", __func__, pos, lpos);

		if (tst % 2) {
			/* 
			 * Write blcount number of easily distinguishable 
			 * data blocks of size blsize in a random order 
			 */
			if (0 == bmask[pos]) {
				size_t written = 0;

				bmask[pos] = 1;
				hits++;
				
				if (!use_pstore) {
					lpos = lseek(fd, lpos, SEEK_SET);
					written = write(fd, cbuf, blsize);
				} else {
					written = pfile->p_write(lpos, cbuf, blsize);
				}
				if (written < blsize) {
					fprintf(stderr, "Error at %d, cannot write (%d < %d) (%s/%s)\n", 
                            (int)pos, written, blsize, strerror(errno), 
                            aegis_crypto_last_error_str());
					break;
				}
			}
		} else if (1 == bmask[pos]) {
			/* 
			 * Read an already written block and check it is what
			 * it is supposed to be.
			 */
            size_t could_read = 0;
            if (!use_pstore) {
                lpos = lseek(fd, lpos, SEEK_SET);
                could_read = read(fd, rbuf, blsize);
            } else {
                could_read = pfile->p_read(lpos, rbuf, blsize);
            }
            if (could_read == blsize) {
                if (0 != memcmp(cbuf, rbuf, blsize)) {
                    *(rbuf + blsize - 1) = '\0';
                    fprintf(stderr, "Error at %d, data does not match (%s)\n",
                            (int)pos, rbuf);
                }
            } else {
                fprintf(stderr, "Error at %d, cannot read (%d < %d) (%s/%s)\n",
                        (int)pos, could_read, blsize, strerror(errno), 
                        aegis_crypto_last_error_str());
            }
        }
	} while (hits < blcount);

    /*
     * Flush
     */
	if (use_pstore) {
		pfile->p_trunc(blsize * blcount);
		pfile->p_close();
        pstore->commit();
        delete pfile;
        pfile = NULL;
	}

	close(rfd);

	/*
	 * Print out and check the data
	 */
	try {
        char* data = NULL;
        size_t len;

		if (use_pstore) {
			if (0 != pstore->get_file(filename, (RAWDATA_RPTR)&data, &len)) {
                fprintf(stderr, "get_file failed (%s)\n", 
                        aegis_crypto_last_error_str());
			}
            delete pstore;
		} else {
            char c, *to;
            struct stat fs;
            if (0 > fstat(fd, &fs)) {
                fprintf(stderr, "Fstat failed (%s)\n", strerror(errno));
                goto error;
            }
            len = fs.st_size;
            to = data = (char*)malloc(len);
            lpos = lseek(fd, 0, SEEK_SET);
            while (0 < read(fd, &c, 1)) {
                *to++ = c;
            }
            close(fd);
        }

        if (len != (blcount * blsize)) {
            fprintf(stderr, "Wrong result file size %u, expected %u\n",
                    len, blcount * blsize);
            goto error;
        }

        size_t i, j;
        for (i = 0; i < blcount; i++) {
            for (j = 0; j < blsize; j++) {
                c = '0' + i % 10;
                if (blsize - 1 == j)
                    c = '\n';
                if (c != *(data + i * blsize + j)) {
                    fprintf(stderr, "Expected '%c' at %u, not '%c'\n",
                            c, i * blsize + j, *(data + i * blsize + j));
                    goto error;
                }
            }
        }

        if (use_pstore)
            aegis_crypto_free(data);
        else
            free(data);
		
	} catch (int e) {
		fprintf(stderr, "Exception (%s)\n", strerror(e));
	}
    fprintf(stderr, "Success\n");
    goto end;

 error:
    fprintf(stderr, "Test failed\n");
    retcode = -1;

 end:
	free(bmask);
	free(cbuf);
	free(rbuf);
	return retcode;
}
