/* -*- mode:c; tab-width:4; c-basic-offset:4; -*-
 *
 * This file is part of maemo-security-certman
 *
 * Copyright (C) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * Contact: Juhani Mäkelä <ext-juhani.3.makela@nokia.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

/*
 * Some examples of how to use the maemosec crypto API
 */

#include <string.h>
#include <stdio.h>
#include <malloc.h>
#include <aegis_crypto.h>

/*
 * Placeholders to simulate message sending
 * and receiving.
 */
static char *msg_ph = NULL;
static size_t len_ph = 0;

static void
show_msg(char *msg, size_t of_len)
{
    char* c;
    for (c = msg; 0 < of_len--; c++) {
        if (' ' <= *c && 127 > *c)
            printf("%c", *c);
        else
            printf("\\x%02hx", *c);
    }
}

static void
send(RAWDATA_PTR msg, size_t of_len)
{
    printf("send   :%d:", of_len);
    show_msg(msg, of_len);
    printf("\n");

    if (NULL != msg_ph)
        free(msg_ph);
    msg_ph = malloc(of_len);
    memcpy(msg_ph, msg, of_len);
    len_ph = of_len;
}

static void 
receive(RAWDATA_RPTR msg, size_t *of_len)
{
    printf("receive:%d:", len_ph);
    show_msg(msg_ph, len_ph);
    printf("\n");

    *msg = msg_ph;
    msg_ph = NULL;
    *of_len = len_ph;
    len_ph = 0;
}

/*
 * The main program 'sends' and 'receives' message to itself
 * just to demonstrate how to sign and verify messages and to
 * encrypt and decrypt them. The methods can be combined, too.
 */
int main(void)
{
    /* A NULL terminated message */
    char *msg_to_be_signed = "A message to be signed";

    RAWDATA_PTR signed_msg;
    char *token_name;
    char* signature_str = NULL;
    struct aegis_signature_t signature;
    size_t msg_tot_length;
    size_t signature_str_len;
    aegis_crypto_result rc;
    aegis_system_mode_t cmode;

    /*
     * Example 1:
     * Make a signature with resource token name "tcb".
     */
    token_name = "tcb";
    rc = aegis_crypto_sign((unsigned char*)msg_to_be_signed, 
                              strlen(msg_to_be_signed), 
                              token_name, &signature);
    if (aegis_crypto_ok != rc)
        /*
         * If calling process does not have the "tcb" credential, 
         * the operation fails
         */
        goto error;

    /*
     * The raw signature has fixed length but does not contain
     * the signer token string. Convert it into a human readable
     * string. Notice that this function does not return an error
     * code but the length of the signature string.
     */
    signature_str_len = aegis_crypto_signature_to_string(&signature, 
                                                            aegis_as_base64, 
                                                            token_name, 
                                                            &signature_str);

    /*
     * Append the signature to the message
     */
    msg_tot_length = strlen(msg_to_be_signed) + 1 + signature_str_len + 1;
    signed_msg = (char*)malloc(msg_tot_length);
    strcpy(signed_msg, msg_to_be_signed);
    strcat(signed_msg, ":");
    strcat(signed_msg, signature_str);
	aegis_crypto_free((RAWDATA_PTR)signature_str);
	signature_str = NULL;

    send(signed_msg, msg_tot_length);
    free(signed_msg);
    signed_msg = NULL;

    /*
     * Example 2:
     * Verify a signed message by using the token name embedded in 
     * the signature.
     */
    receive(&signed_msg, &msg_tot_length);
    if (0 >= msg_tot_length)
        goto error;

    signature_str = strchr(signed_msg, ':');
    if (signature_str) 
        *signature_str++ = '\0';
    else {
        AEGIS_ERROR("invalid message, missing separator");
        goto error;
    }

    rc = aegis_crypto_string_to_signature(signature_str, 
                                             &signature, 
                                             &token_name);
    if (aegis_crypto_ok != rc)
        goto error;

    rc = aegis_crypto_verify(&signature, token_name, 
                                (unsigned char*)signed_msg, 
                                strlen(signed_msg),
                                &cmode);
    if (aegis_crypto_ok != rc)
        goto error;

    /*
     * At this point we know that whoever sent the message has 
     * the "tcb" privilege.
     */
    printf("Signature OK, made in %s mode\n", 
           aegis_system_protected==cmode?"protected":"open");

    /*
     * Example 3:
     * Encrypt a message  with resource token name "tcb".
     */

    /* A NULL terminated message */
    char *msg_to_be_encrypted = "A message to be encrypted";
    RAWDATA_PTR cipher_msg = NULL;
    size_t result_size = 0;

    token_name = "tcb";
    rc = aegis_crypto_encrypt((unsigned char*)msg_to_be_encrypted, 
                                 strlen(msg_to_be_encrypted),
                                 token_name, 
                                 &cipher_msg, 
                                 &result_size);
    if (aegis_crypto_ok != rc)
        /*
         * Fails if the caller doesn't have the "tcb" credential
         */
        goto error;

    send(cipher_msg, result_size);
    aegis_crypto_free(cipher_msg);

    /*
     * Decrypt the message.
     */
    RAWDATA_PTR result_msg;

    receive(&cipher_msg, &msg_tot_length);
    rc = aegis_crypto_decrypt((unsigned char*)cipher_msg, 
                                 msg_tot_length, 
                                 token_name, 
                                 &result_msg, 
                                 &result_size);
    if (aegis_crypto_ok != rc)
        /*
         * Fails if the caller doesn't have the "tcb" credential
         */
        goto error;

    if (strlen(msg_to_be_encrypted) == result_size
        && 0 == memcmp(result_msg, msg_to_be_encrypted, result_size))
        printf("Decrypt OK\n");
    else {
        printf("Message corrupted: original(%d) '%s', result(%d) '", 
			   msg_tot_length, msg_to_be_encrypted, result_size);
        show_msg(result_msg, result_size);
        printf("'\n");
    }
    return(0);

 error:
    printf("Error detected: %s\n", aegis_crypto_last_error_str());
    return(-1);
}
