/* -*- mode:c++; tab-width:4; c-basic-offset:4; -*-
 *
 * This file is part of maemo-security-certman
 *
 * Copyright (C) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * Contact: Juhani Mäkelä <ext-juhani.3.makela@nokia.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <stdio.h>
#include <time.h>
#include <errno.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <pwd.h>
#include <grp.h>
#ifdef USE_CREDS
#include <sys/creds.h>
#endif

#define AEGIS_SHOW_ERRORS
#include <aegis_common.h>
#include <aegis_crypto.h>
#include <aegis_storage.h>

using namespace std;
using namespace aegis;

extern int resolve_symlinks;
static bool underlying_files = false;

static void 
usage(const char* bin_name)
{
	printf(
		"Usage: %s [-ilID]  -r <token> -s <store-name:{G|S|P}{s|e}>\n"
        "  -<a|d|v|p|f> <filename> [<-t|-L> <filename>]\n"
		" -s to give the store name and attributes, separated by a colon\n"
		"    Attribute flags are visibility G=global, S=shared, P=private\n"
        "      and protection s=signed, e=encrypted\n"
		"    The default is a private, signed store if no attributes are given\n"
		" -r to specify the protecting resource token (using application id by default)\n"
        "    (must be owned by the process, use -I to check available tokens)\n\n"
		" -v (default) to verify the store\n"
        "    (if no filename is given, all files will be verified)\n"
		" -a to add a file or a symlink to the store\n"
        "   -i to read the contents of the file to add/update from stdin\n"
		"   -L <points> to add a symlink to an existing file in stead of a new file\n"
		" -p <filename> to print the file contents to stdout\n"
		" -d to remove a file from the store\n"
		" -f to rename a file or a link in the store\n"
        "   -t to give the new name\n\n"
        " -l to list all stores in the system by type or files in a given store\n"
        " -U <user-id> set the effective user-id and group-id of the process\n"
        " -x to prevent resolving symlinks when adding files in a signed store\n"
        " -I to show credentials of the process\n"
		" -D to delete the store permanently (use cautiously)\n",
        bin_name
     );
}

static int
show_storage_name(int pos, void* item, void* ctx)
{
    if (!underlying_files)
        printf("\t%d: %s\n", pos + 1, (char*)item);
    else
        printf("%s:%s\n", (char*)item, (char*)ctx);
    return 0;
}


static void
show_file_details(storage *ss, const char* filename)
{
    struct stat fs;
    char flags[20];

    AEGIS_DEBUG(1, "%s: %s", __func__, filename);
    if (0 <= ss->stat_file(filename, &fs)) {
        if (S_ISLNK(fs.st_mode))
            flags[0] = 'l';
        else if (S_ISDIR(fs.st_mode))
            flags[0] = 'd';
        else
            flags[0] = '-';

        if (S_IRUSR & fs.st_mode)
            flags[1] = 'r';
        else
            flags[1] = '-';
        if (S_IWUSR & fs.st_mode)
            flags[2] = 'w';
        else
            flags[2] = '-';
        if (S_IXUSR & fs.st_mode)
            flags[3] = 'x';
        else
            flags[3] = '-';

        if (S_IRGRP & fs.st_mode)
            flags[4] = 'r';
        else
            flags[4] = '-';
        if (S_IWGRP & fs.st_mode)
            flags[5] = 'w';
        else
            flags[5] = '-';
        if (S_IXGRP & fs.st_mode)
            flags[6] = 'x';
        else
            flags[6] = '-';

        if (S_IROTH & fs.st_mode)
            flags[7] = 'r';
        else
            flags[7] = '-';
        if (S_IWOTH & fs.st_mode)
            flags[8] = 'w';
        else
            flags[8] = '-';
        if (S_IXOTH & fs.st_mode)
            flags[9] = 'x';
        else
            flags[9] = '-';
        flags[10] = '\0';

    } else {
        printf("%s (CORRUPTED)\n", filename);
        return;
    }

    printf("%s %d ", flags, (int)fs.st_nlink);
    if (S_ISLNK(fs.st_mode)) {
        string points_to;
        ss->read_link(filename, points_to);
        printf("%s -> %s\n", filename, points_to.c_str());
        return;
    }
    struct passwd *pwd = getpwuid(fs.st_uid);
    if (pwd)
        printf("%-8s ", pwd->pw_name);
    else
        printf("%-8s ", "?");

    struct group *grp = getgrgid(fs.st_gid);
    if (grp) 
        printf("%-8s ", grp->gr_name);
    else
        printf("%-8s ", "?");

    printf("%8ld ", (long)fs.st_size);

    struct tm *tim = localtime(&fs.st_mtime);
    if (tim) {
        printf("%04d-%02d-%02d %02d:%02d ",
               1900 + tim->tm_year, 1 + tim->tm_mon, 
               tim->tm_mday, tim->tm_hour, tim->tm_min);
    } else
        printf("%s-%s-%s %s:%s ", "????", "??", "??", "??", "??");

    printf("%s\n", filename);
}


enum cmd_type {
	cmd_add,
	cmd_add_list,
	cmd_update,
	cmd_del,
	cmd_del_all,
	cmd_print,
	cmd_verify,
	cmd_list,
    cmd_rename
} cmd = cmd_verify;

#define UNDEFINED_VISIBILITY (storage::visibility_t)-1
#define UNDEFINED_PROTECTION (storage::protection_t)-1

int
main(int argc, char* argv[])
{
	int a, rc = 0;
	string storagename;
	char *flags = NULL, *filename = NULL, *token = NULL, *points = NULL;
	storage* ss = NULL;
	bool was_changed = false;
	bool do_recover = false;
	storage::visibility_t storvis  = UNDEFINED_VISIBILITY;
	storage::protection_t storprot = UNDEFINED_PROTECTION;
    unsigned char* buf = NULL;
    size_t len = 0;
 
	if (1 == argc) {
		usage(argv[0]);
		return(-1);
	}

    while (1) {
		a = getopt(argc, argv, ":s:a:d:p:v:r:f:t:U:L:DhilIxunAF");
		if (a < 0) {
			break;
		}
		switch(a) 
		{
		case 'n':
			break;

		case 'r':
			token = strdup(optarg);
			break;

        case 'x':
            resolve_symlinks = 0;
            break;

        case 'U':
            {
                struct passwd *pwd;
                if (NULL != (pwd = getpwnam(optarg))) {
                    if (0 > setgid(pwd->pw_gid)) {
                        AEGIS_ERROR("cannot set group-id (%s)", strerror(errno));
                        return -1;
                    }
                    if (0 > setuid(pwd->pw_uid)) {
                        AEGIS_ERROR("cannot set user-id (%s)", strerror(errno));
                        return -1;
                    }
                }
            }
            break;

		case 'L':
			points = strdup(optarg);
			break;

		case 's':
			flags = strchr(optarg, ':');
			if (flags) {
				storagename.assign(optarg, flags - optarg);
				flags++;
				while (*flags) {
					switch (*flags) 
						{
						case 'G':
							storvis = storage::vis_global;
							break;

						case 'S':
							storvis = storage::vis_shared;
							break;

						case 'P':
							storvis = storage::vis_private;
							break;

						case 'e':
							storprot = storage::prot_encrypted;
							break;
							
						case 's':
							storprot = storage::prot_signed;
							break;

						default:
							AEGIS_ERROR("Invalid attribute '%c'", *flags);
						}
					flags++;
				}
            }
            if (   UNDEFINED_VISIBILITY == storvis
                || UNDEFINED_PROTECTION == storprot)
            {
                AEGIS_ERROR("missing storage attributes in '%s'", optarg);
                return -1;
            }
			break;

		case 'a':
            filename = optarg;
		    cmd = cmd_add;
			break;

        case 'A':
		    cmd = cmd_add_list;
			break;

		case 'd':
            filename = optarg;
			cmd = cmd_del;
			break;

		case 'v':
            /*
             * Allow leaving option out of verify
             */
            if (':' == a && 'v' != optopt)
                goto wrong_option;
		    filename = optarg;
			cmd = cmd_verify;
			break;

		case 'p':
            if (':' == a && 'p' != optopt)
                goto wrong_option;
			filename = optarg;
			cmd = cmd_print;
			break;

		case ':':
            /* 'v' and 'p' can be given without options
             */
			switch (optopt) {
			case 'p':
				cmd = cmd_print;
				break;
			case 'v':
				cmd = cmd_verify;
				break;
			default:
                goto wrong_option;
			}
		    filename = optarg;
			break;

        case 'f':
			filename = optarg;
			cmd = cmd_rename;
			break;

        case 't':
            points = strdup(optarg);
            break;

        case 'u':
            cmd = cmd_list;
            underlying_files = true;
            break;

		case 'i':
			{
				int c;
                size_t blen = 1024;

				buf = (unsigned char*)malloc(blen);
				if (!buf) {
					AEGIS_ERROR("cannot malloc (%s)", strerror(errno));
					return(-1);
				}
				do {
					c = getchar();
					if (c >= 0) {
						if (len == blen) {
							unsigned char* swp;

							blen *= 2;
							swp = (unsigned char*)malloc(blen);
							if (!swp) {
								AEGIS_ERROR("cannot malloc (%s)", strerror(errno));
								return(-1);
							}
							memcpy(swp, buf, len);
							free(buf);
							buf = swp;
						}
						buf[len++] = (unsigned char)c;
					} else
						break;
				} while (true);
			}
			break;

        case 'l':
			cmd = cmd_list;
			break;

        case 'D':
            cmd = cmd_del_all;
			underlying_files = true;
            break;

        case 'I':
#ifdef USE_CREDS
            {
                int i;
                creds_t creds;

                creds = creds_gettask(0);
                printf("Credentials:\n");
                for (i = 0; i < 1000; i++) {
                    creds_value_t cval;
                    creds_type_t cred = creds_list(creds, i, &cval);
                    if (CREDS_BAD != cred) {
                        char cred_name[256];
                        rc = creds_creds2str(cred, cval, cred_name, sizeof(cred_name));
                        if (0 <= rc)
                            printf("\t%s\n", cred_name);
                    } else
                        break;
                }
                creds_free(creds);
            }
            return 0;
#endif
            break;

		case 'F':
			printf("Recovery mode\n");
			do_recover = true;
			break;
		   
		default:
        wrong_option:
			usage(argv[0]);
			return -1;
		}
	}

    if (0 == aegis_crypto_init()) {
		AEGIS_ERROR("Crypto init failed, exit\n%s", aegis_crypto_last_error_str());
		return -1;
    }

	if ("" != storagename) {
		ss = new storage(storagename.c_str(), token, storvis, storprot);
		if (!ss) {
			AEGIS_ERROR("cannot open/create storage (%s)", 
						aegis_crypto_last_error_str());
			return -1;
		}
		if (storage::writable != ss->status()) {
			AEGIS_DEBUG(1, "Existing storage '%s' with %s access", 
						ss->name(), storage::readable == ss->status() 
						? "read" : "no");
		} else if (0 == ss->nbrof_files()) {
            AEGIS_DEBUG(1, "Created new storage '%s' in %s.", 
                        ss->name(), ss->filename());
		} else {
			AEGIS_DEBUG(1, "Storage '%s' contains %d files.", 
						   ss->name(), ss->nbrof_files());
		}
	}

	storage::stringlist files;

	switch(cmd) 
		{
		case cmd_add:
			if (NULL == ss) {
				AEGIS_ERROR("No storage given");
				rc = -1;
                goto error;
			}
			if (buf) {
				rc = ss->put_file(filename, buf, len);
				if (rc != 0) {
					AEGIS_ERROR("cannot update '%s' (%s)", filename, 
								aegis_crypto_last_error_str());
				} else {
					was_changed = true;
				}

			} else if (points) {
				if (ss->contains_file(points)) {
					ss->add_link(filename, points);
					was_changed = true;
				} else
					AEGIS_ERROR("Store does not contain file '%s'", points);

			} else if (!ss->contains_file(filename)) {
				ss->add_file(filename);
                if (ss->contains_file(filename))
                    was_changed = true;
                else
                    AEGIS_ERROR("'%s' does not exist", filename);
			} else
				AEGIS_ERROR("Store already contains file '%s'", filename);
			break;

		case cmd_add_list:
			if (NULL == ss) {
				AEGIS_ERROR("No storage given");
				rc = -1;
                goto error;
			} else {
                char addname[1024];
                while (NULL != fgets(addname, sizeof(addname), stdin)) {
                    if (strlen(addname) && '\n' == addname[strlen(addname)-1])
                        addname[strlen(addname) - 1] = '\0';
                    ss->add_file(addname);
                    if (ss->contains_file(addname))
                        was_changed = true;
                    else
                        AEGIS_ERROR("'%s' does not exist", addname);
                }
            }
            break;
            
		case cmd_del:
			if (NULL == ss) {
				AEGIS_ERROR("No storage given");
                rc = -1;
                goto error;
			}
			if (ss->contains_file(filename)) {
				ss->remove_file(filename);
                was_changed = true;
			} else if (ss->contains_link(filename)) {
				ss->remove_link(filename);
                was_changed = true;
            } else {
                AEGIS_ERROR("'%s' is not in the given store", filename);
                rc = -1;
            }
			break;

		case cmd_verify:
			bool res;

			if (NULL == ss) {
				AEGIS_ERROR("No storage given");
                rc = -1;
                goto error;
			}

			if (filename) {
				if (   !ss->contains_file(filename)
                    && !ss->contains_link(filename))
                {
					AEGIS_ERROR("'%s' is not in '%s' (%s)", 
								filename, ss->name(), ss->filename());
					break;
				}

				if (NULL == buf)
					res = ss->verify_file(filename);
				else
					res = ss->verify_content(filename, buf, len);

                if (res) {
                    printf("%s: OK\n", filename);
                } else {
					if (do_recover) {
						p_file *pf = ss->member(filename);
						if (pf->p_open(O_RDONLY | O_RECOVER)) {
							pf->p_close();
							printf("%s: RECOVERED\n", filename);
							was_changed = true;
						} else {
							printf("%s: RECOVERY FAILED\n", filename);
						}
					} else {
						printf("%s: FAILED\n", filename);
						rc = -1;
					}
                }

			} else {
				ss->get_files(files);
				for (size_t i = 0; i < files.size(); i++) {
                    printf("%s: ", files[i]);
                    if (ss->verify_file(files[i]))
                        printf("OK\n");
                    else {
						if (do_recover) {
							p_file *pf = ss->member(files[i]);
							if (pf->p_open(O_RDWR | O_RECOVER)) {
								pf->p_close();
								printf("RECOVERED\n");
								was_changed = true;
							} else {
								printf("RECOVERY FAILED\n");
							}
						} else {
							printf("FAILED\n");
							rc = -1;
						}
                    }
				}
                ss->release(files);
			}
			break;

		case cmd_print:
			RAWDATA_PTR buf;
			size_t len;
				
			if (NULL == ss) {
				AEGIS_ERROR("No storage given");
				rc = -1;
                goto error;
			}

			if (filename) {
				rc = ss->get_file(filename, &buf, &len);
				if (0 == rc) {
					for (size_t i = 0; i < len; i++) {
						putchar(*((char*)buf + i));
					}
				} else {
					AEGIS_ERROR("failed to open file (%s)", aegis_crypto_last_error_str());
					goto error;
				}
				ss->release_buffer(buf);
			} else {
				int pos = ss->get_files(files);
				for (int i = 0; i < pos; i++) {
					rc = ss->get_file(files[i], &buf, &len);
					if (0 == rc) {
						for (size_t s = 0; s < len; s++) {
							putchar(*((char*)buf + s));
						}
					} else {
						AEGIS_ERROR("failed to open file (%s)", aegis_crypto_last_error_str());
						goto error;
					}
					ss->release_buffer(buf);
				}
			}
			break;

		case cmd_list:
			if (NULL == ss) {
                if (!underlying_files)
                    printf("Global signed:\n");
				storage::iterate_storage_names(storage::vis_global, 
											   storage::prot_signed, 
											   NULL, show_storage_name, 
                                               (void*)"Gs");
                if (!underlying_files)
                    printf("Shared signed:\n");
				storage::iterate_storage_names(storage::vis_shared, 
											   storage::prot_signed, 
											   NULL, show_storage_name, 
                                               (void*)"Ss");
                if (!underlying_files)
                    printf("Shared encrypted:\n");
				storage::iterate_storage_names(storage::vis_shared, 
											   storage::prot_encrypted, 
											   NULL, show_storage_name, 
                                               (void*)"Se");
                if (!underlying_files)
                    printf("Private signed:\n");
				storage::iterate_storage_names(storage::vis_private, 
											   storage::prot_signed, 
											   NULL, show_storage_name, 
                                               (void*)"Ps");
                if (!underlying_files)
                    printf("Private encrypted:\n");
				storage::iterate_storage_names(storage::vis_private, 
											   storage::prot_encrypted, 
											   NULL, show_storage_name, 
                                               (void*)"Pe");
			} else {
				int pos, i;
                if (!underlying_files) {
                    pos = ss->get_files(files);
                    for (i = 0; i < pos; i++) {
                        show_file_details(ss, files[i]);
                    }
                } else {
                    pos = ss->get_ufiles(files);
                    for (i = 0; i < pos; i++) {
                        printf("%s\n", files[i]);
                    }
                }
                ss->release(files);
			}
			break;

        case cmd_del_all:
			if (NULL == ss) {
				AEGIS_ERROR("No storage given");
				rc = -1;
                goto error;
			}
			if (storage::writable == ss->status()) {
				AEGIS_DEBUG(1, "Remove all files");
				if (ss->remove_all_files()) {
					printf("Storage '%s' removed\n", ss->name());
				} else {
					AEGIS_ERROR("Failed to remove all files (%s)",
								strerror(errno));
				}
            } else if (storage::prot_encrypted == ss->protection()) {
				AEGIS_DEBUG(1, "Get ufiles and remove them manually");
				ssize_t removed = ss->get_ufiles(files);
				if (0 < removed) {
					for (size_t i = 0; i < files.size(); i++) {
						AEGIS_DEBUG(1, "Unlink '%s'", files[i]);
						if (file_exists(files[i]) && 0 > unlink(files[i])) {
							AEGIS_ERROR("Failed to unlink '%s' (%s)",
										files[i], strerror(errno));
							removed--;
						}
					}
					if (removed != (int)files.size())
						printf("Deleted %d files out of %d\n", removed, (int)files.size());

				} else if (0 == removed) {
					printf("Nothing to delete\n");

				} else {
					AEGIS_ERROR("Storage '%s' could not be removed (%s)", 
							ss->name(), aegis_crypto_last_error_str());
					rc = -1;
					goto error;
				}
            } else {
				AEGIS_DEBUG(1, "Remove index file of a signed store");
				if (0 > unlink(ss->filename())) {
					AEGIS_ERROR("Storage '%s' could not be removed (%s)", 
								ss->name(), strerror(errno));
				}
			}
            break;

        case cmd_rename:
			if (NULL == ss) {
				AEGIS_ERROR("No storage given");
				rc = -1;
                goto error;
			}
			if (NULL == points) {
				AEGIS_ERROR("Give also new name when renaming");
				rc = -1;
                goto error;
			}
			errno = 0;
            ss->rename(filename, points);
            if ((ss->contains_file(points) || ss->contains_link(points))
                && !ss->contains_file(filename) && !ss->contains_link(filename)) 
            {
                was_changed = true;
            } else {
				AEGIS_ERROR("Error renaming '%s' to '%s' (%s)", 
							filename, points,
							aegis_crypto_last_error_str());
                rc = -1;
                goto error;
            }
            break;

		default:
			;
	}

	if (was_changed) {
		AEGIS_DEBUG(1, "Updating changes.");
		if (!ss->commit()) {
			AEGIS_ERROR("Failed to commit (%s)", strerror(errno));
			rc = -1;
		}
	} 

error:    
    if (ss) {
        delete ss;
        ss = NULL;
    }
    if (buf) {
        free(buf);
        buf = NULL;
    }
	if (token) {
		free(token);
		token = NULL;
	}
	if (points) {
		free(points);
		points = NULL;
	}
    aegis_crypto_finish();

	return(rc);
}
