/* -*- mode:c; tab-width:4; c-basic-offset:4; -*- */

#include <aegis_common.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <errno.h>
#include <unistd.h>
#include <fcntl.h>
#include <ctype.h>
#include <sys/mman.h>
#include <sys/stat.h>
#include <openssl/sha.h>
#include <aegis_crypto.h>

enum {encode, decode, sha1sum} mode = encode;

static void
usage(void)
{
	printf("%s\n", "Usage: b64 [-e(ncode)|-d(ecode)] <file> [-o <outfile>]");
}

int main(int argc, char* argv[])
{
	int a, fd, ofd = -1;
	char *data, *filename;
	char *edata, *c;
	ssize_t len, flen, wlen;
	struct stat fs;

    while (1) {
		a = getopt(argc, argv, "edso:");
		if (a < 0) {
			break;
		}
		switch(a) 
		{
		case 'e':
			mode = encode;
			break;
		case 'd':
			mode = decode;
			break;
		case 's':
			mode = sha1sum;
			break;
		case 'o':
			ofd = open(optarg, O_CREAT | O_RDWR, 0644);
			if (0 > ofd) {
				fprintf(stderr, "ERROR: cannot open '%s' (%s)\n",
						optarg, strerror(errno));
				exit(1);
			}
			AEGIS_DEBUG(1, "Output file %ld", ofd);
			break;
		default:
			usage();
			exit(0);
		}
	}
	if (optind == argc) {
		usage();
		exit(0);
	}
    filename = argv[optind];
    if (NULL == filename || 4096 < strlen(filename)) {
		usage();
		exit(0);
    }

	fd = open(filename, O_RDONLY);
	if (0 > fd) {
		AEGIS_ERROR("cannot open '%s' (%s)\n",
                    filename, strerror(errno));
		exit(0);
	}

	if (0 > fstat(fd, &fs)) {
		AEGIS_ERROR("cannot stat '%s' (%s)\n",
                    filename, strerror(errno));
		exit(0);
    }

	flen = len = fs.st_size;
	data = (char*)mmap(NULL, flen, PROT_READ, MAP_PRIVATE, fd, 0);

	if (MAP_FAILED == data) {
		close(fd);
		AEGIS_ERROR("cannot map '%s' (%s)\n",
                    filename, strerror(errno));
		exit(0);
	}

	if (encode == mode) {
		edata = base64_encode(data, len);
		if (NULL == edata) {
			close(fd);
			AEGIS_ERROR("cannot encode '%s' (%s)\n",
                        filename, strerror(errno));
			exit(0);
		}
		len = strlen(edata);
		if (-1 == ofd) {
#define LLEN 64
			for (c = edata; len > LLEN; len -= LLEN) {
				char lbuf[LLEN + 1];
				memmove(lbuf, c, LLEN);
				lbuf[LLEN] = '\0';
				printf("%s\n", lbuf);
				c += LLEN;
			}
			printf("%s\n", c);
		} else {
			wlen = write(ofd, data, len);
			if (wlen != len)
				fprintf(stderr, "ERROR: %s (written %zd != tried %zd)\n", 
						strerror(errno), wlen, len);
			close(ofd);
		}

	} else if (decode == mode) {
		void* buf;
		len = base64_decode(data, &buf);
		AEGIS_DEBUG(1, "Decoded %zd bytes, %p write to %d", len, buf, ofd);
		if (-1 == ofd) {
			printf("%s", (char*)buf);
		} else {
			wlen = write(ofd, buf, len);
			if (wlen != len)
				fprintf(stderr, "ERROR: %s (written %zd != tried %zd)\n", 
						strerror(errno), wlen, len);
			close(ofd);
		}
		free(buf);
	} else if (sha1sum == mode) {
		unsigned char csum [SHA_DIGEST_LENGTH];
		if (NULL != SHA1((unsigned char*)data, len, csum)) {
			size_t i;
			for (i = 0; i < SHA_DIGEST_LENGTH; i++)
				printf("%02x", csum[i]);
			printf(" ");
			edata = base64_encode(csum, SHA_DIGEST_LENGTH);
			if (NULL != edata) {
				printf("%s\n", edata);
				aegis_crypto_free(edata);
			}
		}
	}
	munmap(data, flen);
	close(fd);
	return(0);
}
