# -*- coding: utf-8 -*-

'''
Advanced Interface Switcher
2010-2012(c) Kirill Plyashkevich <ru.spirit@gmail.com>
WLAN interface object
'''

import dbus, gobject
from advifswlib.interfaces.Abstract import InterfaceState, AbstractInterface
from advpowcommon.util.execn import *

class WLANInterface(AbstractInterface):

  wifi_module_path = '/org/freedesktop/Hal/Manager'
  wifi_module_iface = 'org.freedesktop.Hal.Manager'
  wifi_module_signal = None
  wifi_cmd = {'up': 'sudo /usr/bin/advifsw_wlan_up.sh',
              'down': 'sudo /usr/bin/advifsw_wlan_down.sh',
              'unload': 'sudo /usr/bin/advifsw_wlan_unload.sh',
              'load0': 'sudo /usr/bin/advifsw_wlan_load0.sh',
              'load1': 'sudo /usr/bin/advifsw_wlan_load1.sh',
              'get_driver': "lsmod | grep '^wl12\([x]\{2\}\|51\)\ ' | sed -n s/\ .*//p",
              'check_be': 'if [ -d /lib/modules/`uname -r` ]; then echo 1; fi'
             }
  # off: down > unload
  wifi_turn_off_cmd = [wifi_cmd['down'], wifi_cmd['unload']]
  wifi_drivers = ['wl12xx', 'wl1251']
  
  wifi_icd2_path = '/com/nokia/icd2'
  wifi_icd2_iface = 'com.nokia.icd2'
  wifi_icd2_state_signal = 'state_sig'
  wifi_icd2_statistics_signal = 'statistics_sig'
  wifi_icdui_path = '/com/nokia/icd_ui'
  wifi_icdui_iface = 'com.nokia.icd_ui'

  def pre_init(self):
    self.__strength_timeout = None
    self.__autooff = None
    self.has_be = get_shell_command_output(self.wifi_cmd['check_be']) != ''
    self.state = {'name': 'WLAN',
                  'dbus': {'on': {'name': self.wifi_module_signal, 
                                  'iface': self.wifi_module_iface, 
                                  'path': self.wifi_module_path, 
                                  'args': {'member_keyword': 'member'}},
                           'connected': {'name': self.wifi_icd2_state_signal, 
                                         'iface': self.wifi_icd2_iface, 
                                         'path': self.wifi_icd2_path, 
                                         'args': {}},
                           'statistics': {'name': self.wifi_icd2_statistics_signal, 
                                          'iface': self.wifi_icd2_iface, 
                                          'path': self.wifi_icd2_path, 
                                          'args': {}}},
                  'images': ('ifsw_statusarea_wlan_off', 
                             'ifsw_statusarea_wlan_signal_0', 
                             'ifsw_statusarea_wlan_signal_3',
                             'ifsw_statusarea_wlan_connecting',
                             'ifsw_statusarea_wlan_disconnecting'),
                  'images_st': ('ifsw_statusarea_wlan_signal_0', 
                                'ifsw_statusarea_wlan_signal_1', 
                                'ifsw_statusarea_wlan_signal_1', 
                                'ifsw_statusarea_wlan_signal_2', 
                                'ifsw_statusarea_wlan_signal_3', 
                                'ifsw_statusarea_wlan_signal_3'),
                  'text': ['Off', 'On', '<>', 'Connecting', 'Disconnecting'],
                  'show': (True, True, False, False, True),
                  'banner': (True, True, True, True, True),
                  'value': InterfaceState.off,
                  'strength': 0,
                  'modes': ['wlan-stock'],
                  'mode': 0}
    if self.has_be:
      self.state['modes'] = ['wlan-stock', 'wlan-bleeding-edge']
    self.interfaces = {'ICD': [self.wifi_icd2_iface, self.wifi_icd2_path, 
                               self.wifi_icd2_iface, '/usr/sbin/icd2', False],
                       'ICD_UI': [self.wifi_icdui_iface, self.wifi_icdui_path, 
                                  self.wifi_icdui_iface, '/usr/sbin/icd2', 
                                  False]}

  def post_init(self):
    self.set_config_option('disconnected-timeout', (-1,10,30,60,300))

  def process_signal_on(self, *args, **kwargs):
    result = None
    mode = None
    if args[0] == '/org/freedesktop/Hal/devices/platform_wl12xx':
      if kwargs['member'] == 'DeviceAdded':
        mode = self.__get_mode(self.__get_driver())
        result = InterfaceState.on
        if self.get_config_value('show-connection-dialog'):
          #self.scan_request()
          gobject.idle_add(self.show_connection_dialog)
          #self.show_connection_dialog()
      elif kwargs['member'] == 'DeviceRemoved':
        result = InterfaceState.off
        self.__cancel_autooff()
        self.__cancel_get_strength()
    return result, mode, None

  def process_signal_connected(self, *args):
    result = None
    if len(args) == 8 and args[3].startswith('WLAN_'):
      if args[7] == 2:
        result = InterfaceState.connected
        self.__cancel_autooff()
        if self.get_config_value('show-signal-strength'):
          self.__strength_timeout = gobject.timeout_add(1000, self.get_strength)
        ap_bytename = args[5]
        apname = ''.join([str(char) for char in ap_bytename if char])
      elif args[7] == 0 and self.state['value'] != 0: #State: On
        result = InterfaceState.on
        if self.state['value'] == InterfaceState.disconnecting:
          #schedule autoff if needed
          autooff = int(self.get_config_value('disconnected-timeout'))
          if autooff > 0:
            self.__autooff = gobject.timeout_add(autooff*1000, self.turn_off)
      elif args[7] in (1,15) and self.state['value'] != InterfaceState.off:
        #result = InterfaceState.on
        result = InterfaceState.connecting
        self.__cancel_autooff()
      elif args[7] == 3 and self.state['value'] == InterfaceState.connected:
        #result = InterfaceState.on
        result = InterfaceState.disconnecting
        self.__cancel_get_strength()
      else:
        result = self.state['value']
    return result, None, None

  def process_signal_statistics(self, *args):
    return None, None, min(int(args[7]), 5)

  def get_status(self):
    driver = self.__get_driver()
    mode = self.__get_mode(driver)
    if driver == '':
      return InterfaceState.off, mode, None
    else:
      try:
        self.get_interface('ICD').state_req()
      except:
        self.reinit_interface('ICD')
      return InterfaceState.on, mode, 0

  def turn_on(self):
    if self.state['value'] == InterfaceState.off:
      #run_async(self.run_commands, (self.wifi_turn_on_cmd,))
      gobject.idle_add(self.run_commands, [self.wifi_cmd['load%s' % self.state['mode']], self.wifi_cmd['up']])
    elif self.state['value'] == InterfaceState.connected:
      try:
        self.get_interface('ICD').disconnect_req(dbus.UInt32(0x8000))
      except:
        self.reinit_interface('ICD')
    return False

  def turn_off(self):
    self.__cancel_autooff()
    if self.state['value'] != InterfaceState.off:
      if self.state['value'] == InterfaceState.connected:
        try:
          self.get_interface('ICD').disconnect_req(dbus.UInt32(0x8000))
        except:
          self.reinit_interface('ICD')
      #run_async(self.run_commands, (self.wifi_turn_off_cmd,))
      gobject.idle_add(self.run_commands, self.wifi_turn_off_cmd)
    return False

  def run_commands(self, cmds):
    run_shell_command(' && '.join(cmds))
#    self.scan_request()
    self.switched()

  def show_connection_dialog(self):
    try:
      self.get_interface('ICD_UI').show_conn_dlg(False)
    except:
      self.reinit_interface('ICD_UI')
      self.get_interface('ICD_UI').show_conn_dlg(False)

  def scan_request(self):
    try:
      self.get_interface('ICD').scan_req(dbus.UInt32(0), dbus.Array([], signature='s'))
    except:
      self.reinit_interface('ICD')

  def get_strength(self):
    if self.state['value'] == InterfaceState.connected and self.get_config_value('show-signal-strength'):
      self.get_interface('ICD').statistics_req()
      self.__strength_timeout = gobject.timeout_add(5000, self.get_strength)
    return False

  def __cancel_get_strength(self):
    if self.__strength_timeout:
      gobject.source_remove(self.__strength_timeout)
      self.__strength_timeout = None

  def turn_mode(self, mode):
    mode = min(mode, len(self.state['modes']) - 1)
    if self.state['mode'] != mode:
      if self.state['value'] == InterfaceState.off:
        pass
      else:
        # down > unload > load%mode% > up
        gobject.idle_add(self.run_commands, [self.wifi_cmd['down'], self.wifi_cmd['load%s' % mode], self.wifi_cmd['up']])
    self.switched()

  def __cancel_autooff(self):
    if self.__autooff:
      gobject.source_remove(self.__autooff)
      self.__autooff = None

  def __get_driver(self):
    return get_shell_command_output(self.wifi_cmd['get_driver']).replace('\n', '')

  def __get_mode(self, driver):
    config = 0
    if self.has_be:
      if driver == '':
        try:
          config = self.state['modes'].index(self.get_config_value('mode-on-startup'))
        except ValueError:
          self.set_config_value('mode-on-startup', self.state['modes'][config], True)
      else:
        config = self.wifi_drivers.index(driver)
    return config
