#
# Copyright 2010 Ruediger Gad <r.c.g@gmx.de>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#

import cairo
from datetime import datetime
import dbus
import dbus.mainloop.glib
import gtk
import gobject
import hildon
import hildondesktop
import math
import os
import re
import sys


sys.path.append("/usr/lib/advanced-clock-plugin")
clocks_path = "/usr/lib/advanced-clock-plugin/clocks"
sys.path.append(clocks_path)

clock_files = os.listdir(clocks_path)
py_files = re.compile("\.py$", re.IGNORECASE)
clock_files = filter(py_files.search, clock_files)
file_to_module = lambda f: os.path.splitext(f)[0]
clock_names = map(file_to_module, clock_files)
clocks = map(__import__, clock_names)


class AdvancedClockPlugin(hildondesktop.StatusMenuItem):
    def __init__(self):
        hildondesktop.StatusMenuItem.__init__(self)

        self.active = True
        self.clock_style = 0

        self.time = None
        self.old_time = None

        button = hildon.Button(gtk.HILDON_SIZE_FINGER_HEIGHT, hildon.BUTTON_ARRANGEMENT_VERTICAL, "Advanced Clock")
        image = gtk.Image()
        image.set_from_file("/usr/share/icons/hicolor/64x64/apps/advanced-clock-plugin.png")
        button.set_image(image)
        button.set_alignment(0, 0.5, 0, 0)
        button.connect("clicked", self.settings_cb)
        self.add(button)
        button.show()

        self.drawing_area = gtk.DrawingArea()
      	self.drawing_area.connect("expose-event", self.expose_cb)
        self.set_status_area_widget(self.drawing_area)
      	self.drawing_area.show()

        self.clock = getattr(clocks[self.clock_style], clock_names[self.clock_style])(self.drawing_area)
        self.clock.resize()

        self.show()

        gobject.timeout_add(1000, self.update_cb)

        dbus.mainloop.glib.DBusGMainLoop(set_as_default = True)
        self.dbus = dbus.SystemBus()
        self.dbus.add_signal_receiver(self.dbus_signal_cb, "display_status_ind", "com.nokia.mce.signal", None, "/com/nokia/mce/signal")
        self.dbus.add_signal_receiver(self.dbus_alarm_cb, "queue_status_ind", "com.nokia.alarmd", None, "/com/nokia/alarmd")

    def dbus_alarm_cb(self, alarms, desktop, actdead, noboot):
        if desktop != actdead :
            self.clock.set_alarm_enabled(True)
            self.clock.set_alarm_time(datetime.fromtimestamp(actdead))
        else :
            self.clock.set_alarm_enabled(False)
        self.drawing_area.queue_draw()

    def dbus_signal_cb(self, msg):
        if msg == "off" :
            self.active = False
        elif msg == "on" :
            self.update_cb()
            self.active = True
            gobject.timeout_add(1000, self.update_cb)

    def expose_cb(self, widget, event):
        self.clock.draw()
        return False

    def settings_cb(self, widget, data=None):
        dialog = hildon.Dialog()
        dialog.set_title("Advanced Clock Plugin Settings")

        self.selector = hildon.TouchSelector(text = True)
        for c in clocks :
            self.selector.append_text(c.get_name())

        self.style_button = hildon.PickerButton(gtk.HILDON_SIZE_FINGER_HEIGHT, hildon.BUTTON_ARRANGEMENT_VERTICAL)
        self.style_button.set_title("Clock Style")
        self.style_button.set_selector(self.selector)
        self.style_button.set_active(self.clock_style)
        dialog.vbox.pack_start(self.style_button, True, True, 0)
        self.style_button.show()

        self.show_seconds_button = hildon.CheckButton(gtk.HILDON_SIZE_FINGER_HEIGHT)
        self.show_seconds_button.set_active(self.clock.get_show_seconds())
        self.show_seconds_button.set_label("Show seconds")
        dialog.vbox.pack_start(self.show_seconds_button, True, True, 0)
        self.show_seconds_button.show()

        self.twelve_hour_mode_button = hildon.CheckButton(gtk.HILDON_SIZE_FINGER_HEIGHT)
        self.twelve_hour_mode_button.set_active(self.clock.get_twelve_hour_mode())
        self.twelve_hour_mode_button.set_label("Twelve Hour Mode")
        dialog.vbox.pack_start(self.twelve_hour_mode_button, True, True, 0)
        self.twelve_hour_mode_button.show()

        self.fill_button = hildon.CheckButton(gtk.HILDON_SIZE_FINGER_HEIGHT)
        self.fill_button.set_active(self.clock.get_fill())
        self.fill_button.set_label("Fill")
        dialog.vbox.pack_start(self.fill_button, True, True, 0)
        self.fill_button.show()

        self.color_button = gtk.ColorButton()
        self.color_button.set_color(self.clock.get_gdk_color())
        self.color_button.set_label("Color")
        dialog.vbox.pack_start(self.color_button, True, True, 0)
        self.color_button.show()

        ok_button = hildon.Button(gtk.HILDON_SIZE_FINGER_HEIGHT, hildon.BUTTON_ARRANGEMENT_VERTICAL, "Ok")
        ok_button.connect("clicked", self.settings_ok_cb, dialog)
        dialog.action_area.pack_start(ok_button, True, True, 0)
        ok_button.show()

        dialog.show_all()

    def settings_ok_cb(self, widget, dialog):
        self.clock_style = self.style_button.get_active()

        old_clock = self.clock
        self.clock = getattr(clocks[self.clock_style], clock_names[self.clock_style])(self.drawing_area)

        self.clock.set_show_seconds(self.show_seconds_button.get_active())
        self.clock.set_twelve_hour_mode(self.twelve_hour_mode_button.get_active())
        self.clock.set_fill(self.fill_button.get_active())
        self.clock.set_gdk_color(self.color_button.get_color())

        self.clock.resize()

        self.clock.set_alarm_enabled(old_clock.get_alarm_enabled())
        self.clock.set_alarm_time(old_clock.get_alarm_time())

        self.drawing_area.queue_draw()
    
        dialog.destroy()

    def update_cb(self):
        if self.time != None:
            self.old_time = self.time

        self.time = datetime.today()

        if self.clock.get_show_seconds() or (self.old_time == None) or (self.old_time.minute != self.time.minute):
            self.drawing_area.queue_draw()

        return self.active


hd_plugin_type = AdvancedClockPlugin


