/*
 * Copyright (c) 2010 Kaj-Michael Lang
 * 
 * Addressbook menu extension to export current contact as a QR Code image.
 *
 * Based on example code provided by:
 * Copyright (C) 2010 Nokia Corporation
 *   @author Mathias Hasselmann <mathias.hasselmann@maemo.org>
 *   @author Marco Barisione <marco.barisione@collabora.co.uk>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include "config.h"

#include <glib.h>
#include <glib/gstdio.h>

#include <libosso.h>

#include <libosso-abook/osso-abook.h>

#include <hildon/hildon.h>
#include <hildon/hildon-fm.h>

#include <sharingdialog/sharing-dialog.h>

#include <cairo.h>

#include "qrcode_export.h"

#define BLOCK_SIZE (12)
#define PADDING (10)
#define MIN_SIZE (480)
#define MAX_SIZE (1024)

/* Would be great, but leads to segfaults for some odd reason */
/* #define USE_FULLSCREEN 0 */

typedef struct {
	OssoABookMenuExtension parent_instance;
} QRcodeMenuExtension;

typedef struct {
	OssoABookMenuExtensionClass parent_class;
} QRcodeMenuExtensionClass;

G_GNUC_INTERNAL GType
qrcode_menu_extension_get_type (void) G_GNUC_CONST;

/* Defines the new plugin object */
OSSO_ABOOK_DEFINE_MENU_PLUGIN(QRcodeMenuExtension, qrcode_menu_extension, OSSO_ABOOK_TYPE_MENU_EXTENSION,,);

typedef struct _PluginData PluginData;
struct _PluginData {
	/* UI */
	GtkWindow *window;
	GtkWidget *dr, *menu, *btn_save, *btn_load, *btn_share, *btn_fs;

	/* Current */
	OssoABookMenuExtension *ext;
	OssoABookContact *contact;
	gchar *basename;
	const gchar *name;
	cairo_surface_t *cs;
};

static PluginData plugindata;

static void
qrcode_menu_extension_init(QRcodeMenuExtension *ext)
{
plugindata.ext=NULL;
plugindata.contact=NULL;
plugindata.cs=NULL;
plugindata.basename=NULL;
}

static cairo_surface_t *
ac_contact_to_qrcode(OssoABookContact *contact)
{
gchar *vcard;
cairo_surface_t *qr_t;

if (!contact) {
	g_warning("Failed to get current contact");
	return FALSE;
}

vcard=osso_abook_contact_to_string(contact, EVC_FORMAT_VCARD_30, FALSE);
if (!vcard) {
	g_warning("Failed to get vcard for contact");
	return FALSE;
}

qr_t=gqre_char_to_qrcode_to_cairo(vcard, MIN_SIZE, MAX_SIZE, PADDING);

g_free(vcard);
return qr_t;
}

static void
ac_qrcode_save_cb(GtkWidget *widget, gpointer data)
{
PluginData *pd=(PluginData *)data;
GtkWidget *dialog;
gchar *file, *dir;

g_return_if_fail(pd);
g_return_if_fail(pd->ext);
g_return_if_fail(pd->contact);
g_return_if_fail(pd->cs);

file=g_strdup_printf("qrcode_%s.png", pd->basename ? pd->basename : "unknown_contact");
dir=g_build_filename(g_get_home_dir(), "MyDocs", NULL);

dialog=hildon_file_chooser_dialog_new(NULL, GTK_FILE_CHOOSER_ACTION_SAVE);
gtk_file_chooser_set_current_name(GTK_FILE_CHOOSER(dialog), file);
gtk_file_chooser_set_current_folder(GTK_FILE_CHOOSER(dialog), dir);

if (gtk_dialog_run(GTK_DIALOG(dialog))==GTK_RESPONSE_OK) {
	gchar *fname;
	
	fname=gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(dialog));

	if (cairo_surface_write_to_png(pd->cs, fname)==CAIRO_STATUS_SUCCESS)
		hildon_banner_show_informationf(NULL, NULL, "QR Code saved for contact:\n %s", pd->name);
	else
		hildon_banner_show_information(NULL, NULL, "QR Code save failed!");

	g_free(fname);
}
gtk_widget_destroy(dialog);
g_free(file);
g_free(dir);
}

static void
ac_qrcode_load_cb(GtkWidget *widget, PluginData *pd)
{
g_return_if_fail(pd);

/* XXX: Write this */
hildon_banner_show_information(NULL, NULL, "Import of QR Code not yet implemented");
}

static void
ac_qrcode_share_cb(GtkWidget *widget, PluginData *pd)
{
gchar *file, *fname;
osso_context_t *osso;

g_return_if_fail(pd);
g_return_if_fail(pd);
g_return_if_fail(pd->ext);
g_return_if_fail(pd->contact);
g_return_if_fail(pd->cs);

file=g_strdup_printf("qr_%d_code_%s.png", getpid(), pd->basename ? pd->basename : "unknown_contact");
fname=g_build_filename(g_get_tmp_dir(), file, NULL);

g_debug("Temp: [%s] in [%s]", file, fname);

if (cairo_surface_write_to_png(pd->cs, fname)!=CAIRO_STATUS_SUCCESS) {
	hildon_banner_show_informationf(NULL, NULL, "Failed to create temporary file for sharing.");
	goto out;
}

osso=osso_initialize("abook-qrcode", "1.0.0.", FALSE, NULL);
g_assert(osso);
sharing_dialog_with_file(osso, GTK_WINDOW(pd->window), fname);
osso_deinitialize(osso);
g_unlink(fname);

out:;
g_free(fname);
g_free(file);
}


static gboolean
ac_draw_expose_event(GtkWidget *widget, GdkEventExpose *event, gpointer data)
{
PluginData *pd=(PluginData *)data;
cairo_t *cr;

g_return_val_if_fail(pd, FALSE);

cr=gdk_cairo_create(widget->window);
gdk_cairo_rectangle(cr, &event->area);

cairo_set_source_rgb(cr, 1.0, 1.0, 1.0);
cairo_paint(cr);

if (pd->cs) {
	gdouble ratio, x;
	gint sh, sw;

	sw=cairo_image_surface_get_width(pd->cs);
	sh=cairo_image_surface_get_height(pd->cs);

	ratio=(gdouble)widget->allocation.height/(gdouble)sh;

	x=(widget->allocation.width/2.0)-((sw*ratio)/2.0);

	g_debug("SW: %d SH: %d", sw, sh);
	g_debug("R: %f X: %f", ratio, x);

	cairo_translate(cr, x, 0.0);

	cairo_scale(cr, ratio, ratio);
	cairo_set_source_surface(cr, pd->cs, 0.0, 0.0);
}

cairo_fill(cr);
cairo_destroy(cr);

return FALSE;
}

#ifdef USE_FULLSCREEN
static void
ac_fs_cb(GtkWidget *widget, PluginData *pd)
{
g_return_if_fail(pd);
g_return_if_fail(GTK_IS_WINDOW(pd->window));

gtk_window_fullscreen(pd->window);
}
#endif

#ifdef USE_FULLSCREEN
static void
ac_unfs_cb(GtkWidget *widget, PluginData *pd)
{
g_return_if_fail(pd);
g_return_if_fail(GTK_IS_WINDOW(pd->window));

gtk_window_unfullscreen(window);
}
#endif

/**
 * hildon_app_menu_button_new:
 *
 * Helper to do all the normal things a menu button should do.
 *
 * Returns: The new button
 */
static GtkWidget *
hildon_app_menu_button_new(HildonAppMenu *menu, gboolean show, const gchar *label, GCallback cb, gpointer data)
{
GtkWidget *button;

button=hildon_gtk_button_new(HILDON_SIZE_AUTO);
gtk_button_set_label(GTK_BUTTON(button), label);
if (cb)
	g_signal_connect_after(button, "clicked", cb, data);
if (show)
	gtk_widget_show(button);
hildon_app_menu_append(menu, GTK_BUTTON(button));

return button;
}

static void
ac_window_cb(GtkWidget *widget, gpointer data)
{
PluginData *pd=&plugindata;
HildonAppMenu *menu;

if (pd->cs) {
	cairo_surface_destroy(pd->cs);
	pd->cs=NULL;
}
if (pd->basename)
	g_free(pd->basename);

pd->ext=(OssoABookMenuExtension *)data;
pd->contact=osso_abook_menu_extension_get_contact(pd->ext);
if (pd->contact) {
	pd->name=osso_abook_contact_get_display_name(pd->contact);
	pd->cs=ac_contact_to_qrcode(pd->contact);
	if (!pd->cs) {
		hildon_banner_show_informationf(NULL, NULL, "Failed to create QRCode for:\n%s", pd->name);
		return;
	}
	pd->basename=osso_abook_contact_get_basename(pd->contact, TRUE);
} else {

}
pd->window=GTK_WINDOW(hildon_stackable_window_new());
if (pd->name)
	gtk_window_set_title(GTK_WINDOW(pd->window), pd->name);

pd->dr=gtk_drawing_area_new();

menu=HILDON_APP_MENU(hildon_app_menu_new());
pd->btn_save=hildon_app_menu_button_new(menu, pd->contact ? TRUE : FALSE, "Save", G_CALLBACK(ac_qrcode_save_cb), pd);
pd->btn_load=hildon_app_menu_button_new(menu, pd->contact ? FALSE : TRUE, "Load", G_CALLBACK(ac_qrcode_load_cb), pd);
pd->btn_share=hildon_app_menu_button_new(menu, pd->contact ? TRUE : FALSE, "Share", G_CALLBACK(ac_qrcode_share_cb), pd);

#ifdef USE_FULLSCREEN
pd->btn_fs=hildon_app_menu_button_new(menu, TRUE, "Fullscreen", G_CALLBACK(ac_fs_cb), pd);
gtk_widget_add_events(pd->dr, GDK_BUTTON_RELEASE_MASK);
g_signal_connect(pd->dr, "button_release_event", G_CALLBACK(ac_unfs_cb), pd);
#endif

gtk_container_add(GTK_CONTAINER(pd->window), GTK_WIDGET(pd->dr));
g_signal_connect(pd->dr, "expose-event", G_CALLBACK(ac_draw_expose_event), pd);


hildon_window_set_app_menu(HILDON_WINDOW(pd->window), menu);

gtk_widget_show_all(GTK_WIDGET(pd->window));
}

/* Entries to show in the menu, the values are:
 * - label: a label for the menu item;
 * - accel_mods:  modifier keys to use for the menu item;
 * - callback: a function to call when the menu item is activated;
 * - name: the name of the menu item.
 */
static OssoABookMenuEntry entries[] = {
	{ "QR Code contact card", 0, 0, G_CALLBACK(ac_window_cb), NULL },
};

static const OssoABookMenuEntry *
get_menu_entries(OssoABookMenuExtension *menu_extension)
{
return entries;
}

static int
get_n_menu_entries(OssoABookMenuExtension *menu_extension)
{
return G_N_ELEMENTS (entries);
}

static void
qrcode_menu_extension_class_init(QRcodeMenuExtensionClass *cls)
{
OssoABookMenuExtensionClass *menu_extension_class;

menu_extension_class = OSSO_ABOOK_MENU_EXTENSION_CLASS (cls);
menu_extension_class->get_n_menu_entries = get_n_menu_entries;
menu_extension_class->get_menu_entries = get_menu_entries;
menu_extension_class->name = OSSO_ABOOK_MENU_NAME_CONTACT_VIEW ";" OSSO_ABOOK_MENU_NAME_MECARD_VIEW;
}

static void
qrcode_menu_extension_class_finalize(QRcodeMenuExtensionClass *cls)
{
}

