/**
 * Copyright (C) 2009, 2010 Floriano Scioscia.
 *
 * This file is part of 100 Boxes.
 *
 * 100 Boxes is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * 100 Boxes is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with 100 Boxes. If not, see <http://www.gnu.org/licenses/>.
 */

#include <locale.h>
#include <libintl.h>
#include <glib.h>
#include <glib/gi18n.h>
#include <hildon/hildon.h>
#include <libosso.h>
#include <gconf/gconf-client.h>
#include <curl/curl.h>
#include <string.h>
#include <libxml/parser.h>
#include <libxml/tree.h>
#include "sound.h"
#include "100boxes.h"

static gboolean box_clicked(GtkWidget *b, GdkEventButton *event, gpointer data);
static void new_clicked(GtkWidget *b, gpointer data);
static void guide_clicked(GtkWidget *b, gpointer data);
static void set_box_image(struct game_data *d, gint box_index, gint img_index);
static gboolean restore_box_image(gpointer data);
static gboolean show_end_dialog(gpointer data);
static gchar* create_end_message(gint score);
static void submit_score(struct game_data *d, gchar *name);
static size_t store_post_reply(void *buffer, size_t size, size_t nmemb, void *userp);
static void show_high_score(struct game_data *d);
static void show_error_message(struct game_data *d, const gchar *msg);
static void save_state(gpointer data);
static void sound_button_toggled(HildonCheckButton *button, gpointer data);

int main (int argc, char **argv) {
	GtkWidget *win;
	GtkWidget *box_table;
	GtkWidget *box[100];
	GtkWidget *container;
	GtkWidget *panel;
	GtkWidget *new_button;
	GtkWidget *score_label;
	GtkWidget *sound_button;
	GtkWidget *guide_button;
	GError *error = NULL;
	gint i,j;
	GSList *seq;
	GSList *current;
	gint len;
	struct state s;
	struct game_data d;
	d.s = &s;
	d.base = box;
	gchar file[8];
	gint err;

	// Initialize i18n
	setlocale(LC_ALL, "");
	bindtextdomain(GETTEXT_PACKAGE, localedir);
	bind_textdomain_codeset(GETTEXT_PACKAGE, "UTF-8");
	textdomain(GETTEXT_PACKAGE);

	// Initialize sound
	sound_init(&argc, &argv);
	d.sounds = add_stock_sound(NULL, "START", SOUND_START, &err);
	d.sounds = add_stock_sound(d.sounds, "CLICK", SOUND_CLICK, &err);
	d.sounds = add_stock_sound(d.sounds, "WIN", SOUND_WIN, &err);
	d.sounds = add_stock_sound(d.sounds, "LOSE", SOUND_LOSE, &err);
	d.sounds = add_stock_sound(d.sounds, "ERROR", SOUND_ERROR, &err);

	// Initialize OSSO and Hildon
	d.osso_context = osso_initialize(OSSO_SERVICE, VERSION, TRUE, NULL);
	if (d.osso_context == NULL) {
        	return OSSO_ERROR;
	}
	hildon_gtk_init(&argc, &argv);
	osso_application_set_autosave_cb(d.osso_context, save_state, &d);
	//g_debug("System init: done");

	// Initializie gconf and load application state and preferences
	gconf_init(argc, argv, NULL);
	d.client = gconf_client_get_default();
	s.played = gconf_client_get_int(d.client, KEY_PLAYED_GAMES, &error);
	s.total = gconf_client_get_int(d.client, KEY_TOTAL_SCORE, &error);
	s.best = gconf_client_get_int(d.client, KEY_BEST_SCORE, &error);

	// Create the main window
	win = hildon_window_new();
	gtk_window_set_title(GTK_WINDOW(win), "100 Boxes");
	d.window = (GtkWindow*) win;

	// Create the box table
	box_table = gtk_table_new(10, 10, TRUE);
	gtk_widget_set_size_request(GTK_WIDGET(box_table), HEIGHT, -1);
	for (i=0; i<10; i++) {
		for (j=0; j<10; j++) {
			box[10*i+j] = gtk_event_box_new();
			gtk_table_attach_defaults(GTK_TABLE(box_table), GTK_WIDGET(box[10*i+j]), j, j+1, i, i+1);
			g_signal_connect(G_OBJECT(box[10*i+j]), "button_press_event", G_CALLBACK(box_clicked), (gpointer)&d);
		}
	}

	// Create other GUI elements
	container = gtk_hbox_new(FALSE, PADDING);
	panel = gtk_vbox_new(TRUE, PADDING);
	gtk_widget_set_size_request(GTK_WIDGET(panel), WIDTH-HEIGHT-PADDING, -1);

	new_button = gtk_button_new_with_label(_("New game"));
	hildon_helper_set_logical_font(GTK_WIDGET(new_button), "LargeSystemFont");
	g_signal_connect(GTK_WIDGET(new_button), "clicked", G_CALLBACK(new_clicked), (gpointer)&d);
	gtk_container_add(GTK_CONTAINER(panel), GTK_WIDGET(new_button));

	score_label = gtk_label_new(_("Score: 0"));
	hildon_helper_set_logical_font(GTK_WIDGET(score_label), "LargeSystemFont");
	gtk_container_add(GTK_CONTAINER(panel), GTK_WIDGET(score_label));
	d.score = score_label;

	sound_button = hildon_check_button_new(HILDON_SIZE_AUTO);
	gtk_button_set_label(GTK_BUTTON(sound_button), _("Sound"));
	s.sound_on = ! gconf_client_get_bool(d.client, KEY_MUTE, NULL);
	hildon_check_button_set_active(HILDON_CHECK_BUTTON(sound_button), s.sound_on);
	gtk_container_add(GTK_CONTAINER(panel), GTK_WIDGET(sound_button));
	g_signal_connect(sound_button, "toggled", G_CALLBACK(sound_button_toggled), (gpointer)&d);

	guide_button = gtk_button_new_with_label(_("Guide"));
	g_signal_connect(GTK_WIDGET(guide_button), "clicked", G_CALLBACK(guide_clicked), (gpointer)&d);
	gtk_container_add(GTK_CONTAINER(panel), GTK_WIDGET(guide_button));
	//g_debug("GUI creation: done");

	// Initialize game state
	seq = gconf_client_get_list(d.client, KEY_GAME_SEQ, GCONF_VALUE_INT, NULL);
	if (seq != NULL) {
		s.score = 0;
		s.last = -1;
		s.error = FALSE;
		s.active = gconf_client_get_bool(d.client, KEY_ACTIVE, &error);
		if (error != NULL) {
			g_error_free(error);
			error = NULL;
			new_game(&d);
		} else {
			play_stock_sound(d.sounds, "START");
			current = seq;
			i = 0;
			while (i<100 && current != NULL) {
				s.b[i] = (gint)GINT_TO_POINTER(current->data);
				set_box_image(&d, i, s.b[i]);
				if (s.b[i] <= 100 && s.b[i] > s.score) {
					s.score = s.b[i];
					s.last = i;
				}
			current = g_slist_next(current);
			i++;
			}
			g_slist_free(seq);
		}
		//g_debug("Game state restore: done");
	} else {
		new_game(&d);
	}

	// Assemble GUI elements
	gtk_container_add(GTK_CONTAINER(container), GTK_WIDGET(box_table));
	gtk_container_add(GTK_CONTAINER(container), GTK_WIDGET(panel));
	gtk_container_add(GTK_CONTAINER(win), GTK_WIDGET(container));
	g_signal_connect(win, "destroy", G_CALLBACK(gtk_main_quit), NULL);
	gtk_widget_show_all(win);

	// Main event loop
	gtk_main();

	osso_application_autosave_force(d.osso_context);

	// Deinitialize OSSO
	osso_deinitialize(d.osso_context);
	
	// Deinitialize sound
	clear_stock(d.sounds);
	return 0;
}

static void save_state(gpointer data) {
	struct game_data* d = (struct game_data *)data;
	GSList *seq, *last;
	gint i;

	for (i=0; i<100; i++) {
		seq = g_slist_prepend( seq, GINT_TO_POINTER(d->s->b[i]) );	
	}
	last = g_slist_nth(seq, 99);
	last->next = NULL;
	seq = g_slist_reverse(seq);
	gconf_client_set_list(d->client, KEY_GAME_SEQ, GCONF_VALUE_INT, seq, NULL);
	g_slist_free(seq);
	gconf_client_set_bool(d->client, KEY_ACTIVE, d->s->active, NULL);
	gconf_client_set_bool(d->client, KEY_MUTE, !(d->s->sound_on), NULL);
	gconf_client_set_int(d->client, KEY_PLAYED_GAMES, d->s->played, NULL);
	gconf_client_set_int(d->client, KEY_BEST_SCORE, d->s->best, NULL);
	gconf_client_set_int(d->client, KEY_TOTAL_SCORE, d->s->total, NULL);
	//g_debug("State saved");
}

static gboolean box_clicked(GtkWidget *b, GdkEventButton *event, gpointer data) {
	gint distx, disty, x, y, poss, n, r, c, old;
	gchar label[4], score_label[11];
	struct game_data *d = (struct game_data *)data;

	for (n = 0; n<100; n++) {
		if (b == d->base[n]) {
			break;
		}
	}
	r = n / 10;
	c = n % 10;
	if (d->s->active && !(d->s->error)) {
		d->s->last = n;
		if ( d->s->b[n] == CLICKABLE) {
			if (d->s->sound_on) play_stock_sound(d->sounds, "CLICK");
			d->s->score++;
			d->s->b[n] = d->s->score;
			sprintf(label, "%d", d->s->score);
			set_box_image(d, n, d->s->score);
			sprintf(score_label, _("Score: %d"), d->s->score);
			gtk_label_set_text(GTK_LABEL(d->score), score_label);
			poss = 0;
			for (x=0; x<10; x++) {
				for (y=0; y<10; y++) {
					if (d->s->b[10*x+y] < 1 || d->s->b[10*x+y] > 100) {
						distx = x-r > 0 ? x-r : r-x;
						disty = y-c > 0 ? y-c : c-y;
						if ( (distx==3 && disty==0) || (disty==3 && distx==0) || (distx==2 && disty==2) ) {
							poss++;
							old = d->s->b[10*x+y];
							d->s->b[10*x+y] = CLICKABLE;
							if (old != CLICKABLE) {
								set_box_image(d, 10*x+y, CLICKABLE);
							}
						} else {
							old = d->s->b[10*x+y];
							d->s->b[10*x+y] = UNUSABLE;
							if (old != UNUSABLE) {
								set_box_image(d, 10*x+y, UNUSABLE);
							}
						}
					}
				}
			}
			// poss = 0; // DEBUG
			if (poss == 0) {
				d->s->active = FALSE;
				if (d->s->score < 100) {
					if (d->s->sound_on) play_stock_sound(d->sounds, "LOSE");
				} else {
					if (d->s->sound_on) play_stock_sound(d->sounds, "WIN");
				}
				d->s->played++;
				d->s->total += d->s->score;
				if (d->s->score > d->s->best) {
					d->s->best = d->s->score;
				}
				g_timeout_add_seconds(2, show_end_dialog, (gpointer)d);
			}
			osso_application_userdata_changed(d->osso_context);
		} else {
			if (d->s->sound_on) play_stock_sound(d->sounds, "ERROR");
			set_box_image(d, n, ERROR);
			d->s->error = TRUE;
			g_timeout_add_seconds(1, restore_box_image, (gpointer)d);
		}
	}
	return TRUE;
}

static void new_clicked(GtkWidget *b, gpointer data) {
	struct game_data *d = (struct game_data *)data;
	osso_application_userdata_changed(d->osso_context);
	new_game(d);
}

static void sound_button_toggled (HildonCheckButton *button, gpointer data) {
	struct game_data *d = (struct game_data *)data;
	d->s->sound_on = hildon_check_button_get_active(button);
	osso_application_userdata_changed(d->osso_context);
}

static void guide_clicked(GtkWidget *b, gpointer data) {
	system(_("dbus-send --print-reply --dest=com.nokia.osso_browser /com/nokia/osso_browser/service com.nokia.osso_browser.open_new_window string:file:///usr/share/100boxes/index.html"));
}

void new_game(struct game_data *d) {
	gint i;
	d->s->active = TRUE;
	d->s->error = FALSE;
	d->s->score = 0;
	d->s->last = -1;
	//g_debug("Starting new game");
	if (d->s->sound_on) play_stock_sound(d->sounds, "START");
	for (i=0; i<100; i++) {
		d->s->b[i] = CLICKABLE;
		set_box_image(d, i, CLICKABLE);
	}
	gtk_label_set_text(GTK_LABEL(d->score), _("Score: 0"));
}

static void set_box_image(struct game_data *d, gint box_index, gint img_index) {
	gchar file[40];
	sprintf(file, "/usr/share/pixmaps/100boxes/%d.png", img_index);
	GtkWidget *img = gtk_image_new_from_file(file);
	GtkWidget *old = gtk_bin_get_child(GTK_BIN(d->base[box_index]));
	if (old != NULL) {
		gtk_widget_destroy(GTK_WIDGET(old));
	}
	gtk_container_add(GTK_CONTAINER(d->base[box_index]), GTK_WIDGET(img));
	gtk_widget_show(img);
}

static gboolean show_end_dialog(gpointer data) {
	gint ans, len;
	gchar *name, *s;
	GtkWidget *area;
	GtkWidget *comment;
	GtkWidget *icon;
	GtkWidget *caption;
	GtkWidget *played_games;
	GtkWidget *best_score;
	GtkWidget *avg_score;
	GtkWidget *submit;
	GtkWidget *label;
	GtkWidget *entry;
	GtkWidget *dialog;
	gchar line[80];
	gchar file[40];
	gchar *message;
	struct game_data *d = (struct game_data *)data;
	
	dialog = gtk_dialog_new_with_buttons (NULL, GTK_WINDOW(d->window),
                                                  GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
                                                  GTK_STOCK_OK,
                                                  GTK_RESPONSE_ACCEPT,
                                                  GTK_STOCK_CANCEL,
                                                  GTK_RESPONSE_REJECT,
                                                  NULL);

	if (d->s->score < 100) {
		gtk_window_set_title(GTK_WINDOW(dialog), _("You lose!"));
	} else {
		gtk_window_set_title(GTK_WINDOW(dialog), _("You win!"));
	}
	area = gtk_dialog_get_content_area(GTK_DIALOG(dialog));
	gtk_box_set_spacing(GTK_BOX(area), DIALOG_PADDING);

	comment = gtk_hbox_new(FALSE, 5);
	sprintf(file, "/usr/share/pixmaps/100boxes/%d.png", d->s->score);
	icon = gtk_image_new_from_file(file);
	gtk_container_add(GTK_CONTAINER(comment), GTK_WIDGET(icon));	
	message = create_end_message(d->s->score);
	caption = gtk_label_new("");
	gtk_label_set_markup(GTK_LABEL(caption), message);
	gtk_label_set_line_wrap(GTK_LABEL(caption), TRUE);
	gtk_container_add(GTK_CONTAINER(comment), GTK_WIDGET(caption));
	gtk_container_add(GTK_CONTAINER(area), GTK_WIDGET(comment));		

	best_score = gtk_label_new("");
	sprintf(line, _("Your best score: <b>%d</b>"), d->s->best);
	gtk_misc_set_alignment(GTK_MISC(best_score), 0.0, 0.5);
	gtk_label_set_markup(GTK_LABEL(best_score), line);
	gtk_container_add(GTK_CONTAINER(area), GTK_WIDGET(best_score));
	played_games = gtk_label_new("");
	sprintf(line, _("Games played: <b>%d</b>"), d->s->played);
	gtk_misc_set_alignment(GTK_MISC(played_games), 0.0, 0.5);
	gtk_label_set_markup(GTK_LABEL(played_games), line);
	gtk_container_add(GTK_CONTAINER(area), GTK_WIDGET(played_games));
	avg_score = gtk_label_new("");
	sprintf(line, _("Average score: <b>%d</b>"), (d->s->total / d->s->played));
	gtk_misc_set_alignment(GTK_MISC(avg_score), 0.0, 0.5);
	gtk_label_set_markup(GTK_LABEL(avg_score), line);
	gtk_container_add(GTK_CONTAINER(area), GTK_WIDGET(avg_score));

	submit = gtk_hbox_new(FALSE, 5);
	label = gtk_label_new(_("Enter your name to submit your score to worldwide high score table:"));
	gtk_label_set_line_wrap(GTK_LABEL(label), TRUE);
	gtk_container_add(GTK_CONTAINER(submit), GTK_WIDGET(label));
	entry = gtk_entry_new();
	gtk_entry_set_max_length(GTK_ENTRY(entry), 12); // Limit of online DB field
	gtk_container_add(GTK_CONTAINER(submit), GTK_WIDGET(entry));	
	gtk_container_add(GTK_CONTAINER(area), GTK_WIDGET(submit));		
	
	gtk_widget_show_all(GTK_WIDGET(dialog));
	ans = gtk_dialog_run(GTK_DIALOG(dialog));
	s = (gchar*) gtk_entry_get_text(GTK_ENTRY(entry));
	len = strlen(s);
	if (len > 0) {
		name = (gchar*) g_malloc(len+1);
		strcpy(name, s);
	}
	gtk_widget_destroy(dialog);
	g_free(message);
	switch (ans) {
		case GTK_RESPONSE_ACCEPT:
			if (len > 0) {
				submit_score(d, name);
			}
         		break;
		default:
			break;
	}
	return FALSE; // no repeat
}

static gchar* create_end_message(gint score) {
	gint i;
	gint l;
	gint n = 7;
	gint s[] = {40, 60, 75, 90, 97, 99, 100};
	gchar* m[] = { 	_("You can do better than this!"), 
			_("Train some more to improve."),
			_("Nice try, the basics are there."),
			_("You are approaching the solution."),
			_("Good, you are near the goal!"),
			_("Oh, you were so close!"),
			_("You are a 100 Boxes champion!") };
	gchar *msg;
	for (i=0; i<n; i++) {
		if (score <= s[i]) {
			l = strlen(m[i]);
			msg = (gchar*) g_malloc(l + 8);
			sprintf(msg, "<b>%s</b>", m[i]);
			break;
		}
	}
	return msg;
}

static gboolean restore_box_image(gpointer data) {
	struct game_data *d = (struct game_data *)data;
	if (d->s->score > 0) {	// Cannot restore if game not started
		gint n = d->s->last;
		set_box_image(d, n, d->s->b[n]);
		d->s->error = FALSE;
	}
	return FALSE; // no repeat		
}

static void submit_score(struct game_data *d, gchar *name) {
	CURLcode res;
	gchar post[MAX_QUERY_LENGTH+1];

	d->curl = curl_easy_init();
	d->post_reply = NULL;

	if (d->curl) {
		// POST data: show_scores callback function will receive the reply
		sprintf(post, "n=%s&s=%d&src=%s", name, d->s->score, SCORE_SRC);
		//g_debug("POST:\n%s", post);
		curl_easy_setopt(d->curl, CURLOPT_VERBOSE, 1);		// Set POST query
		curl_easy_setopt(d->curl, CURLOPT_POST, 1);		// Set POST method
		curl_easy_setopt(d->curl, CURLOPT_POSTFIELDS, post);	// Set POST data
		curl_easy_setopt(d->curl, CURLOPT_URL, POST_URL);	// Set POST URL
		curl_easy_setopt(d->curl, CURLOPT_WRITEFUNCTION, store_post_reply); // Set callback function
		curl_easy_setopt(d->curl, CURLOPT_WRITEDATA, d);	// Set data passed to callback
		res = curl_easy_perform(d->curl); 			// Do the POST
		if (res == 0) { // success
			//g_debug("REPLY:\n%s", d->post_reply);
			show_high_score(d);
		} else {
			show_error_message(d, _("Submission failed."));
		}
		curl_easy_cleanup(d->curl);
	} else {
		show_error_message(d, _("Connection failed."));
	}
}

static size_t store_post_reply(void *buffer, size_t size, size_t nmemb, void *userp) {
	struct game_data *d = (struct game_data *)userp;
	gchar *part = (gchar *)buffer;
	size_t len = size * nmemb;
	if (d->post_reply == NULL) {
		d->post_reply = (gchar*) g_malloc(len + 1);
		strcpy(d->post_reply, part);
	} else {
		d->post_reply = (gchar*) g_realloc(d->post_reply, strlen(d->post_reply) + len);
		strcat(d->post_reply, part);
	}
	return len;
}

static void show_error_message(struct game_data *d, const gchar *msg) {
	GtkWidget *note = hildon_note_new_information(d->window, msg);
	gtk_dialog_run(GTK_DIALOG(note));
	gtk_widget_destroy(note);			
}

static void show_high_score(struct game_data *d) {
	xmlDoc *doc;
	xmlNode *root, *pos, *tot, *top, *rec;
	xmlNode *week, *w_pos, *w_tot, *w_top, *w_rec;
	xmlNode *day, *d_pos, *d_tot, *d_top, *d_rec;
	gchar *line, *pos_v, *tot_v;
	gchar *n_v[MAX_HIGH_SCORES], *name_v[MAX_HIGH_SCORES], *score_v[MAX_HIGH_SCORES], *date_v[MAX_HIGH_SCORES];
	gchar *w_line, *w_pos_v, *w_tot_v;
	gchar *w_n_v[MAX_HIGH_SCORES], *w_name_v[MAX_HIGH_SCORES], *w_score_v[MAX_HIGH_SCORES], *w_date_v[MAX_HIGH_SCORES];
	gchar *d_line, *d_pos_v, *d_tot_v;
	gchar *d_n_v[MAX_HIGH_SCORES], *d_name_v[MAX_HIGH_SCORES], *d_score_v[MAX_HIGH_SCORES], *d_date_v[MAX_HIGH_SCORES];
	gint i, j;
	GtkWidget *tabs;
	GtkWidget *day_area;
	GtkWidget *week_area;
	GtkWidget *all_area;
	GtkWidget *text_l, *n_h, *name_h, *score_h, *date_h;
	GtkWidget *n_l[MAX_HIGH_SCORES], *name_l[MAX_HIGH_SCORES], *score_l[MAX_HIGH_SCORES], *date_l[MAX_HIGH_SCORES];
	GtkWidget *label;
	GtkWidget *table;
	GtkWidget *w_text_l, *w_n_h, *w_name_h, *w_score_h, *w_date_h;
	GtkWidget *w_n_l[MAX_HIGH_SCORES], *w_name_l[MAX_HIGH_SCORES], *w_score_l[MAX_HIGH_SCORES], *w_date_l[MAX_HIGH_SCORES];
	GtkWidget *w_label;
	GtkWidget *w_table;
	GtkWidget *d_text_l, *d_n_h, *d_name_h, *d_score_h;
	GtkWidget *d_n_l[MAX_HIGH_SCORES], *d_name_l[MAX_HIGH_SCORES], *d_score_l[MAX_HIGH_SCORES];
	GtkWidget *d_label;
	GtkWidget *d_table;
	GtkWidget *area;
	GtkWidget *dialog;

	doc = xmlReadDoc((xmlChar*)d->post_reply, NULL, NULL, 0);
	if (doc == NULL) {
		show_error_message(d, _("Bad response received."));
	} else {
		dialog = gtk_dialog_new_with_buttons (_("High scores"), GTK_WINDOW(d->window),
                                                  GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
                                                  GTK_STOCK_OK,
                                                  GTK_RESPONSE_CLOSE,
                                                  NULL);

		area = gtk_dialog_get_content_area(GTK_DIALOG(dialog));
		tabs = gtk_notebook_new();
 		gtk_container_add(GTK_CONTAINER(area), GTK_WIDGET(tabs));
		d_label = gtk_label_new( _("Today"));
		day_area = gtk_vbox_new(FALSE, 0);
		gtk_notebook_append_page(GTK_NOTEBOOK(tabs), GTK_WIDGET(day_area), d_label);
		w_label = gtk_label_new( _("7 Days"));
		week_area = gtk_vbox_new(FALSE, 0);
		gtk_notebook_append_page(GTK_NOTEBOOK(tabs), GTK_WIDGET(week_area), w_label);
		label = gtk_label_new( _("All time"));
		all_area = gtk_vbox_new(FALSE, 0);
		gtk_notebook_append_page(GTK_NOTEBOOK(tabs), GTK_WIDGET(all_area), label);

		//g_debug("Getting root XML element");
		root = xmlDocGetRootElement(doc);
		pos = root->children;
		pos_v = (gchar*) pos->children->content;
		tot = pos->next;
		tot_v = (gchar*) tot->children->content;
		top = tot->next;

		line = (gchar*) g_malloc(100);
		sprintf(line, _("<big>You got place no. <b>%s</b> of <b>%s</b>!</big>"), pos_v, tot_v);
		text_l = gtk_label_new("");
 		gtk_container_add(GTK_CONTAINER(all_area), GTK_WIDGET(text_l));
		gtk_label_set_markup(GTK_LABEL(text_l), line);

		// Extract top scores
		for (rec=top->children, i=0; rec!=NULL && i<MAX_HIGH_SCORES; rec = rec->next, i++) {
			n_v[i] = (gchar*) g_malloc(3);
			sprintf(n_v[i], "%d", (i+1));
			name_v[i] = xmlGetProp(rec, "name");
			score_v[i] = xmlGetProp(rec, "score");
			date_v[i] = xmlGetProp(rec, "date");
		}

		// Create table and header row
		table = gtk_table_new(4, i+1, FALSE);
 		gtk_container_add(GTK_CONTAINER(all_area), GTK_WIDGET(table));
		n_h = gtk_label_new("");
 		gtk_table_attach_defaults(GTK_TABLE(table), GTK_WIDGET(n_h), 0, 1, 0, 1);
		gtk_label_set_markup(GTK_LABEL(n_h), _("<b>Place</b>"));
		name_h = gtk_label_new("");
 		gtk_table_attach_defaults(GTK_TABLE(table), GTK_WIDGET(name_h), 1, 2, 0, 1);
		gtk_label_set_markup(GTK_LABEL(name_h), _("<b>Name</b>"));
		score_h = gtk_label_new("");
 		gtk_table_attach_defaults(GTK_TABLE(table), GTK_WIDGET(score_h), 2, 3, 0, 1);
		gtk_label_set_markup(GTK_LABEL(score_h), _("<b>Score</b>"));
		date_h = gtk_label_new("");
 		gtk_table_attach_defaults(GTK_TABLE(table), GTK_WIDGET(date_h), 3, 4, 0, 1);
		gtk_label_set_markup(GTK_LABEL(date_h), _("<b>Date</b>"));

		// Create data rows
		for (j=0; j<i; j++) {
			n_l[j] = gtk_label_new(n_v[j]);
			gtk_table_attach_defaults(GTK_TABLE(table), GTK_WIDGET(n_l[j]), 0, 1, j+1, j+2);
			name_l[j] = gtk_label_new(name_v[j]);
			gtk_table_attach_defaults(GTK_TABLE(table), GTK_WIDGET(name_l[j]), 1, 2, j+1, j+2);
			score_l[j] = gtk_label_new(score_v[j]);
			gtk_table_attach_defaults(GTK_TABLE(table), GTK_WIDGET(score_l[j]), 2, 3, j+1, j+2);
			date_l[j] = gtk_label_new(date_v[j]);
			gtk_table_attach_defaults(GTK_TABLE(table), GTK_WIDGET(date_l[j]), 3, 4, j+1, j+2);
		}

		//g_debug("Getting week XML element");
		week = top->next;
		w_pos = week->children;
		w_pos_v = (gchar*) w_pos->children->content;
		w_tot = w_pos->next;
		w_tot_v = (gchar*) w_tot->children->content;
		w_top = w_tot->next;

		w_line = (gchar*) g_malloc(100);
		sprintf(w_line, _("<big>You got place no. <b>%s</b> of <b>%s</b>!</big>"), w_pos_v, w_tot_v);
		w_text_l = gtk_label_new("");
 		gtk_container_add(GTK_CONTAINER(week_area), GTK_WIDGET(w_text_l));
		gtk_label_set_markup(GTK_LABEL(w_text_l), w_line);

		// Extract top scores
		for (w_rec=w_top->children, i=0; w_rec!=NULL && i<MAX_HIGH_SCORES; w_rec = w_rec->next, i++) {
			w_n_v[i] = (gchar*) g_malloc(3);
			sprintf(w_n_v[i], "%d", (i+1));
			w_name_v[i] = xmlGetProp(w_rec, "name");
			w_score_v[i] = xmlGetProp(w_rec, "score");
			w_date_v[i] = xmlGetProp(w_rec, "date");
		}

		// Create table and header row
		w_table = gtk_table_new(4, i+1, FALSE);
 		gtk_container_add(GTK_CONTAINER(week_area), GTK_WIDGET(w_table));
		w_n_h = gtk_label_new("");
 		gtk_table_attach_defaults(GTK_TABLE(w_table), GTK_WIDGET(w_n_h), 0, 1, 0, 1);
		gtk_label_set_markup(GTK_LABEL(w_n_h), _("<b>Place</b>"));
		w_name_h = gtk_label_new("");
 		gtk_table_attach_defaults(GTK_TABLE(w_table), GTK_WIDGET(w_name_h), 1, 2, 0, 1);
		gtk_label_set_markup(GTK_LABEL(w_name_h), _("<b>Name</b>"));
		w_score_h = gtk_label_new("");
 		gtk_table_attach_defaults(GTK_TABLE(w_table), GTK_WIDGET(w_score_h), 2, 3, 0, 1);
		gtk_label_set_markup(GTK_LABEL(w_score_h), _("<b>Score</b>"));
		w_date_h = gtk_label_new("");
 		gtk_table_attach_defaults(GTK_TABLE(w_table), GTK_WIDGET(w_date_h), 3, 4, 0, 1);
		gtk_label_set_markup(GTK_LABEL(w_date_h), _("<b>Date</b>"));

		// Create data rows
		for (j=0; j<i; j++) {
			w_n_l[j] = gtk_label_new(w_n_v[j]);
			gtk_table_attach_defaults(GTK_TABLE(w_table), GTK_WIDGET(w_n_l[j]), 0, 1, j+1, j+2);
			w_name_l[j] = gtk_label_new(w_name_v[j]);
			gtk_table_attach_defaults(GTK_TABLE(w_table), GTK_WIDGET(w_name_l[j]), 1, 2, j+1, j+2);
			w_score_l[j] = gtk_label_new(w_score_v[j]);
			gtk_table_attach_defaults(GTK_TABLE(w_table), GTK_WIDGET(w_score_l[j]), 2, 3, j+1, j+2);
			w_date_l[j] = gtk_label_new(w_date_v[j]);
			gtk_table_attach_defaults(GTK_TABLE(w_table), GTK_WIDGET(w_date_l[j]), 3, 4, j+1, j+2);
		}

		//g_debug("Getting day XML element");
		day = week->next;
		d_pos = day->children;
		d_pos_v = (gchar*) d_pos->children->content;
		d_tot = d_pos->next;
		d_tot_v = (gchar*) d_tot->children->content;
		d_top = d_tot->next;

		d_line = (gchar*) g_malloc(100);
		sprintf(d_line, _("<big>You got place no. <b>%s</b> of <b>%s</b>!</big>"), d_pos_v, d_tot_v);
		d_text_l = gtk_label_new("");
 		gtk_container_add(GTK_CONTAINER(day_area), GTK_WIDGET(d_text_l));
		gtk_label_set_markup(GTK_LABEL(d_text_l), d_line);

		// Extract top scores
		for (d_rec=d_top->children, i=0; d_rec!=NULL && i<MAX_HIGH_SCORES; d_rec = d_rec->next, i++) {
			d_n_v[i] = (gchar*) g_malloc(3);
			sprintf(d_n_v[i], "%d", (i+1));
			d_name_v[i] = xmlGetProp(d_rec, "name");
			d_score_v[i] = xmlGetProp(d_rec, "score");
		}

		// Create table and header row
		d_table = gtk_table_new(3, i+1, FALSE);
 		gtk_container_add(GTK_CONTAINER(day_area), GTK_WIDGET(d_table));
		d_n_h = gtk_label_new("");
 		gtk_table_attach_defaults(GTK_TABLE(d_table), GTK_WIDGET(d_n_h), 0, 1, 0, 1);
		gtk_label_set_markup(GTK_LABEL(d_n_h), _("<b>Place</b>"));
		d_name_h = gtk_label_new("");
 		gtk_table_attach_defaults(GTK_TABLE(d_table), GTK_WIDGET(d_name_h), 1, 2, 0, 1);
		gtk_label_set_markup(GTK_LABEL(d_name_h), _("<b>Name</b>"));
		d_score_h = gtk_label_new("");
 		gtk_table_attach_defaults(GTK_TABLE(d_table), GTK_WIDGET(d_score_h), 2, 3, 0, 1);
		gtk_label_set_markup(GTK_LABEL(d_score_h), _("<b>Score</b>"));

		// Create data rows
		for (j=0; j<i; j++) {
			d_n_l[j] = gtk_label_new(d_n_v[j]);
			gtk_table_attach_defaults(GTK_TABLE(d_table), GTK_WIDGET(d_n_l[j]), 0, 1, j+1, j+2);
			d_name_l[j] = gtk_label_new(d_name_v[j]);
			gtk_table_attach_defaults(GTK_TABLE(d_table), GTK_WIDGET(d_name_l[j]), 1, 2, j+1, j+2);
			d_score_l[j] = gtk_label_new(d_score_v[j]);
			gtk_table_attach_defaults(GTK_TABLE(d_table), GTK_WIDGET(d_score_l[j]), 2, 3, j+1, j+2);
		}

		gtk_widget_show_all(GTK_WIDGET(dialog));
		gtk_dialog_run(GTK_DIALOG(dialog));

		// Cleanup
		gtk_widget_destroy(GTK_WIDGET(dialog));
		g_free(line);
		for (j=0; j<i; j++) {
			g_free(n_v[j]);
		}
		xmlFreeDoc(doc);
	}	
	xmlCleanupParser();
	g_free(d->post_reply);
	d->post_reply = NULL;
}

