/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

#include <QWidget>
#include <QMessageBox>
#include <QDir>


#include "widgetmanagerconstants.h"
#include "WidgetInfo.h"
#include "W3CXmlParser/w3cxmlplugin.h"
#include "private/WidgetLinkResolver.h"
#include "private/WidgetUtilsLogs.h"

#include "WebAppRegistry.h"
#include "WidgetProperties.h"
#include "WidgetManager.h"

#include "private/webappmanager_p.h"

#define PDATA ((WidgetManager_p*) m_data)

#include "signatureparser.h"
#include "SuperWidget.h"

#if !defined(Q_OS_MAEMO6) && !defined(Q_OS_MAEMO5)
#include "WgzWidget.h"
#endif

#include "WgtWidget.h"
#include "AsyncInstall.h"
#include "widgetpromptstrings.h"
#include <QDesktopServices>
#include <QDebug>
#include <QThreadPool>

#ifdef __SYMBIAN32__
#include <QTimer>
#include <defaultcaps.hrh>
#include <e32std.h>
#include <e32base.h>
#include <SWInstApi.h>
static const char KSIGNATUREID[]            = "58:BF:92:E1:B3:C6:15:69:8C:6C:5E:F5:9C:25:C4:2D:DF:1F:D8:C5";
static const char KINDEXFILE[]              = "installedWidgets.txt";
static const char KWIDGETINSTALLLAUNCHER[]  = "WidgetInstallLauncher-";
static const char KEXESUFFIX[]              = ".exe";
_LIT8(KSISMIMETYPE, "x-epoc/x-sisx-app");
#endif  // __SYMBIAN32__
#define NO_NAME "NoName"


/*!
 \class WidgetManager
 \brief  Install and uninstall web applications. Public class for installing widgets for 3rd party.
*/

WidgetManager_p::WidgetManager_p(QWidget *parent):
m_parent(parent)
{
}

WidgetManager_p::~WidgetManager_p()
{
}

/*!
Widget manager deletes all temporary created files on destructor

\a widget pointer for displaying confirmation
*/
Q_DECL_EXPORT WidgetManager::WidgetManager(QWidget *parent, bool isChromeless)
: QObject(parent)
, m_silent(false)
, m_disableUnsignWidgetSignCheck(false)
{
      (void)isChromeless; // Argument required for compatability with old branch
    m_data = (void *) new WidgetManager_p(parent);
    m_widgetType = WidgetTypeUnknown;
    m_contentType = CONTENT_TYPE_UNKNOWN;
    m_installingWidget = 0;
}


Q_DECL_EXPORT WidgetManager::~WidgetManager()
{
    delete (WidgetManager_p*)m_data;

    // Delete all of the cached widgets.
    QHash<QString, SuperWidget*>::iterator i = m_widgetCache.begin();
    while (i != m_widgetCache.end()) {
        SuperWidget* widget = i.value();
        if (widget) {
            delete widget;
        }
        ++i;
    }
    m_widgetCache.clear();

    delete m_installingWidget;
    m_installingWidget = 0;
}

/*!
 already installed web applications executable file/path

 \a pkgPath package file path
 \return returns launcher file path (html file path for widgets) or link for web applications
*/
Q_DECL_EXPORT QString WidgetManager::launcherPath(const QString &pkgPath)
{
    if (!checkCallerPermissions()) {
        return "";
    }

    LOG("BEGIN WidgetManager::launcherPath(" << pkgPath << ")");
    // Get the widget
    SuperWidget* widget = getWidgetFromCache(pkgPath);
    if (!widget) {
        // We don't have a widget for this path yet, create one
        widget = createWidget(pkgPath);
    }
    if (widget) {
        LOG("END WidgetManager::launcherpath returns" << widget->launcherPath(pkgPath));
        return widget->launcherPath(pkgPath);
    }

    LOG("END WidgetManager::launcherPath returns ' '");
    return "";
}

/*!
 \a pkgPath package file path
 \return returns true if pkgPath is already installed
*/
Q_DECL_EXPORT bool WidgetManager::isRegistered(const QString &pkgPath)
{
    if (!checkCallerPermissions()) {
        return false;
    }

    WidgetProperties *widgetProp = getWidgetProperties(pkgPath);
    if (widgetProp) {
        LOG("WidgetManager::isRegistered : pkgPath: " << pkgPath << WebAppRegistry::instance()->isRegistered (widgetProp->id()));
        return WebAppRegistry::instance()->isRegistered (widgetProp->id());
    }

    return false;
}

/*!
 install's the package file passed. Registers the application in registry if package file is valid
o
 \a pkgPath package file path
 \return returns true if installation is successful
*/
Q_DECL_EXPORT WidgetInstallError WidgetManager::install(const QString &pkgPath, QString &widgetId , bool silent,  bool update, const QString &rootDirectory)
{
    if (!checkCallerPermissions()) {
        return WidgetInstallPermissionFailed;
    }

    LOG("WidgetManger::install silent =" << silent << "update =" << update);

    if (!validateDriveLetter(rootDirectory)) {
        return WidgetDriveLetterValidationFailed;
    }
    if (update) {
        return install(pkgPath, widgetId, "", m_rootDirectory, true, true);
    }
    return install(pkgPath, widgetId, "", m_rootDirectory, silent, false);
}


/*!
 uninstall's the web app. Removes the entry from registry.

 \a pkgPath package file path
 \return returns true if uninstallation is successful
*/
Q_DECL_EXPORT WidgetUninstallError WidgetManager::uninstall(const QString &uniqueIdentifier, bool silent)
{
    if (!checkCallerPermissions()) {
        return WidgetUninstallPermissionFailed;
    }

    WebAppInfo widget;
    if (WebAppRegistry::instance()->isRegistered(uniqueIdentifier, widget)) {
        enum QMessageBox::StandardButton answer;
        QString widgetName = ((widget.appTitle() == NO_NAME) ? uniqueIdentifier : widget.appTitle());
        if (!silent)
            answer = QMessageBox::question(PDATA->m_parent,
                                       TR_WM_WIDGET_UNINSTALL,
                                       TR_WM_WIDGET_REMOVE_QUERY
                                       + widgetName
                                       + "?",
                                       QMessageBox::Yes | QMessageBox::No);
        if (silent || QMessageBox::Yes == answer) {
            emit installProgress(0);
            LOG("WidgetManager::uninstall : " << uniqueIdentifier);
            WidgetUninstallError err = SuperWidget::uninstall(uniqueIdentifier, true);
#ifdef __SYMBIAN32__
            if (err == WidgetUninstallSuccess) {
                bool removeFromList(silent);
                err = uninstallSisx(uniqueIdentifier, removeFromList);
            }
#endif  // __SYMBIAN32__
            return err;
        }
        else
            if (!silent && QMessageBox::No == answer)
                return WidgetUninstallCancelled;
    }
    return WidgetUninstallFailed;
}

void WidgetManager::tryContinueInstallation() {
    if (m_installingWidget) {
        m_installingWidget->tryContinueInstallation();
    }
}

Q_DECL_EXPORT WidgetProperties* WidgetManager::getProperties(const QString &pkgPath)
{
    if (!checkCallerPermissions()) {
        return 0;
    }

    return getWidgetProperties(pkgPath);
}

//
// function to get widget properties from path. Will create a new widget object
// if one doesn't exist for the path
// parameters:
//     path       path to widget
// return
//     WidgetProperties* pointer to widget properteis
//
WidgetProperties* WidgetManager::getWidgetProperties(const QString &pkgPath)
{
    LOG("BEGIN WidgetManager::getWidgetProperties pkgpath =" << pkgPath);

    if (pkgPath.isEmpty()) {
        LOG("END WidgetManager::getWidgetProperties pkgPath is empty");
        return 0;
    }

    SuperWidget* cachedWidget = getWidgetFromCache(pkgPath);
    if (cachedWidget) {
       LOG( "WidgetManager::getWidgetProperties use cached widget to getProperties");
        WidgetProperties *props = cachedWidget->getProperties(pkgPath);
        if (props == 0) {
            LOG("END WidgetManager::getWidgetProperties: cached widget has properties=0");
            return 0;
        }
        // Check if current widget path and properties path match. Use native
        // separators and remove trailing separator, because properties path
        // was saved using native separators.
        QString widgetPath = QDir::convertSeparators(pkgPath);
        if (widgetPath.endsWith(QDir::separator())) {
            widgetPath = widgetPath.remove(widgetPath.length()-1, widgetPath.length()-1);
        }
        if (props->installPath().compare(widgetPath) == 0) {
            LOG("END WidgetManager::getWidgetProperties cached widget properties are valid");
            return props;
        }
    }

    // We couldn't find a cached widget, so create a new one
    SuperWidget* newWidget = createWidget(pkgPath);
    if (newWidget) {
        WidgetProperties *props = newWidget->getProperties(pkgPath);
        LOG("END WidgetManager::getWidgetProperties using new widget to get properties");
        return props;
    }

    LOG("END WidgetManager::getWidgetProperties FAILED to get widget properties");
    return 0;

}

//
// function to create a widget, uses the path (unique folder name) as key to
// cache widget in hash
// parameters:
//     path       path to widget
// return
//     SuperWidget* pointer to new widget
//
SuperWidget* WidgetManager::createWidget(const QString& pkgPath)
{
    // Generate the install path, use native separators and no ending separator
    QString installPath;
#ifdef Q_OS_SYMBIAN
    QString path = pkgPath[0];
    installPath = QDir::toNativeSeparators(pkgPath).section(QDir::separator(),0,3);
    installPath[0] = path[0];
#elif (Q_OS_MAEMO5 || Q_OS_MAEMO6)
    installPath = WIDGET_INSTALL_PATH;
#else
    QString dataLocation =  QDesktopServices::storageLocation(QDesktopServices::DataLocation);
    installPath = QDir::toNativeSeparators(dataLocation);
    installPath = installPath + QDir::separator() + WIDGET_FOLDER;
#endif


    // Create a new widget
    SuperWidget* newWidget;
    switch (SuperWidget::getWidgetType(pkgPath))
        {
#if !defined(Q_OS_MAEMO6) && !defined(Q_OS_MAEMO5)
        case WidgetTypeWgz:
            newWidget = new WgzWidget(installPath);
            break;
#endif
        case WidgetTypeW3c:
            newWidget = new WgtWidget(installPath);
            break;
        default:
            LOG("END WidgetManager::createWidget - Invalid Widget type");
            newWidget = 0;
            break;
    }

    if (newWidget) {
        setWidgetIntoCache(pkgPath, newWidget);
        return newWidget;
    }

    return 0;
}

SuperWidget* WidgetManager::getWidgetFromCache(QString pkgPath)
{
    LOG("BEGIN WidgetManager::getWidgetFromCache");
    QString uniqueName;
    if (SuperWidget::getUniqueNameFromPath(pkgPath, uniqueName)) {
        LOG("WidgetManager::getWidgetFromCache uniqueName =" << uniqueName);

        if (m_widgetCache.contains(uniqueName)) {
            SuperWidget* widget = m_widgetCache.value(uniqueName);
            LOG("END WidgetManager::getWidgetFromCache found a widget =" << widget);
            return widget;
        }
    }

    LOG("END WidgetManager::getWidgetFromCache can't find widget");
    return 0;
}

void WidgetManager::setWidgetIntoCache(QString pkgPath, SuperWidget* widget)
{
    LOG("BEGIN WidgetManager::setWidgetIntoCache");
    QString uniqueName;
    if (SuperWidget::getUniqueNameFromPath(pkgPath, uniqueName)) {
        LOG("WidgetManager::setWidgetIntoCache uid =" << uniqueName);

        if (!m_widgetCache.contains(uniqueName)) {
            m_widgetCache.insert(uniqueName, widget);
            LOG("WidgetManager::setWidgetIntoCache saved widget =" << widget);
        }
    }
    LOG("END WidgetManager::setWidgetIntoCache");
}

bool WidgetManager::isValidWidgetType(const QString& widgetPath, const QString& contentType)
{
    if (!checkCallerPermissions()) {
        return false;
    }

    return SuperWidget::isValidWidgetType(widgetPath,contentType);
}

void WidgetManager::setContentType(const QString& contentType)
{
    if (!checkCallerPermissions()) {
        return;
    }
    m_contentType = contentType;
}

bool WidgetManager::isValidWidgetType(const QString& widgetPath)
{
    if (!checkCallerPermissions()) {
        return false;
    }

    QString contentType;
    if (getContentType(contentType)) {
        return isValidWidgetType(widgetPath,contentType);
    }
    else {
        return false;
    }
}

bool WidgetManager::getContentType(QString& contentType)
{
    if (!checkCallerPermissions()) {
        return false;
    }

    contentType = m_contentType;
    return true;
}

void WidgetManager::asyncInstall(const QString &pkgPath, bool silent, const QString& rootDirectory, bool update)
{
    if (!checkCallerPermissions()) {
        return;
    }

    LOG(">> WidgetManager::asyncInstall");
    if (validateDriveLetter(rootDirectory)){
       if ( !SuperWidget::isValidWidgetType(pkgPath, QString()) )
           {
           LOG("Invalid Widget type");
           return;
           }
       AsyncInstall *asyncInstall = new AsyncInstall( pkgPath, silent, update, this, m_rootDirectory );
       QThreadPool::globalInstance()->start( asyncInstall );
       LOG("<< WidgetManager::asyncInstall");
    }
}

#ifdef __SYMBIAN32__

/*!
 install's the SISX package file passed.
 Registers the application in registry if package file is valid
 \a pkgPath package file path
 \return UID of installed widget
*/
Q_DECL_EXPORT QString WidgetManager::installSisx(const QString &pkgPath, const QString& rootDirectory)
{
    QString id;
    if (!checkCallerPermissions()) {
        return id;
    }

    LOG(">> WidgetManger::installSisx(" << pkgPath << ")");

    if (!validateDriveLetter(rootDirectory)) {
        return id;
    }

    // First check that only trusted clients can call this method
    RProcess process;
    if (process.VendorId() == TVendorId(VID_DEFAULT) &&
        process.HasCapability(ECapabilityAllFiles)) {

        // Install the widget
        QString widgetId;
        if (WidgetInstallSuccess == install(QDir::toNativeSeparators(pkgPath), widgetId, "",m_rootDirectory, true, false, KSIGNATUREID)) {

            // Get the ID
            if (m_installingWidget) {
                WidgetProperties *props(m_installingWidget->getProperties());
                if (props) {
                    id = props->id();
                    // the id and widgetId shall be the same at this point, need to verify
                    // Save the ID and the current process SID
                    // for use during uninstallation
                    TSecureId sid(process.SecureId());
                    TUint32 sidInt(sid.iId);
                    QHash<QString,uint> widgetList = readSisxWidgetList();
                    widgetList.insert(id, (uint)sidInt);
                    writeSisxWidgetList(widgetList);
                }
            }
        }
    }
    LOG("<< WidgetManger::installSisx(" << pkgPath << ")");
    return id;
}

/*!
 Uninstall's the SISX package file passed.
 \a uniqueIdentifier ID of the widget to uninstall
 \a aRemoveFromlist Flag to remove widget from installed list
 \return returns An error code from WidgetUninstallError enum
*/
WidgetUninstallError WidgetManager::uninstallSisx(const QString &uniqueIdentifier,
                                                  bool aRemoveFromList)
{
    LOG(">> WidgetManager::uninstallSisx(" << uniqueIdentifier << ", " << aRemoveFromList << ")");
    WidgetUninstallError result(WidgetUninstallSuccess);

    // First check if widget was installed as a SISX wrapper
    QHash<QString,uint> widgetList = readSisxWidgetList();
    uint sid = widgetList.value(uniqueIdentifier, -1);
    if (-1 != sid) {
        SwiUI::RSWInstLauncher swinst;
        if (KErrNone == swinst.Connect()) {
            SwiUI::TUninstallOptionsPckg options;
            TInt err(swinst.SilentUninstall(TUid::Uid(sid), options, KSISMIMETYPE));
            if (SwiUI::KSWInstErrBusy == err) {
                result = WidgetUninstallSuccessButIncomplete;
            } else if (KErrNone != err) {
                result = WidgetUninstallFailed;
            }
            swinst.Close();
            if (aRemoveFromList) {
                widgetList.remove(uniqueIdentifier);
                writeSisxWidgetList(widgetList);
            }
        }
    }
    return result;
}

/*!
 Reads the SISX widget list from file
*/
QHash<QString, uint> WidgetManager::readSisxWidgetList()
{
    QHash<QString, uint> widgetList;
    QFile file(sisxWidgetFile());
    if (file.open(QIODevice::ReadOnly)) {
        QDataStream dataStream(&file);
        dataStream >> widgetList; //codescanner::leave
        file.close();
    }
    return widgetList;
}

/*!
 Writes the SISX widget list to file
*/
void WidgetManager::writeSisxWidgetList(QHash<QString, uint> aWidgetList)
{
    QString fileName(sisxWidgetFile());
    if (!QFile::exists(fileName)) {
        QDir dir = QFileInfo(fileName).dir();
        dir.mkpath(dir.path());
    }
    QFile file(fileName);
    if (file.open(QIODevice::WriteOnly)) {
        QDataStream dataStream(&file);
        dataStream << aWidgetList; //codescanner::leave
        file.close();
    }
}

/*!
 Gets the name of the SISX widget list file
*/
QString WidgetManager::sisxWidgetFile()
{
    QString indexFileName(DEFAULT_ROOT_DIRECTORY);
    indexFileName += QDir::separator();
    indexFileName += KINDEXFILE;
    indexFileName = QDir::toNativeSeparators(indexFileName);
    return indexFileName;
}

#endif  // __SYMBIAN32__


// function to install widgets
// parameters:
//     widgetPath          path to widget bundle
//     widgetContentType   content type of widget (optional)
//     silent              true for silent install (optional)
//                         false by default
// return
//     bool                true if successful
//
// creates appropriate widget and installs it
WidgetInstallError WidgetManager::install(const QString& widgetPath,
                                          QString &widgetId,
                                          const QString& widgetContentType,
                                          const QString& rootDirectory,
                                          const bool silent,
                                          const bool update,
                                          const QString &sigId)
{
    Q_UNUSED(sigId)
    m_silent = silent;

    WidgetType widgetType = SuperWidget::getWidgetType(widgetPath,widgetContentType);

    if (WidgetTypeUnknown == widgetType) {
        LOG("WidgetManager::install - Invalid Widget type");
        handleInstallError(WidgetTypeValidationFailed);
        emit installationError(WidgetTypeValidationFailed);
        return WidgetTypeValidationFailed;
    }

    if (m_installingWidget) {
        delete m_installingWidget;
        m_installingWidget = 0;
    }

    // Create a new installing widget
    QString rootDir = rootDirectory;

    switch (widgetType) {
#if !defined(Q_OS_MAEMO6) && !defined(Q_OS_MAEMO5)
    case WidgetTypeWgz:
        m_installingWidget = new WgzWidget(rootDir);
        break;
#endif
    case WidgetTypeW3c:
        m_installingWidget = new WgtWidget(rootDir);
        m_installingWidget->disableBackupRestoreValidation(m_disableUnsignWidgetSignCheck);
        break;
    default:
        LOG("WidgetManager::install Invalid Widget type");
        return WidgetTypeValidationFailed;
    }

    LOG("WidgetManager::install widget installed path: " << m_installingWidget->widgetInstallPath());

    if (!m_installingWidget) {
        return WidgetSystemError;
    }

    // FIXME : Silent / not silent should not visible. SIGNALS or interfacing for this class
    if (!silent) {
        if (QMessageBox::No == QMessageBox::question ( PDATA->m_parent, TR_WM_WIDGET_INSTALL,
                                                   TR_WM_WIDGET_INSTALL_QUERY + QFileInfo(widgetPath).fileName() + " ?",
                                                   QMessageBox::Yes | QMessageBox::No ))
        {
            QMessageBox::information(PDATA->m_parent, TR_WM_WIDGET_INSTALL, TR_WM_WIDGET_INSTALL_CANCELLED);
            return WidgetUserConfirmFailed;
        }
    }

    connect(m_installingWidget, SIGNAL(installProgress(int)), this, SIGNAL(installProgress(int)));

    // Connect all signals from a widget to dialgos handles of this class.
    connect(m_installingWidget, SIGNAL(installationError(WidgetInstallError)),
            this, SLOT(handleInstallError(WidgetInstallError)));
    connect(m_installingWidget, SIGNAL(aboutToReplaceExistingWidget(QString)),
            this, SLOT(handleWidgetReplacement(QString)), Qt::DirectConnection);
    connect(m_installingWidget, SIGNAL(aboutToInstallWidgetWithFeatures(QList<QString>)),
            this, SLOT(handleFeatureInstallation(QList<QString>)),Qt::DirectConnection);
    connect(m_installingWidget, SIGNAL(aboutToInstallUntrustedWidget(WidgetInstallError)),
            this, SLOT(handleUntrustedWidgetInstallation(WidgetInstallError)), Qt::DirectConnection);
    connect(m_installingWidget, SIGNAL(installationSucceed()),
            this, SLOT(handleInstallationSuccess()));

    // Widget installation signals forwarded out of this class.
    // This provides one way to rid of qmessageboxes from this class and applications that are
    // using this would be in charge of showing correct ui dialogs.
    // However, silentInstall is only thingie which could really utilize this.
    connect(m_installingWidget, SIGNAL(installationError(WidgetInstallError)),
            this, SIGNAL(installationError(WidgetInstallError)));
    connect(m_installingWidget, SIGNAL(aboutToReplaceExistingWidget(QString)),
            this, SIGNAL(aboutToReplaceExistingWidget(QString)), Qt::DirectConnection);
    connect(m_installingWidget, SIGNAL(aboutToInstallUntrustedWidget(WidgetInstallError)),
            this, SIGNAL(aboutToInstallUntrustedWidget(WidgetInstallError)), Qt::DirectConnection);
    connect(m_installingWidget, SIGNAL(aboutToInstallWidgetWithFeatures(QList<QString>)),
            this, SIGNAL(aboutToInstallWidgetWithFeatures(QList<QString>)),Qt::DirectConnection);
    connect(m_installingWidget, SIGNAL(installationSucceed()),
            this, SIGNAL(installationSucceed()));

    enum WidgetInstallError status=WidgetInstallSuccess;
    m_installingWidget->setWidgetBundlePath(widgetPath);
    if (update) {
        status = m_installingWidget->install(true);
        disconnect(m_installingWidget, SIGNAL(installProgress(int)), this, SIGNAL(installProgress(int)));
        if (status == WidgetInstallSuccess)
               widgetId = m_installingWidget->getProperties()->id();

        return status;
    }

    if (!sigId.isNull() && !sigId.isEmpty() && widgetType == WidgetTypeW3c) {
        WgtWidget *widget = static_cast<WgtWidget*>(m_installingWidget);
        status = widget->install(sigId);
    } else {
        status = m_installingWidget->install();
    }

    if (status == WidgetInstallSuccess)
        widgetId = m_installingWidget->getProperties()->id();

    disconnect(m_installingWidget, SIGNAL(installProgress(int)), this, SIGNAL(installProgress(int)));
    return status;
}

void WidgetManager::handleInstallError(WidgetInstallError error) {
    // FIXME : We must localize message and use platform specific dialogs / notifications.
    if (!m_silent && ((error != WidgetInstallSuccess) && (error!= WidgetUserConfirmFailed))) {
        QMessageBox::information(PDATA->m_parent, TR_WM_WIDGET_INSTALL_ERROR, TR_WM_WIDGET_INSTALL_FAILED);
    }
}

void WidgetManager::handleInstallationSuccess() {
// FIXME: for now we don't want to display any success messages,
// once the package should have being handled by application manager
// and we have no means to know when it has finished.
#ifndef Q_OS_MAEMO5
    QString message = TR_WM_WIDGET_INSTALL_SUCCESS;
    if (!m_silent)QMessageBox::information ( PDATA->m_parent, TR_WM_WIDGET_INSTALL, message);
#endif
}

void WidgetManager::handleWidgetReplacement(QString title) {
    if (!m_silent) {
        QString msg = TR_WM_WIDGET_REPLACE + " \" " + title + " \"?";
        if (QMessageBox::No ==
               QMessageBox::question ( PDATA->m_parent,
                       TR_WM_WIDGET_INSTALL, msg, QMessageBox::Yes |
                       QMessageBox::No ) )
        {
            return;
        }
    }
    m_installingWidget->tryContinueInstallation();
}

void WidgetManager::handleFeatureInstallation(QList<QString> capList)
{
    if (!m_silent) {
        QString message;
        for (int i=0; i<capList.count(); i++) {
            message += "\n";
            message += QCoreApplication::translate("WidgetManager", QT_TRANSLATE_NOOP("WidgetManager",capList[i].toUtf8().constData()));
        }

        if (QMessageBox::Cancel == QMessageBox::information( PDATA->m_parent,
                                                         TR_WM_SECURITY_WARNING,
                                                         TR_WM_WIDGET_INSTALL_ALLOW +
                                                         m_installingWidget->getProperties()->title() +
                                                         TR_WM_WIDGET_INSTALL_TO_ACCESS +
                                                         message,
                                                         QMessageBox::Ok | QMessageBox::Cancel ))
        {
            return;
        }
    }
    m_installingWidget->tryContinueInstallation();
}

void WidgetManager::handleUntrustedWidgetInstallation(WidgetInstallError errorCode) {

    if (m_silent) {
        // Same code as use to be in the WgtWidget installation.
        // In silent mode must enable installation of untrusted widget.
        // This is just about the error value.
        if (errorCode == WidgetFindSignatureFailed) {
            m_installingWidget->tryContinueInstallation();
        }
        return;
    }

    QString message = "";
    switch (errorCode) {
        case WidgetFindSignatureFailed:
        case WidgetSignatureParsingFailed:
        case WidgetSignatureOrSignedInfoMissing:
        case WidgetSignatureRefExistFailed:
        case WidgetCertValidationFailed:
        case WidgetSignatureValidationFailed:
        case WidgetSignatureRefValidationFailed:
            message = TR_WM_WIDGET_SIGNATURE_INVALID;
            break;
        default:
            message = TR_WM_WIDGET_UNKNOWN_ERROR;
            break;
    }

    if (QMessageBox::Yes == QMessageBox::question( PDATA->m_parent,
            TR_WM_WIDGET_INSTALL, message + TR_WM_WIDGET_INSTALL_CONT_QUERY,
            QMessageBox::Yes | QMessageBox::No ) ) {
            m_installingWidget->tryContinueInstallation();
    }
}

/*
 * Returns the installing widget properties
 */
WidgetProperties* WidgetManager::getProperties()
{
    if (!checkCallerPermissions()) {
        return 0;
    }

    if (m_installingWidget)
    {
        return m_installingWidget->getProperties();
    }
    return 0;
}

/*
 * Constructs the widgets installation path
 */
QString WidgetManager::getWidgetInstallPath(const QString& rootDirectory)
{
  Q_UNUSED(rootDirectory)
  QString installPath("");

#ifdef Q_OS_SYMBIAN
  installPath = QDir::toNativeSeparators(rootDirectory);
  if (installPath.startsWith("Z"))
    installPath.replace(0, 1, "C");
  QString privatePath = installPath[0]+":"+QDir::separator()+"private";
  if (! createDir(privatePath)) {
      LOG("SuperWidget::getWidgetInstallPath - Could not create private dir");
      return QString();
      }
  if (! createDir(installPath)) {
    LOG("SuperWidget::getWidgetInstallPath - Could not create widget UI private dir");
    return QString();
    }
  installPath = installPath + QDir::separator() + WIDGET_FOLDER;
#elif (Q_OS_MAEMO5 || Q_OS_MAEMO6)
  installPath = WIDGET_INSTALL_PATH;
#else
  QString dataLocation =  QDesktopServices::storageLocation(QDesktopServices::DataLocation);
  installPath = QDir::toNativeSeparators(dataLocation);
  installPath = installPath + QDir::separator() + WIDGET_FOLDER;
#endif
  if (! createDir(installPath)) {
      LOG("SuperWidget::getWidgetInstallPath - Could not create widget dir");
      return QString();
      }
  return installPath;
}

/*
 * Directory Creation for Widget Installation
 */
bool WidgetManager::createDir(const QString& path)
{
  QDir dir(path);
  if (!dir.exists()){
    if (!dir.mkpath(dir.absolutePath())) {
      LOG("WidgetManager::createDir could not create dir");
      return false;
      }
    }
    return true;
}
/*
 * Checks for the rootDirectory value
 * If Null or Empty then it assigns the DEFAULT_ROOT_DIRECTORY
 */
QString WidgetManager::setDefaultRootDirectory(const QString& rootDirectory)
{
    if (rootDirectory.isEmpty() || rootDirectory.isNull()){
        QString defaultRootDir = DEFAULT_ROOT_DIRECTORY;
        return(defaultRootDir);
    }
    QString& defaultRootDir = const_cast<QString&> (rootDirectory);
    return(defaultRootDir);
}

#ifdef __SYMBIAN32__
//
// function to construct S60 widget Install path
// parameters:
//     QChar      drive letter
// return
//     QString    Install Path
//
QString WidgetManager::setS60RootDirectory(const QChar& rootDirectory)
{
    if (rootDirectory.isNull()){
        QString defaultRootDir = DEFAULT_ROOT_DIRECTORY;
        return(defaultRootDir);
    }
    else {
    QString rootDrive = DEFAULT_ROOT_DIRECTORY;
    if (!(rootDrive.startsWith("\\") || rootDrive.startsWith("/")))
        rootDrive.remove(0, 2);
   QString rootPath = "";
   rootPath.append(rootDirectory);
   rootPath = rootPath+":"+rootDrive;
   rootPath = QDir::toNativeSeparators(rootPath);
   return(rootPath);
   }
}
#endif  // __SYMBIAN32__

//
// function to validate the input rootDirectory string
// parameters:
//     QString    rootDirectory String
// return
//     bool    True if input is valid on Symbian
//
bool WidgetManager::validateDriveLetter(const QString& rootDirectory)
{
#ifdef __SYMBIAN32__
    if ( rootDirectory.length() >= 1 ) {
        QChar driveLetter = rootDirectory[0];
        if (!driveLetter.isLetter())
            return false;
        m_rootDirectory = setS60RootDirectory(driveLetter);
        m_rootDirectory = getWidgetInstallPath(m_rootDirectory);
        if (m_rootDirectory.isEmpty() || m_rootDirectory.isNull())
            return false;
    } else {
            m_rootDirectory = setS60RootDirectory();
            m_rootDirectory = getWidgetInstallPath(m_rootDirectory);
            if (m_rootDirectory.isEmpty() || m_rootDirectory.isNull())
                 return false;
    }
#else
    m_rootDirectory = setDefaultRootDirectory(rootDirectory);
    m_rootDirectory = getWidgetInstallPath(m_rootDirectory);
    if (m_rootDirectory.isEmpty() || m_rootDirectory.isNull())
        return false;
#endif  // __SYMBIAN32__

    return true;
}

bool WidgetManager::backupRestoreInstall(const QString &pkgPath, bool disableUnsignWidgetSignCheck)
{
#ifdef Q_OS_SYMBIAN
        // for security reasons, any process other than backup-restore should not be
        // allowed to use this API.
        RProcess myUID;
        TUidType uidType(myUID.Type());
        TUint32 uid = uidType[2].iUid;
        LOG("WidgetManager::backupRestoreInstall()");
        if (uid == BACKUP_RESTORE_UID) {
            //set BUR process
            m_disableUnsignWidgetSignCheck = disableUnsignWidgetSignCheck;
            QString widgetId;
            return install(pkgPath, widgetId, true, false, "");
        } else {
            LOG("WidgetManager::backupRestoreInstall() - Aborting install. This is not Backup-Restore process");
            return false;
        }
#else
      return false;
#endif
}

bool WidgetManager::checkCallerPermissions()
{
#ifdef Q_OS_SYMBIAN
    RProcess process;
    static _LIT_SECURITY_POLICY_V0(myVidPolicy, VID_DEFAULT);
    TInt error = myVidPolicy().CheckPolicy(process);
    if (error  == EFalse) {
        LOG("WidgetManager::checkCallerPermissions() return false");
        return false;
    }
#endif
    return true;
}
