/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

#include <QDebug>
#include <QSqlDatabase>
#include <QSqlQuery>
#include <QSqlError>
#include <QFileInfo>
#include <QDir>
#include <QTextStream>
#include "WidgetLinkResolver.h"
#include "WidgetProperties.h"
#include "AttributeMap.h"
#include "PropertyNames.h"
#include "WebAppRegistry.h"
#include "widgetmanagerconstants.h"
#include "private/WidgetUtilsLogs.h"
#include "w3csettingskeys.h"
#include "wrtsettings.h"

#ifdef Q_OS_SYMBIAN
#include "WidgetRegistrationS60.h"
#endif

const QString KEY_VALUE_SEPERATOR = ",";
const QString KEY_VALUE_PAIR_SEPERATOR = ";";

WebAppRegistry* WebAppRegistry::m_webappRegistry = 0;

void encodeDelimiters(QString& inputString)
{
    // encode all "%" chars with "%10"
    inputString.replace(QString("%"), QString("%10"));

    // encode all key value delimiter
    inputString.replace(KEY_VALUE_SEPERATOR, QString("%20"));

    // encode all key value pair delimiter
    inputString.replace(KEY_VALUE_PAIR_SEPERATOR, QString("%30"));
}

void decodeDelimiters(QString& inputString)
{
    // decode all "%20" to ":"
    inputString.replace(QString("%30"), KEY_VALUE_PAIR_SEPERATOR);

   // decode all "%30" to KEY_VALUE_PAIR_SEPERATOR
    inputString.replace(QString("%20"), KEY_VALUE_SEPERATOR);

   // decode all "%10" to "%"
    inputString.replace(QString("%10"), QString("%"));
}

/*!
  Create an empty database if not exists already. This should be the first function called by client before using any API's in this class
*/
void create()
{
    LOG("WebAppRegistry::create");
    QString dataLocation = WidgetLinkResolver::dataStorageLocation();
    QString dbLocation = QDir::toNativeSeparators(dataLocation
                                                  + "/webapp_registry.db");

    QSqlDatabase db = QSqlDatabase::database("webAppConnection", FALSE);

    if (!db.isValid()) {
        db = QSqlDatabase::addDatabase("QSQLITE", "webAppConnection");
        db.setHostName("WebAppRegistry");
        db.setDatabaseName(dbLocation);
    }
    if (!db.isOpen() && !db.open()) {
        LOG(dbLocation);
        LOG("add db Error" << db.lastError().text());
        return;
    }

    QFileInfo dbFile(dbLocation);
    if (dbFile.exists() && dbFile.size() == 0) {
        //create table as it does not exists
        db.exec("CREATE TABLE dbversion (version INTEGER)");
        QString insertStatement;
        QTextStream(&insertStatement) << "INSERT INTO dbversion (version) VALUES (" << CurrentDatabaseVersion << ")";
        db.exec(insertStatement);

        // NOTE for webapp_registry columns:  Per SQLite FAQ:  "SQLite does not enforce the length of a VARCHAR.
        //       You can declare a VARCHAR(10) and SQLite will be happy to let you put 500 characters in it. And
        //       it will keep all 500 characters intact - it never truncates."

        db.exec ("CREATE TABLE webapp_registry ( id VARCHAR(50) PRIMARY KEY,appTitle VARCHAR(100) NOT NULL,appPath VARCHAR(100) NOT NULL,iconPath VARCHAR,attributeList VARCHAR,secureSession VARCHAR,widgetType VARCHAR(20),uid INTEGER,startPath VARCHAR,dataPath VARCHAR(100),capabilityCheck BOOLEAN DEFAULT False,certificateAki VARCHAR(100), nativeId VARCHAR)");
        LOG("db error: " << db.lastError().text());
    }
}

/*!
  upgradeDB is called by checkDB when the version of the database is lower than the expected
  (current) version.  This function takes necessary steps to upgrade the database to the current
  version, maintaining data.

  currently a noop, as there's only ever been this version of the db
 */
void upgradeDB()
{

}

/*!
  Checks the database for existence and version.  If it doesn't exist, calls create() to create it.
  If the version is prior to the expected current version, calls upgrade() to update the database.
 */
void checkDB()
{
    int dbversion = 0;

    // Check for the DBs existence and establish connection
    create();

    QSqlDatabase db = QSqlDatabase::database("webAppConnection");

    if (db.isOpen()) {
        // Check to see if there's a version table
        // ... if not, it's a version 0 database and we need to start upgrading
        QStringList tables = db.tables();
        if (tables.contains("dbversion")) {
            QSqlQuery query = db.exec("SELECT version FROM dbversion");
            if (query.next()) {
                dbversion = query.value(0).toInt();
            }
        } else {
            dbversion = 0;
        }
    }

    if (dbversion < CurrentDatabaseVersion) {
        upgradeDB();
    }
}

/*!
 * \class WebAppRegistry
 *
 * \brief WebAppRegistry class maintains the secured database for registered web applications.
 *  This is a single-ton class and WebAppRegistry::create() has to be called before using any API's in this class.
 *
 * This class is the public API for Web Application Framework. (TBD)
 */

/*!
 Static function returns singleton object of WebAppRegistry

 \return singleton WebAppRegistry pointer object
*/
WebAppRegistry* WebAppRegistry::instance()
{
    // FIXME refcount, add release() and delete on 0
    if (!m_webappRegistry) {
        // create a self pointer
        m_webappRegistry = new WebAppRegistry();
    }
    return m_webappRegistry;
}

WebAppRegistry::WebAppRegistry()
{
#ifdef Q_OS_SYMBIAN
    m_s60Instance = new WidgetRegistrationS60();
#endif

    // Ensure that the DB exists and is up-to-date
    checkDB();
}

WebAppRegistry::~WebAppRegistry()
{
    // FIXME when refcount implemented, set to NULL on 0
    m_webappRegistry = NULL;
}

int WebAppInfo::widgetType() const
{
    int type(WidgetUnknown);

    if (m_widgetType == WIDGET_PACKAGE_FORMAT_WGT) {
        type = WidgetW3C;
    }
    else if (m_widgetType == WIDGET_PACKAGE_FORMAT_WGZ) {
        type = WidgetNokia;
    }
    else if (m_widgetType == WIDGET_PACKAGE_FORMAT_SHARED_LIBRARY) {
        type = WidgetSharedLibraryWidget;
    }
    else if (m_widgetType == WIDGET_PACKAGE_FORMAT_JIL) {
        type = WidgetJIL;
    }
    return type;
}

/*!
 \return list of all matching W3CElements, note that reciever responsible to free the W3CElements
*/
QList<W3CElement*> WebAppInfo::getElement(const QString &aKey, const W3CElement *parent, const QString &aLang)
{
     QString rawKey, lang;

     if (aLang.isEmpty()){
         WRT::WrtSettings* settings;
         settings = WRT::WrtSettings::createWrtSettings();
         QString userAgentLang = settings->valueAsString("UserAgentLanguage");

         if (userAgentLang.isEmpty())  {
             QLocale language;
             userAgentLang = language.name();
             userAgentLang.replace(QString("_"),QString("-"));
         }
         lang = userAgentLang.toLower();
     }
     else
         lang = aLang;

     if ((aKey == W3CSettingsKey::WIDGET_NAME)
        || (aKey == W3CSettingsKey::WIDGET_DESCRIPTION)
        || (aKey == W3CSettingsKey::WIDGET_LICENSE)
        || (aKey == W3CSettingsKey::WIDGET_ICON))
          rawKey = lang + QString("/") + aKey;
     else
          rawKey = aKey;

     QStringList rawKeys = m_data.keys();
     QStringList keys = rawKeys.filter(rawKey, Qt::CaseInsensitive);

     qDebug() << "Keys found -> " << keys;

     QList<W3CElement*> w3cElementsList;
     QString keyValue;
     int count = 1;

     if (parent) {
         QString parentNum = parent->elementName();
         parentNum = parentNum.right(parentNum.length() - parentNum.lastIndexOf("/") - 1);

         QString temp1, temp2;
         temp1 = rawKey.left(rawKey.lastIndexOf("/") + 1);
         temp2 = rawKey.right(rawKey.length() - rawKey.lastIndexOf("/"));

         keyValue = temp1 + parentNum + temp2;
     }
     else
         keyValue = rawKey;

     QString newKey;

     if (m_data.contains(keyValue)) // Exact key present. Only one element.
        newKey = keyValue;
     else
        newKey = keyValue + "/" + QString::number(count);

     while (m_data.contains(newKey)) {
         W3CElement *element = new W3CElement(newKey);
         element->setText(m_data.value(newKey).toString());
         QStringList attrs = keys.filter(newKey + ":");
         foreach(QString key, attrs) {
            QString attrName = key.right(key.length() - key.lastIndexOf(":") -1);
            element->addAttribute(attrName, m_data.value(key).toString());
         }

         w3cElementsList.append(element);

         count++;
         newKey = keyValue + "/" + QString::number(count);
     }

     return w3cElementsList;
}

bool deleteAppRow(const QString& appID)
{
    bool ret(false);

    QSqlDatabase db = QSqlDatabase::database("webAppConnection");

    if (db.isOpen()) {
        QSqlQuery query(db);
        query.prepare("DELETE FROM webapp_registry WHERE id = :id");
        query.bindValue(0, QVariant (appID));
        query.exec();
        LOG(query.lastError().text());
        ret = true;
    }

    return ret;
}


/*!
 register application, takes all required inputs to register.

 \a appID a unique ID for the webApp,(mandatory)
 \a title a display name for webApp(mandatory)
 \a appPath path to the executable. URL if its a link(mandatory)
 \a iconPath iconPath for the webApp. If not specified default icon will be used(not mandatory)
  \a attributes is a map having <key, value> pairs for attributes(not mandatory)
 \return true if registration is successful
*/
bool WebAppRegistry::registerApp(const QString& appId,
                                       const QString& appTitle,
                                       const QString& appPath,
                                       const QString& iconPath,
                                       const AttributeMap& attributes,
                                       const QString& widgetType,
                                       unsigned long size,
                                       const QString& startPath)
{
    LOG ("WebAppRegistry::registerApp -" << appId << appTitle << appPath);
    if (appId.isEmpty() || appTitle.isEmpty() || appPath.isEmpty())
        return false;

    bool ret(false);

    QSqlDatabase db = QSqlDatabase::database("webAppConnection");
#ifdef Q_OS_SYMBIAN
    QString dataPath = appPath.section(QDir::separator(),0,2) + QDir::separator()+ QString(DATA_FOLDER)+ QDir::separator() + appPath.section(QDir::separator(),4)+ QDir::separator();
#else
    QString dataPath = DATA_PATH ;
#endif
    if (db.isOpen()) {
        QSqlQuery query(db);
        query.prepare("INSERT INTO webapp_registry (id, appTitle, appPath, iconPath, attributeList, secureSession, widgetType, startPath, dataPath) "
            "VALUES (:id, :appTitle, :appPath, :iconPath, :attributeList, :secureSession, :widgetType, :startPath, :dataPath)");
        query.bindValue(":id", QVariant (appId));
        query.bindValue(":appTitle", QVariant (appTitle));

        QString appPathNative = QDir::toNativeSeparators(appPath);
        // FIXME enforce canonicalization in caller
        // must end in QDir::separator()
        if (QDir::separator() != appPathNative.at(appPathNative.count()-1))
        {
            appPathNative.append(QDir::separator());
        }

        query.bindValue(":appPath", QVariant (appPathNative));
        query.bindValue(":iconPath", QVariant (iconPath));

        QString attributesList;
        AttributeMap::const_iterator i;
        QString key;
        QString value;
        for (i=attributes.begin(); i != attributes.end(); i++) {
            key = i.key();
            value = i.value().toString();

            encodeDelimiters(key);
            encodeDelimiters(value);
            attributesList = attributesList + key +
                KEY_VALUE_SEPERATOR + value + KEY_VALUE_PAIR_SEPERATOR;
        }
        query.bindValue(":attributeList", QVariant (attributesList));

        query.bindValue(":widgetType", QVariant (widgetType));
        query.bindValue(":startPath", QVariant (startPath));
        query.bindValue(":dataPath", QVariant (dataPath));

        if (query.exec())
            ret = true;
    }

#ifdef Q_OS_SYMBIAN
    if (ret == true){
        LOG("WebAppRegistry::registerApp() - Registering using WidgetRegistrationS60 API to store data in S60 Registry");

        ret = m_s60Instance->registerApp(appId,
                                         appTitle,
                                         appPath,
                                         iconPath,
                                         attributes,
                                         widgetType,
                                         size,
                                         startPath);

                if (!ret) {
                    // S60 registration failed, remove SQL DB entry for registration
                    deleteAppRow(appId);
                }
    }
#endif

    return ret;
}

/*!
 unregister webapp

 \a appID web app ID to unregister
 \return true if unregister is successful.
*/
bool WebAppRegistry::unregister (const QString& appID, bool update)
{
    LOG ("WebAppRegistry::unregister -" << appID);

    bool ret;
#ifdef Q_OS_SYMBIAN

    LOG("WebAppRegistry::unregister() - UnRegistering using WidgetRegistrationS60 API to update data in S60 Registry");
    ret = m_s60Instance->unregister(appID, update);
    if (ret==true)
#endif
    ret = deleteAppRow(appID);

    return ret;
}

/*!
 \a appID id to check if already registered
 \return true if webapp is registered
*/
bool WebAppRegistry::isRegistered (const QString& appID) const
{
    LOG ("WebAppRegistry::isRegistered -" << appID);

    QSqlDatabase db = QSqlDatabase::database("webAppConnection");

    if (db.isOpen()) {
         QSqlQuery query(db);
         query.prepare("SELECT id FROM webapp_registry WHERE id = :id");
         query.bindValue(0, QVariant (appID));
         query.exec();
         LOG(query.lastError().text());
         while (query.next()) {
             if (!query.value(0).toString().isEmpty()) {
                 LOG ("WebAppRegistry::isRegisters returns true");
                 return true;
             }
         }
    }

    LOG ("WebAppRegistry::isRegistered returns false");
    return false;
}

/*!
 This is an overloaded member function, provided for convenience

 \a appID web app ID to unregister
 \a WebAppInfo to fill related information for the webApp
 \return true if webApp is registered.
*/
bool WebAppRegistry::isRegistered(const QString& appID, WebAppInfo& info)
{
    LOG ("WebAppRegistry::registered : " << appID);

    QSqlDatabase db = QSqlDatabase::database("webAppConnection");

    if (db.isOpen()) {
        QSqlQuery query(db);
        query.prepare("SELECT id,appTitle,appPath,iconPath,attributeList,secureSession,widgetType,uid,startPath,dataPath,capabilityCheck,certificateAki,nativeId FROM webapp_registry WHERE id = \'" + appID + "\'");
        query.exec();
        LOG(query.lastError().text());
        if (query.next()) {
           info.m_appId = query.value(0).toString();
           info.m_appTitle = query.value(1).toString();
           info.m_appPath = query.value(2).toString();
           info.m_iconPath= query.value(3).toString();
           QString attributes = query.value(4).toString();
           AttributeMap attributeList;
           QString key, value;
           QStringList pairs = attributes.split(KEY_VALUE_PAIR_SEPERATOR,
                                                QString::SkipEmptyParts);
           for (int i=0; i<pairs.count(); i++) {
               QStringList pair = pairs[i].split(KEY_VALUE_SEPERATOR,
                                                 QString::SkipEmptyParts);
               if (pair.count() >1 )
               {
                   key = pair[0];
                   value = pair[1];
               }
               else if (pair.count() == 1 )
               {
                   key = pair[0];
                   value = "";
               }

               decodeDelimiters(key);
               decodeDelimiters(value);
               attributeList[key] = value;
           }

           info.m_data = attributeList;
           info.m_widgetType = query.value(6).toString();
           info.m_uid = query.value(7).toInt();
           info.m_startPath = query.value(8).toString();
           info.m_dataPath = query.value(9).toString();
           info.m_capabilityCheck = query.value(10).toBool();
           info.m_certificateAki = query.value(11).toString();
           info.m_nativeId = query.value(12).toString();
           return true;
        }
        LOG ("Error: Notfound " << appID);
    }

    LOG ("WebAppRegistry::registered: error");
    return false;
}

/*!
 \return list of all already registered web apps, note that reciever responsible to free the list
*/
QList<WebAppInfo>* WebAppRegistry::registered()
{
    LOG ("WebAppRegistry::registered");

    QSqlDatabase db = QSqlDatabase::database("webAppConnection");

    if (db.isOpen()) {
        QSqlQuery query(db);
        query.prepare("SELECT id,appTitle,appPath,iconPath,attributeList,secureSession,widgetType,uid,startPath,dataPath,capabilityCheck,certificateAki,nativeId FROM webapp_registry");
        query.exec();
        LOG("[SELECT * FROM webapp_registry] returns: " << query.lastError().text());
        QList<WebAppInfo> *list = new QList<WebAppInfo>();

        while (query.next()) {
           WebAppInfo appInfo;
           appInfo.m_appId = query.value(0).toString();
           appInfo.m_appTitle = query.value(1).toString();
           appInfo.m_appPath = query.value(2).toString();
           appInfo.m_iconPath = query.value(3).toString();

           QString attributes = query.value(4).toString();
           AttributeMap attributeList;
           QString key, value;
           QStringList pairs = attributes.split(KEY_VALUE_PAIR_SEPERATOR, QString::SkipEmptyParts);
           for (int i=0; i<pairs.count(); i++) {
               QStringList pair = pairs[i].split(KEY_VALUE_SEPERATOR, QString::SkipEmptyParts);
               if (pair.count() >1 )
               {
                   key = pair[0];
                   value = pair[1];
               }
               else if (pair.count() == 1 )
               {
                   key = pair[0];
                   value = "";
               }
               decodeDelimiters(key);
               decodeDelimiters(value);
               attributeList[key] = value;
           }
           appInfo.m_data = attributeList;

           appInfo.m_widgetType = query.value(6).toString();
           appInfo.m_uid = query.value(7).toInt();
           appInfo.m_startPath = query.value(8).toString();
           appInfo.m_dataPath = query.value(9).toString();
           appInfo.m_capabilityCheck = query.value(10).toBool();
           appInfo.m_certificateAki = query.value(11).toString();
           appInfo.m_nativeId = query.value(12).toString();

           list->append(appInfo);
        }
        LOG ("Total: " << list->count());
        return list;
    }
    LOG ("WebAppRegistry::registered: error");
    return NULL;
}

/*!
 sets the web application attribute.

 \a appID a unique ID for the webApp
 \a attribute is attribute key
 \a value is value to be set.
 \return true if successful
*/
bool WebAppRegistry::setWebAppAttribute(const QString& appID,
                                              const QString& attribute,
                                              const QVariant& value)
{
    LOG ("WebAppRegistry::setWebAppAttribute -" << appID);

    bool ret(false);

    QSqlDatabase db = QSqlDatabase::database("webAppConnection");

    WebAppInfo info;
    if (isRegistered(appID, info)) {
        QVariant presentValue = info.value(attribute);
        if (db.isOpen()) {
            if ((!value.isNull()) && (presentValue != value)) {
                QList<QString> stringList = info.m_data.keys();
                bool keyFound = false;
                // case insensitive search for the key
                for (int i=0; i<stringList.count(); i++) {
                    if (0 == stringList[i].compare(attribute,
                                                   Qt::CaseInsensitive)) {
                        info.m_data[stringList[i]] = value;
                        keyFound = true;
                        break;
                    }
                }
                // add new attribute
                if (!keyFound)
                    info.m_data[attribute] = value;

                QString attributesList;
                AttributeMap::const_iterator i;
                QString key;
                QString val;
                for (i=info.m_data.begin(); i != info.m_data.end(); i++) {
                    key = i.key();
                    val = i.value().toString();

                    encodeDelimiters(key);
                    encodeDelimiters(val);
                    attributesList = attributesList + key
                        + KEY_VALUE_SEPERATOR + val + KEY_VALUE_PAIR_SEPERATOR;
                }
                QSqlQuery query(db);
                QString queryStr = "UPDATE webapp_registry SET attributeList = :value WHERE id = :id";
                query.prepare(queryStr);
                query.bindValue(":value", attributesList);
                query.bindValue(":id", QVariant (appID));
                query.exec();

                LOG("WebAppRegistry::setWebAppAttribute - "<<query.lastQuery());
                LOG(query.lastError().text());
                ret = true;
            }
            ret = true;
        }
    }

    return ret;
}

QVariant WebAppRegistry::getAttribute(const QString& appID, const QString& attribute, const QVariant& defaultValue)
{
    WebAppInfo info;
    if (isRegistered(appID, info)) {
        QVariant v = defaultValue;

        if (info.m_data.contains(attribute)) {
            v = info.value(attribute);
        }

        return v;
    }

    return defaultValue;
}

/*!
 sets the web application version.
  \a appID a unique ID for the webApp
  \a value is value to be set.

  The newIconPath parameter is ignored in this implementation.

 \return true if successful
*/
bool WebAppRegistry::setWebAppVersion(const QString& appID,
                                            const QVariant& value,
                                            const QString& newIconPath)
{
    bool ret = setWebAppAttribute(appID, W3CSettingsKey::WIDGET_VERSION , value);

#ifdef Q_OS_SYMBIAN
    if ( ret ) {
        ret = m_s60Instance->setWebAppVersion(appID, value, newIconPath);
    }
#endif

    return ret;
}

QString WebAppRegistry::nativeIdToAppId(const QString& nativeID)
{

    LOG("WebAppRegistry::nativeIdToAppId() - nativeIdToAppId using WidgetRegistrationS60 API");

    QSqlDatabase db = QSqlDatabase::database("webAppConnection");

    if (db.isOpen()) {
        QSqlQuery query(db);

#ifdef Q_OS_SYMBIAN
        query.prepare("SELECT id FROM webapp_registry WHERE uid = :uid");
        bool ok;
        int uid = nativeID.toInt(&ok, 10);

        if (!ok)
          return QString();

        query.bindValue(0, QVariant (nativeID.toInt(&ok, 10)));
#else
        query.prepare("SELECT id FROM webapp_registry WHERE nativeId = :nativeId");
        query.bindValue(0, QVariant (nativeID));
#endif

        query.exec();
        LOG(query.lastError().text());

        if (query.next())
          return query.value(0).toString();

        return QString();
    }

    return QString();
}

bool WebAppRegistry::setNativeId(const QString& appID,
                                 const QString& nativeID)
{
    LOG ("WebAppRegistry::setNativeId -" << appID);

    bool ret(false);
    QString attribute("nativeId");
    //    WebAppInfo info;
    if (isRegistered(appID)) {
        ret = updateAttribute(appID, attribute, QVariant(nativeID));
    }
    return ret;
}


bool WebAppRegistry::setUid(const QString& appID, int uid)
{
    LOG ("WebAppRegistry::setUid -" << appID);

    bool ret(false);

    QSqlDatabase db = QSqlDatabase::database("webAppConnection");

    WebAppInfo info;
    if (isRegistered(appID, info)) {
        int presentValue = info.m_uid;

        if (db.isOpen()) {
            if (presentValue != uid) {
                QSqlQuery query(db);
                query.prepare("UPDATE webapp_registry SET uid = :uid WHERE id = :id");
                query.bindValue(":uid", uid);
                query.bindValue(":id", QVariant(appID));
                query.exec();

                LOG("WebAppRegistry::setWebAppAttribute - "<<query.lastQuery());
                LOG(query.lastError().text());
                ret = true;
            }
            ret = true;
        }
    }

    return ret;
}

bool WebAppRegistry::updateAttribute(const QString& appID, QString& attribute, QVariant value)
{
    bool ret(false);
    QSqlDatabase db = QSqlDatabase::database("webAppConnection");
    if (db.isOpen()) {
        QSqlQuery query(db);
        query.prepare("UPDATE webapp_registry SET \'" + attribute + "\' = \'" + value.toString() + "\' WHERE id = \'" + appID + "\'");
        query.exec();
        LOG("WebAppRegistry::setWebAppAttribute - "<<query.lastQuery());
        LOG(query.lastError().text());
        ret = true;
    }
    return ret;
}


bool WebAppRegistry::setCapabilityCheck(const QString& appID, bool value)
{
    LOG ("WebAppRegistry::setcapabilityCheck -" << value);
    bool ret(false);
    QString attribute = "capabilityCheck";
    WebAppInfo info;
    if (isRegistered(appID, info)) {
        ret = updateAttribute(appID,attribute,value);
    }
    return ret;
}


bool WebAppRegistry::setCertificateAki(const QString& appID, const QString& value)
{
    LOG ("WebAppRegistry::setCertificateAki -" << value);

    bool ret(false);
    QString attribute = "certificateAki";
    WebAppInfo info;
    if (isRegistered(appID, info)) {
        ret = updateAttribute(appID,attribute,value);
    }
    return ret;
}

/*!
  nextAvailableUid() returns the lowest non-used UID in the available ranges.
  This is a UID needed by Symbian only, but directly access the DB for used UIDs.
  only called from WidgetRegistrationS60::registerApp

  The WRT Widget Registry (platform) reserves a block of 999 UIDs to be allocated
  to widgets.  This range is split in two, with the first half being reserved for
  widgets installed on drive C, and the rest for those installed on external drives.

  We have taken the latter half of each of these ranges to be used for cWRT widgets,
  so we allocate in the ranges of KWidgetUidCWRTInternalMemoryStart through
  KWidgetUidCWRTInternalMemoryStop (inclusive) and KWidgetUidCWRTExternalMemoryStart
  through KWidgetUidCWRTExternalMemoryStop (inclusive)

  If no UID could be found, 0 is returned
 */
int WebAppRegistry::nextAvailableUid()
{
    const int KWidgetUidLowerBound = 0x2000DAD2;
    const int KWidgetUidUpperBound = 0x2000DEB9;

    // The current allocation scheme splits the range into two so that
    // in-device memory uses one range and removable memory cards use a
    // separate range.  Eventually, removable memory is probably going to
    // have to use a reallocation scheme on insertion.
    const int KWidgetUidInternalMemoryStart = KWidgetUidLowerBound;
    const int KWidgetUidExternalMemoryStart = (KWidgetUidLowerBound + KWidgetUidUpperBound + 1) / 2; // half way
    const int KWidgetUidExternalMemoryStop = KWidgetUidUpperBound;

    // Additions for separation of CWRT Widget UID space from WRT Widget UID space
    const int KWidgetUidWRTInternalMemoryStop = (KWidgetUidInternalMemoryStart + KWidgetUidExternalMemoryStart + 1) / 2;
    const int KWidgetUidCWRTInternalMemoryStart = KWidgetUidWRTInternalMemoryStop;
    const int KWidgetUidCWRTInternalMemoryStop = KWidgetUidExternalMemoryStart;
    const int KWidgetUidWRTExternalMemoryStop = (KWidgetUidExternalMemoryStart + KWidgetUidExternalMemoryStop + 1) / 2;
    const int KWidgetUidCWRTExternalMemoryStart = KWidgetUidWRTExternalMemoryStop;
    const int KWidgetUidCWRTExternalMemoryStop = KWidgetUidUpperBound + 1;

    QSqlDatabase db = QSqlDatabase::database("webAppConnection");

    if (db.isOpen()) {
        int nextPossibleUid = KWidgetUidCWRTInternalMemoryStart;

        QSqlQuery query(db);
        query.prepare("SELECT DISTINCT uid FROM webapp_registry WHERE uid NOTNULL ORDER BY uid ASC");
        query.exec();
        LOG(query.lastError().text());
        while (query.next()) {
            if (query.value(0).toInt() == nextPossibleUid)
                nextPossibleUid++;

            if (nextPossibleUid >= KWidgetUidCWRTExternalMemoryStop)
                return 0;

            if (nextPossibleUid >= KWidgetUidCWRTInternalMemoryStop)
                nextPossibleUid = KWidgetUidCWRTExternalMemoryStart;
         }

         return nextPossibleUid;
    }

    return 0;
}

