/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

#include "jswidgetinterface.h"
#include "wrtsettings.h"
#include <QLocale>
#include <QUrl>
#include <QDesktopServices>

#include <qwebpage.h>
#include <qwebframe.h>

namespace WRT {


JSWidgetInterface::JSWidgetInterface()
  : JSResourceWrapper("widgetinterface.js", "widgetInterfaceCPP")
#ifdef WIDGET_EXTENSION
  , m_commandline("")
  , m_message("")
#endif
  , m_viewMode("")
  , m_visible(false)
  , m_delayedVisibilityNotification(false) {

    connect(this, SIGNAL(loadFinished()), this, SLOT(loadFinished()));
}

JSWidgetInterface::~JSWidgetInterface() {

}

void JSWidgetInterface::openUrl(QString iri) {
    QUrl url(iri, QUrl::TolerantMode);
    if (url.isEmpty() || !url.isValid()) return;

    bool ret = QDesktopServices::openUrl(url);
}

void JSWidgetInterface::setViewMode(QString viewMode)
{
    QStringList w3cViewModes;
    w3cViewModes << "windowed" << "floating" << "fullscreen" << "maximized" << "minimized";
    if (w3cViewModes.contains(viewMode))
        emit viewModeChangeRequested(viewMode);
}

QString JSWidgetInterface::viewMode() const
{
    return m_viewMode;
}

void JSWidgetInterface::onViewModeChange(QString viewMode)
{
    m_viewMode = viewMode;

    webPage()->currentFrame()->evaluateJavaScript(QString("if (typeof(widget.onviewmodechange) === 'function') { widget.onviewmodechange('%1'); }").arg(viewMode));
}

void JSWidgetInterface::beforeJavaScriptEvaluation() {
    WRT::WrtSettings* m_settings = WRT::WrtSettings::createWrtSettings();
    QString myLang = m_settings->valueAsString("UserAgentLanguage");
    if (myLang.isEmpty() || myLang.compare(" ") == 0) {
       QLocale language = QLocale::system();
       myLang = language.name();
    }

    myLang.toLower();
    myLang.replace(QString("_"),QString("-"));

    QString uaLang = QString("var __uaLang='%1';").arg(myLang);
    webPage()->currentFrame()->evaluateJavaScript(uaLang);

}

void JSWidgetInterface::pageOnReload() {

    JSResourceWrapper::pageOnReload();

#ifdef WIDGET_EXTENSION
    QString msgJS = "widget.message = \"" + m_message  + "\";";
    QString cmdJS = "widget.commandline = \"" + m_commandline  + "\";";
    webPage()->currentFrame()->evaluateJavaScript(msgJS + cmdJS);
#endif
}

void JSWidgetInterface::setCommandline(QString commandline)
{
#ifdef WIDGET_EXTENSION
    m_commandline = commandline;
#else
    Q_UNUSED(commandline)
#endif
}

void JSWidgetInterface::setMessage(QString message)
{
#ifdef WIDGET_EXTENSION
    m_message = message;
    webPage()->currentFrame()->evaluateJavaScript(
        "widget.message = \"" + message  + "\";");

    webPage()->currentFrame()->evaluateJavaScript("if ((typeof(widget.onmessagereceived) != 'undefined') && (widget.onmessagereceived != null)) {widget.onmessagereceived();}");
#else
    Q_UNUSED(message)
#endif
}

void JSWidgetInterface::onShow()
{
    m_visible = true;
    if (isLoading()) {
        m_delayedVisibilityNotification = true;
        return;
    }
    webPage()->currentFrame()->evaluateJavaScript("if ((typeof(widget.onshow) != 'undefined') && (widget.onshow != null)) {widget.onshow();}");

}


void JSWidgetInterface::onHide()
{
    m_visible = false;
    if (isLoading()) {
        m_delayedVisibilityNotification = true;
        return;
    }

    webPage()->currentFrame()->evaluateJavaScript("if ((typeof(widget.onhide) != 'undefined') && (widget.onhide != null)) {widget.onhide();}");
}


void JSWidgetInterface::loadFinished()
{
    if (m_delayedVisibilityNotification) {
        m_delayedVisibilityNotification = false;
        if (m_visible) {
            onShow();
        } else {
            onHide();
        }
    }
}

}
