/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

#include "jsmenuitem.h"
#include "jsmenuitemfactory.h"
#include "action.h"

#include <QWebPage>
#include <QWebFrame>
#include <QDebug>

namespace WRT {

/*!
    \class WRT::JSMenuItem
    \brief The WRT::JSMenuItem represents the native instance of the MenuItem exposed to JavaScript.

    \inmodule WidgetCore

    WRT::JSMenuItem represents the native instance of the MenuItem expose to JavaScript. WRT::JSMenuItem holds
    a WRT::Action internally which is used as native menu item.
*/

/*!
    Constructs an instance of JSMenuItem with expose name \a exposeName
 */
JSMenuItem::JSMenuItem(QString exposeName)
    : JSResourceWrapper("", exposeName, JSResourceWrapper::DontEvalauteJavaScript)
    , m_action(new Action()){
}

JSMenuItem::~JSMenuItem() {
    delete m_action;
}

/*!
    Sets \a name for the menu item.
*/
void JSMenuItem::setText(QString name) {
    m_action->setText(name);
}

/*!
    Sets native instance as \a disabled.
*/
void JSMenuItem::setDisabled(bool disabled) {
    m_action->setEnabled(!disabled);
}

/*!
    Returns a pointer to the native action of this action.
*/
Action* JSMenuItem::action() {
    return m_action;
}


void JSMenuItem::addToJavaScriptWindowObject() {
    if (!webPage()) {
        return;
    }

    webPage()->mainFrame()->addToJavaScriptWindowObject(name(), this);
}

/*!
    Adds a menu item owned by \a nativeInstanceHandle.
*/
bool JSMenuItem::addMenuItem(int nativeInstanceHandle) {
    // Maemo6 support as of now only one level of menu items
#if defined(SUPPORT_ONLY_FLAT_MENUS)
    return false;
#else
    JSMenuItem* item = JSMenuItemFactory::item(nativeInstanceHandle);
    m_action->addSubAction(item->action());
    return true;
#endif
}

/*!
    Removes a menu item owned by \a nativeInstanceHandle as a sub menu item to this menu item.
*/
bool JSMenuItem::removeMenuItem(int nativeInstanceHandle) {
    // Maemo6 support as of now only one level of menu items
#if defined(SUPPORT_ONLY_FLAT_MENUS)
    return false;
#else
    JSMenuItem* item = JSMenuItemFactory::item(nativeInstanceHandle);
    m_action->removeSubAction(item->action());
    return true;
#endif
}

/*!
    Inserts a menu item owned by \a nativeInstanceHandle as a sub menu item to this menu item before given \a beforeIndex.
*/
bool JSMenuItem::insertMenuItem(int nativeInstanceHandle, int beforeIndex) {
    // Maemo6 support as of now only one level of menu items
#if defined(SUPPORT_ONLY_FLAT_MENUS)
    return false;
#else
    JSMenuItem* item = JSMenuItemFactory::item(nativeInstanceHandle);
    m_action->insertSubAction(item->action(), beforeIndex);
    return true;
#endif
}

}
