/*
 * Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * This file is part of Qt Web Runtime.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

// Mappings between native instances and JavaScript instances.
menuItemFactoryCPP.mapping = {};

/**
 * Helper function for searching native instance (native instance id) that is mapped to the menuItem
 */
menuItemFactoryCPP.nativeId = function(menuItem) {
    var id = -1;
    var x = null;
    for (x in menuItemFactoryCPP.mapping) {
        if (menuItemFactoryCPP.mapping[x] == menuItem) {
            id = x;
            break;
        }
    }
    return id;
};

/** Constructor for nokia.device.MenuItem objects. 
  *  @param {string}
 *             label Text to be shown in the menu. Empty strings are ignored and string over 40 characters are 
 *             cut to 40 characters.
 *  @param {onselect}
 *             onselect Optinal argument. You pass the onselect callback, it must be a function.  
 */
window.nokia.device.MenuItem = function(name, onselect)
{
    
    var _onselect = undefined;
    if ('function' === typeof onselect) {
        _onselect = onselect;
    }
    
    var _disabled = false;
    var _name = undefined;
    var _iconUrl = undefined;
    var _submenu = new nokia.device.MenuBase();

    this.__defineGetter__("onselect", function() {
        return _onselect;
    });
    
    /**
     * Sets onselect callback function. Callback must be type of a function, otherwise it is ignored.
     * When user selects the menu item that has been added to a Context Menu, the id of the element, which has the Context Menu, 
     * is passed as argument to the callback function, otherwise no arguments are passed to onselect. 
     */
    this.__defineSetter__("onselect", function(callback) {
        var callbackType = typeof callback;
        if (callbackType === 'function') {
            _onselect = callback;
        }
    });
    
    this.__defineGetter__("disabled", function() {
        return _disabled;
    });
    
    this.__defineSetter__("disabled", function(disabled) {
        // Should we try-catch here
        if (typeof disabled === 'boolean') {
            _disabled = disabled;
            _nativeInstance.disabled = disabled;
        }
    });

    this.__defineGetter__("name", function() {
        return _name;
    });

    this.__defineSetter__("name", function(name) {
        var typeName = typeof name;
        if (typeName === 'string' && name != "") {
            if (name.length > 40) {
                _name = name.substr(0, 40);
            }
            else {
                _name = name;
            }
            
            if (_nativeInstance) {
                _nativeInstance.name = _name;
            }
        }
    });

    this.__defineGetter__("iconUrl", function(){
        return _iconUrl;
    });
    
    this.__defineSetter__("iconUrl", function(icon){
        var iconType = typeof icon;
        if (iconType === 'string') {
            _iconUrl = icon;
        }
    });

    /**
     * Appends menu item as a sub menu item for this menu item.
     */
    this.addMenuItem = function(item) {
        if (item instanceof nokia.device.MenuItem) {
            // Get native of  
            var itemId = menuItemFactoryCPP.nativeId(item);
            var added = false;
            if (_nativeInstance) {
                added = _nativeInstance.addMenuItem(itemId);
            }
            
            if (added) {
                _submenu.add(item);
            }
        }
    };
    

    /**
     * Revomes sub menu item for this menu item.
     */
    this.removeMenuItem = function(item) {
        if (item instanceof nokia.device.MenuItem) {
            var itemId = menuItemFactoryCPP.nativeId(item);
            var removed = false;
            if (_nativeInstance) {
                removed = _nativeInstance.removeMenuItem(itemId);
            }
            if (removed) {
                return _submenu.remove(item);
            }
        }
    };
    

    /**
     * Inserts sub menu item as a sub menu item of this menu item before given index.
     */
    this.insertMenuItemBefore = function(item, index) {
        if (item instanceof nokia.device.MenuItem) {        
            var itemId = menuItemFactoryCPP.nativeId(item);
            var inserted = false;
            if (_nativeInstance) {
                inserted = _nativeInstance.insertMenuItem(itemId, index);
            }
            if (inserted) {
                _submenu.insert(item, index);
            }
        }
    };
    
    

    this.item = function(index) {
        return _submenu.item(index);
    };
    
    this.name = name;

    var nativeId = menuItemFactoryCPP.createMenuItem();
    menuItemFactoryCPP.mapping[nativeId] = this;
    var _nativeInstance = tmpMenuItemFactoryCPPMenuItem;
    _nativeInstance.name = this.name;
}

/** Constructor.
 *  MenuBase is used by other objects needing basic menu item managing
 *  functions.
 */
window.nokia.device.MenuBase = function() {
    this.children = [];
};


/** Adds new nokia.device.MenuItem
 *  @param {nokia.device.MenuItem}
 *             menuItem
 */
nokia.device.MenuBase.prototype.add = function (menuItem) {
    var valid = menuItem instanceof nokia.device.MenuItem;
    if (!valid) {
        return;
    }

    this.children.push(menuItem);
};


/** Removes a nokia.device.MenuItem
 *  @param {nokia.device.MenuItem}
 *             menuItem
 *  @return {boolean}
 *             true, if menuItem was found and removed, false otherwise
 */
nokia.device.MenuBase.prototype.remove = function (menuItem) {
    var valid = menuItem instanceof nokia.device.MenuItem;
    if (!valid) {
        return false;
    }

    var len = this.children.length;
    for (var i = 0; i < len; i++) {
      // Comparing objects
        if (this.children[i] == menuItem) {
            this.children.splice(i, 1);
            return true;
        }
    }
    
    return false;
};


nokia.device.MenuBase.prototype.insert = function(menuItem, index) {
    var valid = menuItem instanceof nokia.device.MenuItem;
    if (!valid) {
        return false;
    }
    var len = this.children.length;
    if (index >= 0 && index < len) {
        this.children.splice(index, 0, menuItem);
    } else {
        this.children.push(menuItem);
    }
    return true;
};



/** Searches nokia.device.MenuItem from main this menu level by index.
 *  @param {number}
 *            index Used as a search index
 *  @return {nokia.device.MenuItem}
 *            Returns a nokia.device.MenuItem with a matching index, or undefined, if match
 *            was not found.
 */
nokia.device.MenuBase.prototype.item = function(index) {
    //TODO: error handling? error codes?
    if (this.children.length >= index+1) {
        return this.children[index];
    } else {
        return undefined;
    }
};


/** Returns all MenuItems in this MenuBase as an array.
 *  @return {Array}
 *             Array of MenuItems
 */
nokia.device.MenuBase.prototype.getMenu = function () {
    return this.children;
};


/** Removes all MenuItems from this MenuBase.
 */
nokia.device.MenuBase.prototype.clear = function () {
    this.children = [];
};
